"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.computeGraph = void 0;
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const merge = require("lodash.merge"); // eslint-disable-line @typescript-eslint/no-require-imports
const uniq = require("lodash.uniq"); // eslint-disable-line @typescript-eslint/no-require-imports
const uniqBy = require("lodash.uniqby"); // eslint-disable-line @typescript-eslint/no-require-imports
const constants_1 = require("./constants");
const graph_1 = require("./graph");
const types_1 = require("./types");
const utils_1 = require("./utils");
/** List of cdk stack children used by cdk that the graph ignores */
const IGNORED_STACK_CHILDREN = [
    "CDKMetadata",
    "BootstrapVersion",
    "CheckBootstrapVersion",
];
/**
 * Computes the graph store for a given Cdk construct, usually App.
 * @internal
 */
function computeGraph(root) {
    const store = new graph_1.Graph.Store();
    // List of nodes that should not be processed (extraneous nodes detected during compute)
    // NB: example is stack template construct which cdk creates as sibling of nested stacks,
    // we care about the stack and not the marshalled construct.
    const nodesToIgnore = [];
    // List of all unresolved referenced detected during compute, which are resolved after all nodes have been stored
    const allUnresolvedReferences = [];
    // List of all unresolved dependencies detected during compute, which are resolved after all nodes have been stored
    const allUnresolvedDependencies = [];
    const visit = (construct, parent) => {
        // Do not graph the CdkGraph itself
        if (construct.node.id === constants_1.GRAPH_ID) {
            return;
        }
        // Do not graph the Tree construct (synthesizes tree.json)
        if (graph_1.Graph.AppNode.isAppNode(parent) && construct.node.id === "Tree") {
            return;
        }
        // Do not graph stack CDKMetadata, BootstrapVersion, and similar constructs
        if (graph_1.Graph.StackNode.isStackNode(parent) &&
            IGNORED_STACK_CHILDREN.includes(construct.node.id)) {
            return;
        }
        const { uuid, attributes = {}, metadata = [], tags = {}, logicalId, cfnType, constructInfo, dependencies, unresolvedReferences, flags, } = utils_1.inferNodeProps(construct);
        if (nodesToIgnore.includes(uuid)) {
            return;
        }
        // Infer the stack this construct belongs to
        let stack;
        try {
            if (construct.node.scope) {
                const stackUUID = utils_1.getConstructUUID(aws_cdk_lib_1.Stack.of(construct));
                stack = store.getStack(stackUUID);
            }
        }
        catch {
            // ignore - expected to throw if construct is not contained in a stack
        }
        const nodeProps = {
            store,
            stack,
            parent,
            uuid,
            id: construct.node.id,
            path: construct.node.path,
            attributes,
            metadata,
            tags,
            constructInfo,
            logicalId,
            cfnType,
            flags,
        };
        let node;
        switch (constructInfo?.fqn) {
            case types_1.ConstructInfoFqnEnum.APP: {
                node = new graph_1.Graph.AppNode({
                    store,
                    parent,
                    attributes,
                    metadata,
                    tags,
                    constructInfo,
                    logicalId,
                    flags,
                });
                break;
            }
            case types_1.ConstructInfoFqnEnum.STAGE: {
                node = new graph_1.Graph.StageNode(nodeProps);
                break;
            }
            case types_1.ConstructInfoFqnEnum.STACK: {
                node = new graph_1.Graph.StackNode(nodeProps);
                break;
            }
            case types_1.ConstructInfoFqnEnum.NESTED_STACK: {
                // NB: handle NestedStack<->CfnStack as single Node with NestedStack construct as source
                // https://github.com/aws/aws-cdk/blob/119c92f65bf26c3fdf4bb818a4a4812022a3744a/packages/%40aws-cdk/core/lib/nested-stack.ts#L119
                const parentStack = utils_1.inferNodeProps(construct.nestedStackParent);
                const _nestedStack = construct.node.scope.node.findChild(construct.node.id + ".NestedStack");
                const cfnStackWrapper = utils_1.inferNodeProps(_nestedStack);
                const cfnStack = utils_1.inferNodeProps(_nestedStack.node.findChild(construct.node.id + ".NestedStackResource"));
                // ignore parent scope cfn stack (template) constructs
                nodesToIgnore.push(cfnStackWrapper.uuid, cfnStackWrapper.uuid);
                node = new graph_1.Graph.NestedStackNode({
                    ...nodeProps,
                    logicalId: cfnStack.logicalId,
                    attributes: merge({}, attributes, cfnStackWrapper.attributes, cfnStack.attributes),
                    metadata: [
                        ...metadata,
                        ...(cfnStackWrapper.metadata || []),
                        ...(cfnStack.metadata || []),
                    ],
                    parentStack: store.getStack(parentStack.uuid),
                });
                // Only add uniq dependencies as wrapper and stack may duplicate
                dependencies.push(...uniq([...cfnStackWrapper.dependencies, ...cfnStack.dependencies]));
                // Only add uniq references as wrapper and stack may duplicate
                unresolvedReferences.push(...uniqBy([
                    ...cfnStackWrapper.unresolvedReferences.map((ref) => ({
                        ...ref,
                        source: uuid,
                    })),
                    ...cfnStack.unresolvedReferences.map((ref) => ({
                        ...ref,
                        source: uuid,
                    })),
                ], (v) => `${v.referenceType}::${v.source}::${v.target}::${JSON.stringify(v.value || "")}`));
                break;
            }
            case types_1.ConstructInfoFqnEnum.CFN_STACK: {
                // CfnStack always proceeds NestedStack, based on above case we merge CfnStack into
                // NestedStack (mirror developer expectations) and then ignore CfnStack.
                throw new Error(`CfnStack should be ignored by NestedStack: ${uuid}`);
            }
            case types_1.ConstructInfoFqnEnum.CFN_OUTPUT: {
                node = new graph_1.Graph.OutputNode({
                    ...nodeProps,
                    value: aws_cdk_lib_1.Stack.of(construct).resolve(construct.value),
                    description: aws_cdk_lib_1.Stack.of(construct).resolve(construct.description),
                    exportName: aws_cdk_lib_1.Stack.of(construct).resolve(construct.exportName),
                });
                // extract unresolved references from value
                unresolvedReferences.push(...utils_1.extractUnresolvedReferences(node.uuid, node.value || {}));
                break;
            }
            case types_1.ConstructInfoFqnEnum.CFN_PARAMETER: {
                const cfnParameter = construct;
                node = new graph_1.Graph.ParameterNode({
                    ...nodeProps,
                    value: aws_cdk_lib_1.Stack.of(construct).resolve(cfnParameter.value),
                    description: aws_cdk_lib_1.Stack.of(construct).resolve(cfnParameter.description),
                    parameterType: cfnParameter.type,
                });
                break;
            }
            default: {
                if (aws_cdk_lib_1.Resource.isResource(construct)) {
                    node = new graph_1.Graph.ResourceNode({
                        ...nodeProps,
                        cdkOwned: aws_cdk_lib_1.Resource.isOwnedResource(construct),
                    });
                }
                else if (cfnType) {
                    node = new graph_1.Graph.CfnResourceNode(nodeProps);
                }
                else {
                    node = new graph_1.Graph.Node({
                        nodeType: types_1.NodeTypeEnum.DEFAULT,
                        ...nodeProps,
                    });
                    // Cdk Stack.Exports is proxy to actual Cfn exports and extraneous in the graph
                    if (construct.node.id === types_1.CdkConstructIds.EXPORTS &&
                        aws_cdk_lib_1.Stack.isStack(construct.node.scope)) {
                        node.addFlag(types_1.FlagEnum.EXTRANEOUS);
                    }
                }
            }
        }
        // Track unresolved dependencies, since nodes might not be created in the store yet
        allUnresolvedDependencies.push(...dependencies.map((dep) => [uuid, dep]));
        // Track all logicalId references in the nodes attributes
        // this will get resolved after all nodes have been stored
        allUnresolvedReferences.push(...unresolvedReferences);
        // Visit all child to compute the tree
        for (const child of construct.node.children) {
            try {
                visit(child, node);
            }
            catch (e) {
                aws_cdk_lib_1.Annotations.of(root).addWarning(`Failed to render graph for node ${child.node.path}. Reason: ${e}`);
                throw e;
            }
        }
    };
    visit(root, store.root);
    // Resolve all references - now that the tree is stored
    for (const unresolved of allUnresolvedReferences) {
        try {
            resolveReference(store, unresolved);
        }
        catch (e) {
            console.warn(e, unresolved);
            // TODO: consider saving unresolved references if become valuable.
        }
    }
    // Resolve all dependencies - now that the tree is stored
    for (const unresolved of allUnresolvedDependencies) {
        resolveDependency(store, unresolved);
    }
    return store;
}
exports.computeGraph = computeGraph;
/**
 * Resolve reference. During initial graph traversal not all nodes have been added at the time
 * a reference has been detected, as such we need to resolve all references after the graph tree
 * has been stored.
 * @internal
 */
function resolveReference(store, unresolved) {
    const source = store.getNode(unresolved.source);
    if (source.stack == null) {
        console.warn(String(source), source);
        throw new Error(`Node ${source} is not within stack`);
    }
    let target;
    switch (unresolved.referenceType) {
        case types_1.ReferenceTypeEnum.REF: {
            // ref logicalId is only unique in the stack
            target = store.findNodeByLogicalId(source.stack, unresolved.target);
            return new graph_1.Graph.Reference({
                store,
                uuid: utils_1.generateConsistentUUID(unresolved, graph_1.Graph.Reference.PREFIX),
                referenceType: types_1.ReferenceTypeEnum.REF,
                source,
                target,
            });
        }
        case types_1.ReferenceTypeEnum.IMPORT: {
            // imports already contain the stack id (stack:logicalId)
            target = store.findNodeByLogicalUniversalId(unresolved.target);
            return new graph_1.Graph.ImportReference({
                store,
                uuid: utils_1.generateConsistentUUID(unresolved, graph_1.Graph.ImportReference.PREFIX),
                source,
                target,
            });
        }
        case types_1.ReferenceTypeEnum.ATTRIBUTE: {
            const attribute = unresolved.value;
            if (attribute && attribute.startsWith("Outputs.")) {
                // Stack output reference
                const stacksToSearch = source.rootStack?.stage?.stacks || store.stacks;
                const potentialRefStacks = Object.values(stacksToSearch).filter((_stack) => _stack.logicalId === unresolved.target);
                if (potentialRefStacks.length === 1) {
                    const refStack = potentialRefStacks[0];
                    target = refStack.findOutput(attribute.replace("Outputs.", ""));
                }
                else {
                    console.warn("Failed to find logical id from attribute reference:", unresolved.target);
                    if (potentialRefStacks.length) {
                        console.warn("Found multiple matching stacks:", Object.values(potentialRefStacks).map((stack) => `${String(stack)}:${stack.logicalId || "ROOT"}`));
                    }
                    else {
                        console.warn("Available stacks:", Object.values(store.stacks).map((stack) => `${String(stack)}:${stack.logicalId || "ROOT"}`));
                    }
                    throw new Error(`Failed to find Fn::GetAtt stack for output reference "${unresolved.target}": ${potentialRefStacks.length}`);
                }
            }
            else {
                target = store.findNodeByLogicalId(source.stack, unresolved.target);
            }
            if (target) {
                return new graph_1.Graph.AttributeReference({
                    store,
                    uuid: utils_1.generateConsistentUUID(unresolved, graph_1.Graph.AttributeReference.PREFIX),
                    source,
                    target,
                    value: attribute,
                });
            }
        }
    }
    throw new Error(`Failed to resolve reference: ${JSON.stringify(unresolved)}`);
}
/**
 * Resolve dependency. During initial graph traversal not all nodes have been added at the time
 * a dependency has been detected, as such we need to resolve all dependencies after the graph tree
 * has been stored.
 * @internal
 */
function resolveDependency(store, unresolved) {
    const source = store.getNode(unresolved[0]);
    const target = store.getNode(unresolved[1]);
    return new graph_1.Graph.Dependency({
        store,
        uuid: utils_1.generateConsistentUUID(unresolved, graph_1.Graph.Dependency.PREFIX),
        source,
        target,
    });
}
//# sourceMappingURL=data:application/json;base64,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