"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Instance = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const security_group_1 = require("./security-group");
const volume_1 = require("./volume");
const vpc_1 = require("./vpc");
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
/**
 * This represents a single EC2 instance
 */
class Instance extends core_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.securityGroups = [];
        if (props.securityGroup) {
            this.securityGroup = props.securityGroup;
        }
        else {
            this.securityGroup = new security_group_1.SecurityGroup(this, 'InstanceSecurityGroup', {
                vpc: props.vpc,
                allowAllOutbound: props.allowAllOutbound !== false,
            });
        }
        this.connections = new connections_1.Connections({ securityGroups: [this.securityGroup] });
        this.securityGroups.push(this.securityGroup);
        core_1.Tag.add(this, NAME_TAG, props.instanceName || this.node.path);
        this.role = props.role || new iam.Role(this, 'InstanceRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        const iamProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.role.roleName],
        });
        // use delayed evaluation
        const imageConfig = props.machineImage.getImage(this);
        this.userData = (_a = props.userData) !== null && _a !== void 0 ? _a : imageConfig.userData;
        const userDataToken = core_1.Lazy.stringValue({ produce: () => core_1.Fn.base64(this.userData.render()) });
        const securityGroupsToken = core_1.Lazy.listValue({ produce: () => this.securityGroups.map(sg => sg.securityGroupId) });
        const { subnets } = props.vpc.selectSubnets(props.vpcSubnets);
        let subnet;
        if (props.availabilityZone) {
            const selected = subnets.filter(sn => sn.availabilityZone === props.availabilityZone);
            if (selected.length === 1) {
                subnet = selected[0];
            }
            else {
                this.node.addError(`Need exactly 1 subnet to match AZ '${props.availabilityZone}', found ${selected.length}. Use a different availabilityZone.`);
            }
        }
        else {
            if (subnets.length > 0) {
                subnet = subnets[0];
            }
            else {
                this.node.addError(`Did not find any subnets matching '${JSON.stringify(props.vpcSubnets)}', please use a different selection.`);
            }
        }
        if (!subnet) {
            // We got here and we don't have a subnet because of validation errors.
            // Invent one on the spot so the code below doesn't fail.
            subnet = vpc_1.Subnet.fromSubnetAttributes(this, 'DummySubnet', {
                subnetId: 's-notfound',
                availabilityZone: 'az-notfound',
            });
        }
        this.instance = new ec2_generated_1.CfnInstance(this, 'Resource', {
            imageId: imageConfig.imageId,
            keyName: props.keyName,
            instanceType: props.instanceType.toString(),
            securityGroupIds: securityGroupsToken,
            iamInstanceProfile: iamProfile.ref,
            userData: userDataToken,
            subnetId: subnet.subnetId,
            availabilityZone: subnet.availabilityZone,
            sourceDestCheck: props.sourceDestCheck,
            blockDeviceMappings: props.blockDevices !== undefined ? volume_1.synthesizeBlockDeviceMappings(this, props.blockDevices) : undefined,
            privateIpAddress: props.privateIpAddress,
        });
        this.instance.node.addDependency(this.role);
        this.osType = imageConfig.osType;
        this.node.defaultChild = this.instance;
        this.instanceId = this.instance.ref;
        this.instanceAvailabilityZone = this.instance.attrAvailabilityZone;
        this.instancePrivateDnsName = this.instance.attrPrivateDnsName;
        this.instancePrivateIp = this.instance.attrPrivateIp;
        this.instancePublicDnsName = this.instance.attrPublicDnsName;
        this.instancePublicIp = this.instance.attrPublicIp;
        this.applyUpdatePolicies(props);
    }
    /**
     * Add the security group to the instance.
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup) {
        this.securityGroups.push(securityGroup);
    }
    /**
     * Add command to the startup script of the instance.
     * The command must be in the scripting language supported by the instance's OS (i.e. Linux/Windows).
     */
    addUserData(...commands) {
        this.userData.addCommands(...commands);
    }
    /**
     * Adds a statement to the IAM role assumed by the instance.
     */
    addToRolePolicy(statement) {
        this.role.addToPolicy(statement);
    }
    /**
     * Apply CloudFormation update policies for the instance
     */
    applyUpdatePolicies(props) {
        if (props.resourceSignalTimeout !== undefined) {
            this.instance.cfnOptions.creationPolicy = {
                ...this.instance.cfnOptions.creationPolicy,
                resourceSignal: {
                    timeout: props.resourceSignalTimeout && props.resourceSignalTimeout.toISOString(),
                },
            };
        }
    }
}
exports.Instance = Instance;
//# sourceMappingURL=data:application/json;base64,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