/**
 * Implementation of the orbital elements Neural Network
 * for the Cambioni et al. (2019) model.
 *
 * Copyright (c) 2019 Arizona Board of Regents
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @file
 * @author Alexandre Emsenhuber
 * @author Saverio Cambioni
 */

#include <math.h>

#include "accretion_efficiency.h"

/**
 * Neural network predicting the mass of the two largest remnants in a giant impact.
 *
 * @param X predictors. Array of 4 values:
 *          - log10( mass of the target / Earth mass )
 *          - ratio mass projectile / mass target
 *          - impact angle [degree]
 *          - ration impact velocity / escape velocity
 *
 * @param Y output. Array of 2 values:
 *          - accretion efficiency of the largest remnant
 *          - accretion efficiency of the second remnant
 */
void accretion_efficiency( const double X[4], double Y[2] ) {
	static const double bi[ 4 ] = { -2.0, 0.1, 0.1, 1.0 };
	static const double ai[ 4 ] = { 1.0, 3.33333333333333, 0.0223713646532438, 0.666666666666667 };

	double v1[ 4 ];

	static const double a1[ 4 ][ 10 ] = {
		{ -0.99259839053989929, 0.449187477909109, 0.075714686288260766, -0.020452590896203708, -0.58210516295726789, -0.013375250174277573, -0.48230946052460555, 0.16155327325554553, 0.10059744565500377, 0.13487202760093472 },
		{ -0.54850611624671042, -1.734400064893842, 0.12915537201987323, 0.51316347585462074, -1.8203395764481449, -1.3335444899051745, -1.3867410094961816, -0.052339441637571961, 0.036215339838831268, 1.207085191429994 },
		{ -0.74716820627104474, 1.2785409210221961, 1.2652684857190708, 2.3907390150920693, -0.027941098981490296, 1.1616118117854528, 1.1396670154660771, 1.4257102657167255, 0.87383172088677019, 1.3677821392485054 },
		{ -1.4700873121241802, -0.31715783969951872, 2.3291763959280321, -0.69301858150785567, 0.927299850211842, 1.6777369443192311, 1.0506189979021969, 3.18380459684492, -2.329702148199758, -0.75080966510173219 }
	};
	static const double b1[ 10 ] = { 2.6073917552308483, -1.0543123438098263, 0.96084184343995738, 0.69887250663650113, -0.98100464756141359, -0.906472183922059, -1.1101160800828256, 2.8254372986637817, -2.9400814516311091, 1.9583962053001855 };

	double v2[ 10 ];

	static const double a2[ 10 ][ 7 ] = {
		{ -0.87990456358613534, 0.335543031246743, 0.3581652750145079, 0.58460887529764316, -0.45423132008533629, 0.64895072596147585, -0.021596476946219717 },
		{ 0.22475402459940549, -0.17485625690465434, -0.67256888021927785, -0.11002848850414457, 0.14385499031001098, 0.013825160492508339, 0.39563080034309944 },
		{ 2.1447041514117435, 1.0671671005511982, -0.68997027346305106, -0.27342197218341729, 1.1158837017809888, 1.4084051655938017, -0.61034530498228023 },
		{ 0.56030858631123337, 2.0206686741989626, 1.0360206955111424, -1.3364754692561343, 0.91162901010430253, 0.7376073950802593, 0.67893901758448283 },
		{ 0.74389027973146282, 0.048848059692643719, -0.626257013955689, 0.84295465866421748, -0.11000121156688385, 0.61945691827036709, 0.562156280988821 },
		{ 1.3382486343175444, -0.20922932495558832, 0.38144254432319952, -0.52674278682366749, 0.2243971147692253, 0.016630541015777888, -0.24014117875728225 },
		{ -1.0718243092718496, 0.24617839735881439, 0.24867754258386898, -0.81352902234736746, 0.17254616035570938, -0.020410456814524996, 0.068276600591814229 },
		{ -0.72208026395591307, 2.4518440419430951, -0.65330625662048869, -0.58097734345701046, 0.78955639574419723, -0.824353140870295, -0.33162033495057408 },
		{ -2.8858350457936264, -1.7494915582502779, -0.79025917768403275, -0.18106778121125583, -1.7844260497243247, -0.47059907849077875, 0.1609032536893332 },
		{ 0.46928043044999546, -0.60635394949707533, -0.20210050646504393, -0.7942878116668044, -0.48335886235441328, 0.087153327942112557, 0.74539974921278862 }
	};
	static const double b2[ 7 ] = { 1.9419204301505677, -2.3602001842178129, 0.063446988508151655, 1.0048637842885846, -0.55849965456441653, 1.5882132856273441, 2.4891174723212859 };

	double v3[ 7 ];

	static const double a3[ 7 ][ 2 ] = {
		{ -0.34035179476411376, 2.0954445444771603 },
		{ -0.29216640695761847, 1.3584206111291943 },
		{ 0.030079636529542306, 0.36228562742969384 },
		{ -1.731732654101318E-5, -0.60968713698202848 },
		{ 0.12300830290120388, -0.97034128143693188 },
		{ 0.31073727469560125, -0.90262887636934974 },
		{ 1.2083519248324135, 0.55989998922383477 }
	};

	static const double b4[ 2 ] = { 0.5941676668573361, -0.9515431408306219 };

	static const double ao[ 2 ] = { 0.404858299595142, 2.0 };
	static const double bo[ 2 ] = { -3.94, -1.0 };

	/* Input layer */
	for ( int k = 0; k < 4; k++ ) {
		v1[ k ] = ( ( X[ k ] - bi[ k ] ) * ai[ k ] ) - 1.;
	}

	/* Layer 1 */
	/* Sigmoid Symmetric Transfer Function */
	for ( int k = 0; k < 10; k++ ) {
		double d = 0.0;
		for ( int l = 0; l < 4; l++ ) {
			d += a1[ l ][ k ] * v1[ l ];
		}

		v2[ k ] = 2. / ( 1. + exp( -2. * ( b1[ k ] + d ) ) ) - 1.;
	}

	/* Layer 2 */
	/* Sigmoid Symmetric Transfer Function */
	for ( int k = 0; k < 7; k++ ) {
		double d = 0.0;
		for ( int l = 0; l < 10; l++ ) {
			d += a2[ l ][ k ] * v2[ l ];
		}

		v3[ k ] = 2. / ( 1. + exp( -2. * ( b2[ k ] + d ) ) ) - 1.;
	}

	/* Output layer */
	/* Map Minimum and Maximum Output Reverse-Processing Function */
	for ( int k = 0; k < 2; k++ ) {
		double d = 0.0;
		for ( int l = 0; l < 7; l++ ) {
			d += a3[ l ][ k ] * v3[ l ];
		}

		Y[ k ] = ( ( d + b4[ k ] ) / ao[ k ] ) + bo[ k ];
	}
}
