"""PyPresseportal - Python wrapper for the presseportal.de API

PyPresseportal makes data from the presseportal.de API
accessible as Python objects. You need an API key from
presseportal.de to use PyPresseportal (https://api.presseportal.de/en).
"""

import json

from datetime import datetime
from typing import Dict, List, Tuple, Union

import requests

from pypresseportal.pypresseportal_constants import (
    MEDIA_TYPES,
    PUBLIC_SERVICE_MEDIA_TYPES,
    INVESTOR_RELATIONS_NEWS_TYPES,
    PUBLIC_SERVICE_REGIONS,
    TOPICS,
    KEYWORDS,
)
from pypresseportal.pypresseportal_errors import (
    ApiError,
    ApiConnectionFail,
    ApiKeyError,
    ApiDataError,
    MediaError,
    RegionError,
    TopicError,
    KeywordError,
    NewsTypeError,
    SearchTermError,
    SearchEntityError,
)


class Company:
    """ Represents information about a company.

    A ``Company`` object can have the following attributes:
    ``id``, ``url``, ``name``, ``isin`` (optional), ``wkn`` (optional), ``shortname``, ``rss``, ``logo``, ``web``, and ``homepage``.
    See original API documentation for details (https://api.presseportal.de/en/doc/format/company).
    """

    def __init__(self, data: dict):
        self.data = data
        data_keys = self.data.keys()
        required_keys = ("id", "url", "name")

        for required_key in required_keys:
            if required_key in data_keys:
                setattr(self, required_key, data[required_key])
            else:
                raise ApiDataError(f"Required key {required_key} missing.")

        optional_keys = ("isin", "wkn", "shortname", "rss", "logo", "web", "homepage")
        for optional_key in optional_keys:
            if optional_key in data_keys:
                setattr(self, optional_key, data[optional_key])


class Entity:
    """Represents a company or a public service office search result.
    
    An ``Entity`` object can have the following attributes:
    ``id``, ``url``, ``name``, and ``type`` (company or office).
    See original API documentation for details (https://api.presseportal.de/doc/format/result).
    """

    def __init__(self, data: dict):
        self.data = data
        data_keys = self.data.keys()
        required_keys = ("id", "url", "name", "type")
        for required_key in required_keys:
            if required_key not in data_keys:
                raise ApiDataError(f"Required key {required_key} missing.")

        self.id = data["id"]
        self.url = data["url"]
        self.name = data["name"]
        self.type = data["type"]


class Office:
    """ Represents information about a public service office.

    An ``Office`` object can have the following attributes:
    ``id``, ``url``, ``name``, ``shortname``, ``rss``, ``logo``, ``web``, and ``homepage``.
    See original API documentation for details (https://api.presseportal.de/doc/format/office).
    """

    def __init__(self, data: dict):
        self.data = data
        data_keys = self.data.keys()
        required_keys = ("id", "url", "name")

        for required_key in required_keys:
            if required_key in data_keys:
                setattr(self, required_key, data[required_key])
            else:
                raise ApiDataError(f"Required key {required_key} missing.")

        optional_keys = ("shortname", "rss", "logo", "web", "homepage")
        for optional_key in optional_keys:
            if optional_key in data_keys:
                setattr(self, optional_key, data[optional_key])


class Story:
    """Represents a story retrieved from the API.

    See original API documentation for details (https://api.presseportal.de/en/doc/format/story).

    A ``Story`` object can contain different attributes, depending on the API query method:

        * ``data`` - The raw json data returned by the API.
        * ``id`` - The story's unique id as generated by the API.
        * ``url`` - URL of the story on the presseportal.de website.
        * ``title`` - Headline of the story.
        * ``body`` - Full text body.
        * optional: ``teaser`` - Teaser text. Only included if ``teaser=True`` is passed to the API query method. If a teaser is requested, it will replace the ``body`` attribute.
        * ``published`` - Publication date (as datetime object).
        * ``language`` - Language of the story (usually 'de' or 'en').
        * ``ressort`` - Editorial department.
        * ``company_id``/``office_id`` - Unique id of the company or office publishing the story.
        * ``company_url``/``office_url`` - URL of the company or office publishing the story.
        * ``company_name``/``office_name`` - Name of the company or office publishing the story.
        * ``keywords`` - List of keywords assigned to the story.
        * optional: ``media`` - Information on media attachments, if present.
        * ``highlight`` - Promoted story flag (on/off).
        * ``short`` - Shortened URL.
    """

    def __init__(self, data: dict):
        """Constructor method.

        Args:
            data (dict): Raw data from API request
        """
        self.data = data
        data_keys = self.data.keys()
        required_keys = ("id", "url", "title", "published", "highlight", "short")
        for required_key in required_keys:
            if required_key not in data_keys:
                raise ApiDataError(f"Required key {required_key} missing.")

        self.id = data["id"]
        self.url = data["url"]
        self.title = data["title"]
        self.published = datetime.strptime(data["published"], "%Y-%m-%dT%H:%M:%S%z")
        self.highlight = data["highlight"]
        self.short = data["short"]

        # Check whether data contains body or teaser
        if "body" in data_keys:
            self.body = data["body"]
        elif "teaser" in data_keys:
            self.teaser = data["teaser"]
        else:
            raise ApiDataError("'body' or 'teaser' not included in response.")

        if "language" in data_keys:
            self.language = data["language"]
        if "ressort" in data_keys:
            self.ressort = data["ressort"]

        # TBD: "Extended" info: https://api.presseportal.de/doc/format/company?
        # Check whether data contains company or office
        if "company" in data_keys:
            self.company_id = data["company"]["id"]
            self.company_url = data["company"]["url"]
            self.company_name = data["company"]["name"]
        elif "office" in data_keys:
            self.office_id = data["office"]["id"]
            self.office_url = data["office"]["url"]
            self.office_name = data["office"]["name"]

        # Check if keywords are present, map keywords
        if type(data["keywords"]) is dict and "keyword" in data["keywords"]:
            self.keywords = data["keywords"]["keyword"]

        # Check if media information is present
        if "media" in data_keys:
            media_keys = data["media"].keys()
            # Map media information, one list of dicts per media type
            # MEDIA_TYPES are "image", "document", "audio", "video"
            for media_type in MEDIA_TYPES:
                if media_keys and media_type in media_keys:
                    # Dynamically create attributes for media_type
                    setattr(self, media_type, data["media"][media_type])


class PresseportalApi:
    """A Python interface into the presseportal.de API.

    Presseportal is a service provided by 'news aktuell', owned by dpa
    (Deutsche Presse Agentur). An API key from presseportal.de is required.
    You can find more information and apply for an API key at
    https://api.presseportal.de/en.

    First, create an instance of the ``PresseportalApi`` class, using your API key:

    >>> from pypresseportal import PresseportalApi
    >>> api_object = PresseportalApi(YOUR_API_KEY)

    Next, request data from the API, using the various methods in this class.
    
    For example:

    Use the method get_stories() to access the most recently published stories (https://api.presseportal.de/doc/article/all).
    If no arguments are provided, PyPresseportal defaults to retrieving the 50 most 
    recent stories available. For example:

        >>> stories = api_object.get_stories()

    Story data can be accessed through the individual Story object's attributes. For example:

        >>> stories[0].title
        "Kohls Wohnhaus hat keinen Denkmalwert"
        >>> stories[0].id
        "4622388"
    """

    def __init__(self, api_key: str):
        """Constructor method.
        """
        self.data_format = "json"
        if type(api_key) is str and len(api_key) > 5:
            self.api_key = api_key
        else:
            raise ApiKeyError(api_key)

    def build_request(
        self,
        base_url: str,
        media: Union[str, None] = None,
        start: Union[int, None] = None,
        limit: Union[int, None] = None,
        teaser: Union[bool, None] = None,
        search_term: Union[str, None] = None,
    ) -> Tuple[str, Dict[str, str], Dict[str, str]]:
        """Assembles the request components to be passed on to requests.get()

        Args:
            base_url (str): Base URL for request.
            media (Union[str, None], optional): Specific media type. Defaults to None.
            start (Union[int, None], optional): Start offset. Defaults to None.
            limit (Union[int, None], optional): Maximum amount of results. Defaults to None.
            teaser (Union[bool, None], optional): Request teaser or body. Defaults to None.
            search_term (Union[str, None], optional): Search term. Defaults to None.

        Returns:
            Tuple[str, Dict[str, str], Dict[str, str]]: url, params and headers for requests.get().

        :meta private:
        """

        # Set up url and append media type, if required
        url = base_url
        if media:
            url += f"/{media.lower()}"

        # Set up params (all arguments that are not None)
        params = {
            "api_key": self.api_key,
            "format": self.data_format,
        }
        if start is not None:
            params["start"] = str(start)
        if limit is not None:
            params["limit"] = str(limit)
        if teaser is not None:
            params["teaser"] = str(int(teaser))
        if search_term is not None:
            params["q"] = search_term

        # Set up headers
        headers = {
            "User-Agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64; rv:77.0) Gecko/20100101 Firefox/77.0"
        }

        return url, params, headers

    def get_data(self, url: str, params: dict, headers: dict) -> dict:
        """Connects to API and maps raw data into objects.

        Args:
            url (str): URL for query.
            params (dict): Parameters for query.
            headers (dict): Headers for query.

        Raises:
            ApiConnectionFail: Could not connect to API.
            NotImplementedError: Unknown error.
            ApiError: API returned an error.

        Returns:
            List[Story]: List of Story objects.

        :meta private:
        """
        #######################Disable for testing ##############
        try:
            request = requests.get(url=url, params=params, headers=headers)
        except (
            requests.exceptions.ConnectionError,
            requests.exceptions.TooManyRedirects,
            requests.exceptions.Timeout,
        ) as error:
            raise ApiConnectionFail(error)
        json_data = json.loads(request.text)
        # with open("out.json", "w") as outfile:
        #     json.dump(json_data, outfile)
        #######################Enable for testing ###############
        # # read file
        # with open("out.json", "r") as in_file:
        #     data = in_file.read()
        # # parse file
        # json_data = json.loads(data)
        #########################################################

        # Raise error if API does not report success
        if "error" in json_data:
            error_code = json_data["error"]["code"]
            error_msg = json_data["error"]["msg"]
            raise ApiError(error_code, error_msg)

        return json_data

    def is_media_valid(
        self, media: Union[str, None], allowed_media_type: tuple = MEDIA_TYPES
    ) -> bool:
        """Check if media type is supported by API.

        Args:
            media (str): media type to check.
            allowed_media_type (tuple, optional): Collection of media types to check against. Defaults to MEDIA_TYPES.

        Raises:
            MediaError: API does not support the requested media type.

        Returns:
            bool: True if media is valid
        """
        if media and media.lower() not in allowed_media_type:
            raise MediaError(media, allowed_media_type)
        else:
            return True

    def parse_story_data(self, json_data: dict) -> List[Story]:
        """Parses json data into list of Story objects.

        Args:
            json_data (dict): Json data for parsing.

        Returns:
            List[Story]: List of processed Story objects.
        
        :meta private:
        """
        stories_list = []
        for item in json_data["content"]["story"]:
            stories_list.append(Story(item))

        return stories_list

    def parse_search_results(self, json_data: dict) -> Union[List[Entity], None]:
        """Parses search result data into list of Entity objects.

        Args:
            json_data (dict): Json data for parsing.

        Returns:
            Union[List[Entity], None]: Return list of processed Entity objects.
            Return None if no results.
        
        :meta private:
        """

        if "content" in json_data:
            search_results_list = []
            for item in json_data["content"]["result"]:
                search_results_list.append(Entity(item))
            return search_results_list
        else:
            return None

    def get_public_service_news(
        self, media: str = None, start: int = 0, limit: int = 50, teaser: bool = False,
    ) -> List[Story]:
        """Queries API for public service news (police and fire departments, etc.).

        Returns a list of :class:`pypresseportal.Story` objects. More information: https://api.presseportal.de/doc/article/publicservice.

        Args:
            media (str, optional): Only request stories containing this specific media type (``image`` or ``document``). Defaults to None.
            start (int, optional): Start/offset of the result article list. Defaults to 0.
            limit (int, optional): Limit number of articles in response (API maximum is 50). Defaults to 50.
            teaser (bool, optional): Returns stories with ``teaser`` instead of ``body`` (fulltext) if set to True. Defaults to False.

        Raises:
            ApiConnectionFail: Could not connect to API.
            ApiError: API returned an error.
            MediaError: API does not support the requested media type.

        Returns:
            List[Story]: List of Story objects
        """

        if self.is_media_valid(media, PUBLIC_SERVICE_MEDIA_TYPES):
            # Set up query components
            base_url = "https://api.presseportal.de/api/article/publicservice"
            url, params, headers = self.build_request(
                base_url, media, start, limit, teaser
            )

            # Query API and map results
            json_data = self.get_data(url=url, params=params, headers=headers)
            stories_list = self.parse_story_data(json_data)
        else:
            stories_list = []

        return stories_list

    def get_public_service_specific_office(
        self,
        id: str,
        media: str = None,
        start: int = 0,
        limit: int = 50,
        teaser: bool = False,
    ) -> List[Story]:
        """Queries API for stories released by a specific public service office.

        Returns a list of :class:`pypresseportal.Story` objects. More information: https://api.presseportal.de/doc/article/publicservice/office/id

        Args:
            id (str): id of office (read Entity.id of a :meth:`get_entity_search_results()` search for this id).
            media (str, optional): Only request stories containing this specific media type (``image`` or ``document``). Defaults to None.
            start (int, optional): Start/offset of the result article list. Defaults to 0.
            limit (int, optional): Limit number of articles in response (API maximum is 50). Defaults to 50.
            teaser (bool, optional): Returns stories with ``teaser`` instead of ``body`` (fulltext) if set to True. Defaults to False.

        Raises:
            ApiConnectionFail: Could not connect to API.
            ApiError: API returned an error.
            MediaError: API does not support the requested media type.

        Returns:
            List[Story]: List of Story objects
        """
        if self.is_media_valid(media, PUBLIC_SERVICE_MEDIA_TYPES):
            # Set up query components
            if type(id) is not str:
                id = str(id)
            base_url = (
                f"https://api.presseportal.de/api/article/publicservice/office/{id}"
            )
            url, params, headers = self.build_request(
                base_url, media, start, limit, teaser
            )

            # Query API and map results
            json_data = self.get_data(url=url, params=params, headers=headers)
            stories_list = self.parse_story_data(json_data)
        else:
            stories_list = []

        return stories_list

    def get_public_service_specific_region(
        self,
        region_code: str,
        media: str = None,
        start: int = 0,
        limit: int = 50,
        teaser: bool = False,
    ) -> List[Story]:
        """Queries API for stories by public service offices in a specific region.

        Returns a list of :class:`pypresseportal.Story` objects. List of region codes and more information: https://api.presseportal.de/doc/article/publicservice/region.

        Args:
            region_code (str): Only request stories located in this specific region. 
            media (str, optional): Only request stories containing this specific media type (``image`` or ``document``). Defaults to None.
            start (int, optional): Start/offset of the result article list. Defaults to 0.
            limit (int, optional): Limit number of articles in response (API maximum is 50). Defaults to 50.
            teaser (bool, optional): Returns stories with ``teaser`` instead of ``body`` (fulltext) if set to True. Defaults to False.

        Raises:
            ApiConnectionFail: Could not connect to API.
            ApiError: API returned an error.
            MediaError: API does not support the requested media type.
            RegionError: API does not support the requested region code.

        Returns:
            List[Story]: List of Story objects
        """
        # Check if region is supported by API
        if region_code not in PUBLIC_SERVICE_REGIONS:
            raise RegionError(region_code, PUBLIC_SERVICE_REGIONS)

        if self.is_media_valid(media, PUBLIC_SERVICE_MEDIA_TYPES):
            # Set up query components
            base_url = f"https://api.presseportal.de/api/article/publicservice/region/{region_code}"
            url, params, headers = self.build_request(
                base_url, media, start, limit, teaser
            )

            # Query API and map results
            json_data = self.get_data(url=url, params=params, headers=headers)
            stories_list = self.parse_story_data(json_data)
        else:
            stories_list = []

        return stories_list

    def get_stories(
        self, media: str = None, start: int = 0, limit: int = 50, teaser: bool = False,
    ) -> List[Story]:
        """Queries API for most recent press releases.

        Returns a list of :class:`pypresseportal.Story` objects. More information: https://api.presseportal.de/doc/article/all.

        Args:
            media (str, optional): Only request stories containing this specific media type (``image``, ``document``, ``audio`` or ``video``). Defaults to None.
            start (int, optional): Start/offset of the result article list. Defaults to 0.
            limit (int, optional): Limit number of articles in response (API maximum is 50). Defaults to 50.
            teaser (bool, optional): Returns stories with ``teaser`` instead of ``body`` (fulltext) if set to True. Defaults to False.

        Raises:
            ApiConnectionFail: Could not connect to API.
            ApiError: API returned an error.
            MediaError: API does not support the requested media type.

        Returns:
            List[Story]: List of Story objects
        """
        if self.is_media_valid(media):
            # Set up query components
            base_url = "https://api.presseportal.de/api/article/all"
            url, params, headers = self.build_request(
                base_url, media, start, limit, teaser
            )
            # Query API and map results
            json_data = self.get_data(url=url, params=params, headers=headers)
            stories_list = self.parse_story_data(json_data)
        else:
            stories_list = []

        return stories_list

    def get_stories_specific_company(
        self,
        id: str,
        media: str = None,
        start: int = 0,
        limit: int = 50,
        teaser: bool = False,
    ) -> List[Story]:
        """Queries API for press releases of a specific company.

        Returns a list of :class:`pypresseportal.Story` objects. More information: https://api.presseportal.de/en/doc/article/company/id

        Args:
            id (str): id of company (read Entity.id of a :meth:`get_entity_search_results()` search for this id).
            media (str, optional): Only request stories containing this specific media type (``image``, ``document``, ``audio`` or ``video``). Defaults to None.
            start (int, optional): Start/offset of the result article list. Defaults to 0.
            limit (int, optional): Limit number of articles in response (API maximum is 50). Defaults to 50.
            teaser (bool, optional): Returns stories with ``teaser`` instead of ``body`` (fulltext) if set to True. Defaults to False.

        Raises:
            ApiConnectionFail: Could not connect to API.
            ApiError: API returned an error.
            MediaError: API does not support the requested media type.

        Returns:
            List[Story]: List of Story objects
        """
        if self.is_media_valid(media):
            # Set up query components
            if type(id) is not str:
                id = str(id)
            base_url = f"https://api.presseportal.de/api/article/company/{id}"
            url, params, headers = self.build_request(
                base_url, media, start, limit, teaser
            )

            # Query API and map results
            json_data = self.get_data(url=url, params=params, headers=headers)
            stories_list = self.parse_story_data(json_data)
        else:
            stories_list = []

        return stories_list

    def get_stories_topic(
        self,
        topic: str,
        media: str = None,
        start: int = 0,
        limit: int = 50,
        teaser: bool = False,
    ) -> List[Story]:
        """Queries API for most recent press releases assigned to a specific topic.
        
        Returns a list of :class:`pypresseportal.Story` objects. More information: https://api.presseportal.de/doc/article/topic/ident

        Args:
            topic (str): One specific topic from https://api.presseportal.de/doc/value/topic
            media (str, optional): Only request stories containing this specific media type (``image``, ``document``, ``audio`` or ``video``). Defaults to None.
            start (int, optional): Start/offset of the result article list. Defaults to 0.
            limit (int, optional): Limit number of articles in response (API maximum is 50). Defaults to 50.
            teaser (bool, optional): Returns stories with ``teaser`` instead of ``body`` (fulltext) if set to True. Defaults to False.

        Raises:
            ApiConnectionFail: Could not connect to API.
            ApiError: API returned an error.
            MediaError: API does not support the requested media type.
            TopicError: API does not support the requested topic.

        Returns:
            List[Story]: List of Story objects
        """
        # Check if topic is supported by API
        if topic not in TOPICS:
            raise TopicError(topic, TOPICS)

        if self.is_media_valid(media):
            # Set up query components
            base_url = f"https://api.presseportal.de/api/article/topic/{topic}"
            url, params, headers = self.build_request(
                base_url, media, start, limit, teaser
            )

            # Query API and map results
            json_data = self.get_data(url=url, params=params, headers=headers)
            stories_list = self.parse_story_data(json_data)
        else:
            stories_list = []

        return stories_list

    def get_stories_keywords(
        self,
        keywords: List[str],
        media: str = None,
        start: int = 0,
        limit: int = 50,
        teaser: bool = False,
    ) -> List[Story]:
        """Queries API for most recent press releases assigned to specific keywords.

        Returns a list of :class:`pypresseportal.Story` objects. More information: https://api.presseportal.de/doc/article/keyword/ident

        Args:
            keywords (List[str]):  A list of one ore more keywords from https://api.presseportal.de/doc/value/keyword
            media (str, optional): Only request stories containing this specific media type (``image``, ``document``, ``audio`` or ``video``). Defaults to None.
            start (int, optional): Start/offset of the result article list. Defaults to 0.
            limit (int, optional): Limit number of articles in response (API maximum is 50). Defaults to 50.
            teaser (bool, optional): Returns stories with ``teaser`` instead of ``body`` (fulltext) if set to True. Defaults to False.

        Raises:
            ApiConnectionFail: Could not connect to API.
            ApiError: API returned an error.
            MediaError: API does not support the requested media type.
            KeywordError: API does not support the requested keyword(s).

        Returns:
            List[Story]: List of Story objects
        """
        # Check if keywords are supported by API
        for keyword in keywords:
            if keyword not in KEYWORDS:
                raise KeywordError(keyword, KEYWORDS)

        # Construct keyword string
        keywords_str = ",".join(keywords)

        if self.is_media_valid(media):
            # Set up query components
            base_url = f"https://api.presseportal.de/api/article/keyword/{keywords_str}"
            url, params, headers = self.build_request(
                base_url, media, start, limit, teaser
            )

            # Query API and map results
            json_data = self.get_data(url=url, params=params, headers=headers)
            stories_list = self.parse_story_data(json_data)
        else:
            stories_list = []

        return stories_list

    def get_investor_relations_news(
        self,
        news_type: str = "all",
        start: int = 0,
        limit: int = 50,
        teaser: bool = False,
    ) -> List[Story]:
        """Queries API for most recent investor relations press releases.

        Returns a list of :class:`pypresseportal.Story` objects. More information: https://api.presseportal.de/doc/ir/list

        Args:
            news_type (str, optional): Investor relations news type (https://api.presseportal.de/doc/value/ir_type). Defaults to "all".
            start (int, optional): Start/offset of the result article list. Defaults to 0.
            limit (int, optional): Limit number of articles in response (API maximum is 50). Defaults to 50.
            teaser (bool, optional): Returns stories with ``teaser`` instead of ``body`` (fulltext) if set to True. Defaults to False.

        Raises:
            ApiConnectionFail: Could not connect to API.
            ApiError: API returned an error.
            NewsTypeError: API does not support the requested news type.

        Returns:
            List[Story]: List of Story objects
        """

        # Check if investor relations news type is supported by API
        if news_type.lower() not in INVESTOR_RELATIONS_NEWS_TYPES:
            raise NewsTypeError(news_type, INVESTOR_RELATIONS_NEWS_TYPES)

        # Set up query components
        base_url = f"https://api.presseportal.de/api/ir/{news_type.lower()}"
        url, params, headers = self.build_request(
            base_url=base_url, media=None, start=start, limit=limit, teaser=teaser
        )

        # Query API and map results
        json_data = self.get_data(url=url, params=params, headers=headers)
        stories_list = self.parse_story_data(json_data)

        return stories_list

    def get_investor_relations_news_company(
        self,
        id: str,
        news_type: str = "all",
        start: int = 0,
        limit: int = 50,
        teaser: bool = False,
    ) -> List[Story]:
        """Queries API for investor relations press releases of a specific company.

        Returns a list of :class:`pypresseportal.Story` objects. More information: https://api.presseportal.de/doc/ir/company/id/list

        Args:
            id (str): id of company (read Entity.id of a :meth:`get_entity_search_results()` search for this id).
            news_type (str, optional): Investor relations news type (https://api.presseportal.de/doc/value/ir_type). Defaults to "all".
            start (int, optional): Start/offset of the result article list. Defaults to 0.
            limit (int, optional): Limit number of articles in response (API maximum is 50). Defaults to 50.
            teaser (bool, optional): Returns stories with ``teaser`` instead of ``body`` (fulltext) if set to True. Defaults to False.

        Raises:
            ApiConnectionFail: Could not connect to API.
            ApiError: API returned an error.
            NewsTypeError: API does not support the requested news type.

        Returns:
            List[Story]: List of Story objects
        """

        # Check if investor relations news type is supported by API
        if news_type.lower() not in INVESTOR_RELATIONS_NEWS_TYPES:
            raise NewsTypeError(news_type, INVESTOR_RELATIONS_NEWS_TYPES)

        # Set up query components
        if type(id) is not str:
            id = str(id)
        base_url = (
            f"https://api.presseportal.de/api/ir/company/{id}/{news_type.lower()}"
        )
        url, params, headers = self.build_request(
            base_url=base_url, media=None, start=start, limit=limit, teaser=teaser
        )

        # Query API and map results
        json_data = self.get_data(url=url, params=params, headers=headers)
        stories_list = self.parse_story_data(json_data)

        return stories_list

    def get_entity_search_results(
        self,
        search_term: Union[str, List[str]],
        entity: str = "company",
        limit: int = 20,
    ) -> Union[List[Entity], None]:
        """Search for company or public service office by location or name.

        Returns a list of :class:`pypresseportal.Entity` objects. More information: https://api.presseportal.de/doc/search/company

        Args:
            search_term (Union[str, List[str]]): One or multiple search terms (min 3 chars.).
            entity (str, optional): Search for 'office' or 'company'. Defaults to "company".
            limit (int, optional): Limit number of objects in response (API maximum is 20). Defaults to 20.

        Raises:
            ApiConnectionFail: Could not connect to API.
            ApiError: API returned an error.
            SearchTermError: Api does not support the requested search term.
            SearchEntityError: Api does not support the requested entity.

        Returns:
            Union[List[Entity], None]: List of Entity objects. None if nothing found.
        """

        # Check search term
        if isinstance(search_term, list):
            search_term = ",".join(search_term).lower()
        elif isinstance(search_term, str) and len(search_term) > 3:
            search_term = search_term.lower()
        else:
            raise SearchTermError(search_term)

        # Check entity and define base_url
        if entity.lower() == "office":
            base_url = "https://api.presseportal.de/api/search/office"
        elif entity.lower() == "company":
            base_url = "https://api.presseportal.de/api/search/company"
        else:
            raise SearchEntityError(entity)

        # Set up query components
        url, params, headers = self.build_request(
            base_url=base_url,
            media=None,
            start=None,
            limit=limit,
            teaser=None,
            search_term=search_term,
        )

        # Query API and map results
        json_data = self.get_data(url=url, params=params, headers=headers)
        search_results_list = self.parse_search_results(json_data)

        return search_results_list

    def get_company_information(self, id: str) -> Company:
        """Queries API for detailed information about a specific company.

        Returns a :class:`pypresseportal.Company` object. More information: https://api.presseportal.de/doc/info/company/id

        Args:
            id (str): id of company (read Entity.id of a :meth:`get_entity_search_results()` search for this id).

        Raises:
            ApiConnectionFail: Could not connect to API.
            ApiError: API returned an error.

        Returns:
            Company: An object containing details about the requested company.
        """

        # Check id and define base_url
        if type(id) is not str:
            id = str(id)
        base_url = f"https://api.presseportal.de/api/info/company/{id}"

        # Set up query components
        url, params, headers = self.build_request(base_url=base_url)

        # Query API and map results
        json_data = self.get_data(url=url, params=params, headers=headers)
        company_info = Company(json_data["company"])

        return company_info

    def get_public_service_office_information(self, id: str) -> Office:
        """Queries API for detailed information about a specific public service office (police or fire deparment, etc.).

        Returns a :class:`pypresseportal.Office` object. More information: https://api.presseportal.de/doc/info/office/id

        Args:
            id (str): id of office (read Entity.id of a :meth:`get_entity_search_results()` search for this id).

        Raises:
            ApiConnectionFail: Could not connect to API.
            ApiError: API returned an error.

        Returns:
            Office: An object containing details about the requested office.
        """

        # Check id and define base_url
        if type(id) is not str:
            id = str(id)
        base_url = f"https://api.presseportal.de/api/info/office/{id}"

        # Set up query components
        url, params, headers = self.build_request(base_url=base_url)

        # Query API and map results
        json_data = self.get_data(url=url, params=params, headers=headers)
        office_info = Office(json_data["office"])

        return office_info
