"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
// tslint:disable-next-line:max-line-length
const lib_1 = require("../lib");
module.exports = {
    'gateway endpoint': {
        'add an endpoint to a vpc'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            new lib_1.Vpc(stack, 'VpcNetwork', {
                gatewayEndpoints: {
                    S3: {
                        service: lib_1.GatewayVpcEndpointAwsService.S3
                    }
                }
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                ServiceName: {
                    'Fn::Join': [
                        '',
                        [
                            'com.amazonaws.',
                            {
                                Ref: 'AWS::Region'
                            },
                            '.s3'
                        ]
                    ]
                },
                VpcId: {
                    Ref: 'VpcNetworkB258E83A'
                },
                RouteTableIds: [
                    {
                        Ref: 'VpcNetworkPrivateSubnet1RouteTableCD085FF1'
                    },
                    {
                        Ref: 'VpcNetworkPrivateSubnet2RouteTableE97B328B'
                    },
                ],
                VpcEndpointType: 'Gateway'
            }));
            test.done();
        },
        'routing on private and public subnets'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            new lib_1.Vpc(stack, 'VpcNetwork', {
                gatewayEndpoints: {
                    S3: {
                        service: lib_1.GatewayVpcEndpointAwsService.S3,
                        subnets: [
                            {
                                subnetType: lib_1.SubnetType.PUBLIC
                            },
                            {
                                subnetType: lib_1.SubnetType.PRIVATE
                            }
                        ]
                    }
                }
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                ServiceName: {
                    'Fn::Join': [
                        '',
                        [
                            'com.amazonaws.',
                            {
                                Ref: 'AWS::Region'
                            },
                            '.s3'
                        ]
                    ]
                },
                VpcId: {
                    Ref: 'VpcNetworkB258E83A'
                },
                RouteTableIds: [
                    {
                        Ref: 'VpcNetworkPublicSubnet1RouteTable25CCC53F'
                    },
                    {
                        Ref: 'VpcNetworkPublicSubnet2RouteTableE5F348DF'
                    },
                    {
                        Ref: 'VpcNetworkPrivateSubnet1RouteTableCD085FF1'
                    },
                    {
                        Ref: 'VpcNetworkPrivateSubnet2RouteTableE97B328B'
                    },
                ],
                VpcEndpointType: 'Gateway'
            }));
            test.done();
        },
        'add statements to policy'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            const endpoint = vpc.addGatewayEndpoint('S3', {
                service: lib_1.GatewayVpcEndpointAwsService.S3
            });
            // WHEN
            endpoint.addToPolicy(new aws_iam_1.PolicyStatement({
                principals: [new aws_iam_1.AnyPrincipal()],
                actions: ['s3:GetObject', 's3:ListBucket'],
                resources: ['*']
            }));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                's3:GetObject',
                                's3:ListBucket'
                            ],
                            Effect: 'Allow',
                            Principal: '*',
                            Resource: '*'
                        }
                    ],
                    Version: '2012-10-17'
                }
            }));
            test.done();
        },
        'throws when adding a statement without a principal'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            const endpoint = vpc.addGatewayEndpoint('S3', {
                service: lib_1.GatewayVpcEndpointAwsService.S3
            });
            // THEN
            test.throws(() => endpoint.addToPolicy(new aws_iam_1.PolicyStatement({
                actions: ['s3:GetObject', 's3:ListBucket'],
                resources: ['*']
            })), /`Principal`/);
            test.done();
        },
        'import/export'(test) {
            // GIVEN
            const stack2 = new core_1.Stack();
            // WHEN
            const ep = lib_1.GatewayVpcEndpoint.fromGatewayVpcEndpointId(stack2, 'ImportedEndpoint', 'endpoint-id');
            // THEN
            test.deepEqual(ep.vpcEndpointId, 'endpoint-id');
            test.done();
        },
        'works with an imported vpc'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = lib_1.Vpc.fromVpcAttributes(stack, 'VPC', {
                vpcId: 'id',
                privateSubnetIds: ['1', '2', '3'],
                privateSubnetRouteTableIds: ['rt1', 'rt2', 'rt3'],
                availabilityZones: ['a', 'b', 'c']
            });
            // THEN
            vpc.addGatewayEndpoint('Gateway', { service: lib_1.GatewayVpcEndpointAwsService.S3 });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                ServiceName: { 'Fn::Join': ['', ['com.amazonaws.', { Ref: 'AWS::Region' }, '.s3']] },
                VpcId: 'id',
                RouteTableIds: ['rt1', 'rt2', 'rt3'],
                VpcEndpointType: 'Gateway',
            }));
            test.done();
        },
        'throws with an imported vpc without route tables ids'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = lib_1.Vpc.fromVpcAttributes(stack, 'VPC', {
                vpcId: 'id',
                privateSubnetIds: ['1', '2', '3'],
                availabilityZones: ['a', 'b', 'c']
            });
            test.throws(() => vpc.addGatewayEndpoint('Gateway', { service: lib_1.GatewayVpcEndpointAwsService.S3 }), /route table/);
            test.done();
        }
    },
    'interface endpoint': {
        'add an endpoint to a vpc'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            // WHEN
            vpc.addInterfaceEndpoint('EcrDocker', {
                service: lib_1.InterfaceVpcEndpointAwsService.ECR_DOCKER
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                ServiceName: {
                    'Fn::Join': [
                        '',
                        [
                            'com.amazonaws.',
                            {
                                Ref: 'AWS::Region'
                            },
                            '.ecr.dkr'
                        ]
                    ]
                },
                VpcId: {
                    Ref: 'VpcNetworkB258E83A'
                },
                PrivateDnsEnabled: true,
                SecurityGroupIds: [
                    {
                        'Fn::GetAtt': [
                            'VpcNetworkEcrDockerSecurityGroup7C91D347',
                            'GroupId'
                        ]
                    }
                ],
                SubnetIds: [
                    {
                        Ref: 'VpcNetworkPrivateSubnet1Subnet07BA143B'
                    },
                    {
                        Ref: 'VpcNetworkPrivateSubnet2Subnet5E4189D6'
                    },
                ],
                VpcEndpointType: 'Interface'
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
                GroupDescription: 'VpcNetwork/EcrDocker/SecurityGroup',
                VpcId: {
                    Ref: 'VpcNetworkB258E83A'
                }
            }));
            test.done();
        },
        'import/export'(test) {
            // GIVEN
            const stack2 = new core_1.Stack();
            // WHEN
            const importedEndpoint = lib_1.InterfaceVpcEndpoint.fromInterfaceVpcEndpointAttributes(stack2, 'ImportedEndpoint', {
                securityGroupId: 'security-group-id',
                vpcEndpointId: 'vpc-endpoint-id',
                port: 80
            });
            importedEndpoint.connections.allowDefaultPortFromAnyIpv4();
            // THEN
            assert_1.expect(stack2).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
                GroupId: 'security-group-id'
            }));
            test.deepEqual(importedEndpoint.vpcEndpointId, 'vpc-endpoint-id');
            test.done();
        }
    }
};
//# sourceMappingURL=data:application/json;base64,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