"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
class SomethingConnectable {
    constructor(connections) {
        this.connections = connections;
    }
}
module.exports = {
    'peering between two security groups does not recursive infinitely'(test) {
        // GIVEN
        const stack = new core_1.Stack(undefined, 'TestStack', { env: { account: '12345678', region: 'dummy' } });
        const vpc = new lib_1.Vpc(stack, 'VPC');
        const sg1 = new lib_1.SecurityGroup(stack, 'SG1', { vpc });
        const sg2 = new lib_1.SecurityGroup(stack, 'SG2', { vpc });
        const conn1 = new SomethingConnectable(new lib_1.Connections({ securityGroups: [sg1] }));
        const conn2 = new SomethingConnectable(new lib_1.Connections({ securityGroups: [sg2] }));
        // WHEN
        conn1.connections.allowTo(conn2, lib_1.Port.tcp(80), 'Test');
        // THEN -- it finishes!
        test.done();
    },
    '(imported) SecurityGroup can be used as target of .allowTo()'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        const sg1 = new lib_1.SecurityGroup(stack, 'SomeSecurityGroup', { vpc, allowAllOutbound: false });
        const somethingConnectable = new SomethingConnectable(new lib_1.Connections({ securityGroups: [sg1] }));
        const securityGroup = lib_1.SecurityGroup.fromSecurityGroupId(stack, 'ImportedSG', 'sg-12345');
        // WHEN
        somethingConnectable.connections.allowTo(securityGroup, lib_1.Port.allTcp(), 'Connect there');
        // THEN: rule to generated security group to connect to imported
        assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::SecurityGroupEgress", {
            GroupId: { "Fn::GetAtt": ["SomeSecurityGroupEF219AD6", "GroupId"] },
            IpProtocol: "tcp",
            Description: "Connect there",
            DestinationSecurityGroupId: "sg-12345",
            FromPort: 0,
            ToPort: 65535
        }));
        // THEN: rule to imported security group to allow connections from generated
        assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::SecurityGroupIngress", {
            IpProtocol: "tcp",
            Description: "Connect there",
            FromPort: 0,
            GroupId: "sg-12345",
            SourceSecurityGroupId: { "Fn::GetAtt": ["SomeSecurityGroupEF219AD6", "GroupId"] },
            ToPort: 65535
        }));
        test.done();
    },
    'security groups added to connections after rule still gets rule'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        const sg1 = new lib_1.SecurityGroup(stack, 'SecurityGroup1', { vpc, allowAllOutbound: false });
        const sg2 = new lib_1.SecurityGroup(stack, 'SecurityGroup2', { vpc, allowAllOutbound: false });
        const connections = new lib_1.Connections({ securityGroups: [sg1] });
        // WHEN
        connections.allowFromAnyIpv4(lib_1.Port.tcp(88));
        connections.addSecurityGroup(sg2);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            GroupDescription: "SecurityGroup1",
            SecurityGroupIngress: [
                {
                    Description: "from 0.0.0.0/0:88",
                    CidrIp: "0.0.0.0/0",
                    FromPort: 88,
                    ToPort: 88,
                    IpProtocol: 'tcp'
                }
            ]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            GroupDescription: "SecurityGroup2",
            SecurityGroupIngress: [
                {
                    Description: "from 0.0.0.0/0:88",
                    CidrIp: "0.0.0.0/0",
                    FromPort: 88,
                    ToPort: 88,
                    IpProtocol: 'tcp'
                }
            ]
        }));
        test.done();
    },
    'when security groups are added to target they also get the rule'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        const sg1 = new lib_1.SecurityGroup(stack, 'SecurityGroup1', { vpc, allowAllOutbound: false });
        const sg2 = new lib_1.SecurityGroup(stack, 'SecurityGroup2', { vpc, allowAllOutbound: false });
        const sg3 = new lib_1.SecurityGroup(stack, 'SecurityGroup3', { vpc, allowAllOutbound: false });
        const connections1 = new lib_1.Connections({ securityGroups: [sg1] });
        const connections2 = new lib_1.Connections({ securityGroups: [sg2] });
        const connectable = new SomethingConnectable(connections2);
        // WHEN
        connections1.allowTo(connectable, lib_1.Port.tcp(88));
        connections2.addSecurityGroup(sg3);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
            GroupId: { "Fn::GetAtt": ["SecurityGroup23BE86BB7", "GroupId"] },
            SourceSecurityGroupId: { "Fn::GetAtt": ["SecurityGroup1F554B36F", "GroupId"] },
            FromPort: 88,
            ToPort: 88
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
            GroupId: { "Fn::GetAtt": ["SecurityGroup3E5E374B9", "GroupId"] },
            SourceSecurityGroupId: { "Fn::GetAtt": ["SecurityGroup1F554B36F", "GroupId"] },
            FromPort: 88,
            ToPort: 88
        }));
        test.done();
    },
    'multiple security groups allows internally between them'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        const sg1 = new lib_1.SecurityGroup(stack, 'SecurityGroup1', { vpc, allowAllOutbound: false });
        const sg2 = new lib_1.SecurityGroup(stack, 'SecurityGroup2', { vpc, allowAllOutbound: false });
        const connections = new lib_1.Connections({ securityGroups: [sg1] });
        // WHEN
        connections.allowInternally(lib_1.Port.tcp(88));
        connections.addSecurityGroup(sg2);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
            GroupId: { "Fn::GetAtt": ["SecurityGroup1F554B36F", "GroupId"] },
            SourceSecurityGroupId: { "Fn::GetAtt": ["SecurityGroup1F554B36F", "GroupId"] },
            FromPort: 88,
            ToPort: 88
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            DestinationSecurityGroupId: { "Fn::GetAtt": ["SecurityGroup1F554B36F", "GroupId"] },
            GroupId: { "Fn::GetAtt": ["SecurityGroup1F554B36F", "GroupId"] },
            FromPort: 88,
            ToPort: 88
        }));
        test.done();
    },
    'can establish cross stack Security Group connections - allowFrom'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack1 = new core_1.Stack(app, 'Stack1');
        const vpc1 = new lib_1.Vpc(stack1, 'VPC');
        const sg1 = new lib_1.SecurityGroup(stack1, 'SecurityGroup', { vpc: vpc1, allowAllOutbound: false });
        const stack2 = new core_1.Stack(app, 'Stack2');
        const vpc2 = new lib_1.Vpc(stack2, 'VPC');
        const sg2 = new lib_1.SecurityGroup(stack2, 'SecurityGroup', { vpc: vpc2, allowAllOutbound: false });
        // WHEN
        sg2.connections.allowFrom(sg1, lib_1.Port.tcp(100));
        // THEN -- both rules are in Stack2
        core_1.ConstructNode.prepare(app.node);
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
            GroupId: { "Fn::GetAtt": ["SecurityGroupDD263621", "GroupId"] },
            SourceSecurityGroupId: { "Fn::ImportValue": "Stack1:ExportsOutputFnGetAttSecurityGroupDD263621GroupIdDF6F8B09" },
        }));
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            GroupId: { "Fn::ImportValue": "Stack1:ExportsOutputFnGetAttSecurityGroupDD263621GroupIdDF6F8B09" },
            DestinationSecurityGroupId: { "Fn::GetAtt": ["SecurityGroupDD263621", "GroupId"] },
        }));
        test.done();
    },
    'can establish cross stack Security Group connections - allowTo'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack1 = new core_1.Stack(app, 'Stack1');
        const vpc1 = new lib_1.Vpc(stack1, 'VPC');
        const sg1 = new lib_1.SecurityGroup(stack1, 'SecurityGroup', { vpc: vpc1, allowAllOutbound: false });
        const stack2 = new core_1.Stack(app, 'Stack2');
        const vpc2 = new lib_1.Vpc(stack2, 'VPC');
        const sg2 = new lib_1.SecurityGroup(stack2, 'SecurityGroup', { vpc: vpc2, allowAllOutbound: false });
        // WHEN
        sg2.connections.allowTo(sg1, lib_1.Port.tcp(100));
        // THEN -- both rules are in Stack2
        core_1.ConstructNode.prepare(app.node);
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
            GroupId: { "Fn::ImportValue": "Stack1:ExportsOutputFnGetAttSecurityGroupDD263621GroupIdDF6F8B09" },
            SourceSecurityGroupId: { "Fn::GetAtt": ["SecurityGroupDD263621", "GroupId"] },
        }));
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            GroupId: { "Fn::GetAtt": ["SecurityGroupDD263621", "GroupId"] },
            DestinationSecurityGroupId: { "Fn::ImportValue": "Stack1:ExportsOutputFnGetAttSecurityGroupDD263621GroupIdDF6F8B09" },
        }));
        test.done();
    },
    'can establish multiple cross-stack SGs'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack1 = new core_1.Stack(app, 'Stack1');
        const vpc1 = new lib_1.Vpc(stack1, 'VPC');
        const sg1a = new lib_1.SecurityGroup(stack1, 'SecurityGroupA', { vpc: vpc1, allowAllOutbound: false });
        const sg1b = new lib_1.SecurityGroup(stack1, 'SecurityGroupB', { vpc: vpc1, allowAllOutbound: false });
        const stack2 = new core_1.Stack(app, 'Stack2');
        const vpc2 = new lib_1.Vpc(stack2, 'VPC');
        const sg2 = new lib_1.SecurityGroup(stack2, 'SecurityGroup', { vpc: vpc2, allowAllOutbound: false });
        // WHEN
        sg2.connections.allowFrom(sg1a, lib_1.Port.tcp(100));
        sg2.connections.allowFrom(sg1b, lib_1.Port.tcp(100));
        // THEN -- both egress rules are in Stack2
        core_1.ConstructNode.prepare(app.node);
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            GroupId: { "Fn::ImportValue": "Stack1:ExportsOutputFnGetAttSecurityGroupAED40ADC5GroupId1D10C76A" },
            DestinationSecurityGroupId: { "Fn::GetAtt": ["SecurityGroupDD263621", "GroupId"] },
        }));
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            GroupId: { "Fn::ImportValue": "Stack1:ExportsOutputFnGetAttSecurityGroupB04591F90GroupIdFA7208D5" },
            DestinationSecurityGroupId: { "Fn::GetAtt": ["SecurityGroupDD263621", "GroupId"] },
        }));
        test.done();
    },
    'Imported SecurityGroup does not create egress rule'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        const sg1 = new lib_1.SecurityGroup(stack, 'SomeSecurityGroup', { vpc, allowAllOutbound: false });
        const somethingConnectable = new SomethingConnectable(new lib_1.Connections({ securityGroups: [sg1] }));
        const securityGroup = lib_1.SecurityGroup.fromSecurityGroupId(stack, 'ImportedSG', 'sg-12345');
        // WHEN
        somethingConnectable.connections.allowFrom(securityGroup, lib_1.Port.allTcp(), 'Connect there');
        // THEN: rule to generated security group to connect to imported
        assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::SecurityGroupIngress", {
            GroupId: { "Fn::GetAtt": ["SomeSecurityGroupEF219AD6", "GroupId"] },
            IpProtocol: "tcp",
            Description: "Connect there",
            SourceSecurityGroupId: "sg-12345",
            FromPort: 0,
            ToPort: 65535
        }));
        // THEN: rule to imported security group to allow connections from generated
        assert_1.expect(stack).notTo(assert_1.haveResource("AWS::EC2::SecurityGroupEgress"));
        test.done();
    },
    'Imported SecurityGroup with allowAllOutbound: false DOES create egress rule'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        const sg1 = new lib_1.SecurityGroup(stack, 'SomeSecurityGroup', { vpc, allowAllOutbound: false });
        const somethingConnectable = new SomethingConnectable(new lib_1.Connections({ securityGroups: [sg1] }));
        const securityGroup = lib_1.SecurityGroup.fromSecurityGroupId(stack, 'ImportedSG', 'sg-12345', {
            allowAllOutbound: false
        });
        // WHEN
        somethingConnectable.connections.allowFrom(securityGroup, lib_1.Port.allTcp(), 'Connect there');
        // THEN: rule to generated security group to connect to imported
        assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::SecurityGroupIngress", {
            GroupId: { "Fn::GetAtt": ["SomeSecurityGroupEF219AD6", "GroupId"] },
            IpProtocol: "tcp",
            Description: "Connect there",
            SourceSecurityGroupId: "sg-12345",
            FromPort: 0,
            ToPort: 65535
        }));
        // THEN: rule to imported security group to allow connections from generated
        assert_1.expect(stack).to(assert_1.haveResource("AWS::EC2::SecurityGroupEgress", {
            IpProtocol: "tcp",
            Description: "Connect there",
            FromPort: 0,
            GroupId: "sg-12345",
            DestinationSecurityGroupId: { "Fn::GetAtt": ["SomeSecurityGroupEF219AD6", "GroupId"] },
            ToPort: 65535
        }));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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