"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const port_1 = require("./port");
const security_group_1 = require("./security-group");
const util_1 = require("./util");
const vpc_1 = require("./vpc");
class VpcEndpoint extends core_1.Resource {
    /**
     * Adds a statement to the policy document of the VPC endpoint. The statement
     * must have a Principal.
     *
     * Not all interface VPC endpoints support policy. For more information
     * see https://docs.aws.amazon.com/vpc/latest/userguide/vpce-interface.html
     *
     * @param statement the IAM statement to add
     */
    addToPolicy(statement) {
        if (!statement.hasPrincipal) {
            throw new Error('Statement must have a `Principal`.');
        }
        if (!this.policyDocument) {
            this.policyDocument = new iam.PolicyDocument();
        }
        this.policyDocument.addStatements(statement);
    }
}
exports.VpcEndpoint = VpcEndpoint;
/**
 * The type of VPC endpoint.
 */
var VpcEndpointType;
(function (VpcEndpointType) {
    /**
     * Interface
     *
     * An interface endpoint is an elastic network interface with a private IP
     * address that serves as an entry point for traffic destined to a supported
     * service.
     */
    VpcEndpointType["INTERFACE"] = "Interface";
    /**
     * Gateway
     *
     * A gateway endpoint is a gateway that is a target for a specified route in
     * your route table, used for traffic destined to a supported AWS service.
     */
    VpcEndpointType["GATEWAY"] = "Gateway";
})(VpcEndpointType = exports.VpcEndpointType || (exports.VpcEndpointType = {}));
/**
 * An AWS service for a gateway VPC endpoint.
 */
class GatewayVpcEndpointAwsService {
    constructor(name, prefix) {
        this.name = `${prefix || 'com.amazonaws'}.${core_1.Aws.REGION}.${name}`;
    }
}
exports.GatewayVpcEndpointAwsService = GatewayVpcEndpointAwsService;
GatewayVpcEndpointAwsService.DYNAMODB = new GatewayVpcEndpointAwsService('dynamodb');
GatewayVpcEndpointAwsService.S3 = new GatewayVpcEndpointAwsService('s3');
/**
 * A gateway VPC endpoint.
 * @resource AWS::EC2::VPCEndpoint
 */
class GatewayVpcEndpoint extends VpcEndpoint {
    constructor(scope, id, props) {
        super(scope, id);
        const subnets = props.subnets || [{ subnetType: vpc_1.SubnetType.PRIVATE }];
        const routeTableIds = util_1.allRouteTableIds(...subnets.map(s => props.vpc.selectSubnets(s)));
        if (routeTableIds.length === 0) {
            throw new Error(`Can't add a gateway endpoint to VPC; route table IDs are not available`);
        }
        const endpoint = new ec2_generated_1.CfnVPCEndpoint(this, 'Resource', {
            policyDocument: core_1.Lazy.anyValue({ produce: () => this.policyDocument }),
            routeTableIds,
            serviceName: props.service.name,
            vpcEndpointType: VpcEndpointType.GATEWAY,
            vpcId: props.vpc.vpcId
        });
        this.vpcEndpointId = endpoint.ref;
        this.vpcEndpointCreationTimestamp = endpoint.attrCreationTimestamp;
        this.vpcEndpointDnsEntries = endpoint.attrDnsEntries;
        this.vpcEndpointNetworkInterfaceIds = endpoint.attrNetworkInterfaceIds;
    }
    static fromGatewayVpcEndpointId(scope, id, gatewayVpcEndpointId) {
        class Import extends VpcEndpoint {
            constructor() {
                super(...arguments);
                this.vpcEndpointId = gatewayVpcEndpointId;
            }
        }
        return new Import(scope, id);
    }
}
exports.GatewayVpcEndpoint = GatewayVpcEndpoint;
/**
 * An AWS service for an interface VPC endpoint.
 */
class InterfaceVpcEndpointAwsService {
    constructor(name, prefix, port) {
        this.name = `${prefix || 'com.amazonaws'}.${core_1.Aws.REGION}.${name}`;
        this.port = port || 443;
    }
}
exports.InterfaceVpcEndpointAwsService = InterfaceVpcEndpointAwsService;
InterfaceVpcEndpointAwsService.SAGEMAKER_NOTEBOOK = new InterfaceVpcEndpointAwsService('notebook', 'aws.sagemaker');
InterfaceVpcEndpointAwsService.CLOUDFORMATION = new InterfaceVpcEndpointAwsService('cloudformation');
InterfaceVpcEndpointAwsService.CLOUDTRAIL = new InterfaceVpcEndpointAwsService('cloudtrail');
InterfaceVpcEndpointAwsService.CODEBUILD = new InterfaceVpcEndpointAwsService('codebuild');
InterfaceVpcEndpointAwsService.CODEBUILD_FIPS = new InterfaceVpcEndpointAwsService('codebuil-fips');
InterfaceVpcEndpointAwsService.CODECOMMIT = new InterfaceVpcEndpointAwsService('codecommit');
InterfaceVpcEndpointAwsService.CODECOMMIT_FIPS = new InterfaceVpcEndpointAwsService('codecommit-fips');
InterfaceVpcEndpointAwsService.CODEPIPELINE = new InterfaceVpcEndpointAwsService('codepipeline');
InterfaceVpcEndpointAwsService.CONFIG = new InterfaceVpcEndpointAwsService('config');
InterfaceVpcEndpointAwsService.EC2 = new InterfaceVpcEndpointAwsService('ec2');
InterfaceVpcEndpointAwsService.EC2_MESSAGES = new InterfaceVpcEndpointAwsService('ec2messages');
InterfaceVpcEndpointAwsService.ECR = new InterfaceVpcEndpointAwsService('ecr.api');
InterfaceVpcEndpointAwsService.ECR_DOCKER = new InterfaceVpcEndpointAwsService('ecr.dkr');
InterfaceVpcEndpointAwsService.ECS = new InterfaceVpcEndpointAwsService('ecs');
InterfaceVpcEndpointAwsService.ECS_AGENT = new InterfaceVpcEndpointAwsService('ecs-agent');
InterfaceVpcEndpointAwsService.ECS_TELEMETRY = new InterfaceVpcEndpointAwsService('ecs-telemetry');
InterfaceVpcEndpointAwsService.ELASTIC_INFERENCE_RUNTIME = new InterfaceVpcEndpointAwsService('elastic-inference.runtime');
InterfaceVpcEndpointAwsService.ELASTIC_LOAD_BALANCING = new InterfaceVpcEndpointAwsService('elasticloadbalancing');
InterfaceVpcEndpointAwsService.CLOUDWATCH_EVENTS = new InterfaceVpcEndpointAwsService('events');
InterfaceVpcEndpointAwsService.APIGATEWAY = new InterfaceVpcEndpointAwsService('execute-api');
InterfaceVpcEndpointAwsService.CODECOMMIT_GIT = new InterfaceVpcEndpointAwsService('git-codecommit');
InterfaceVpcEndpointAwsService.CODECOMMIT_GIT_FIPS = new InterfaceVpcEndpointAwsService('git-codecommit-fips');
InterfaceVpcEndpointAwsService.KINESIS_STREAMS = new InterfaceVpcEndpointAwsService('kinesis-streams');
InterfaceVpcEndpointAwsService.KMS = new InterfaceVpcEndpointAwsService('kms');
InterfaceVpcEndpointAwsService.CLOUDWATCH_LOGS = new InterfaceVpcEndpointAwsService('logs');
InterfaceVpcEndpointAwsService.CLOUDWATCH = new InterfaceVpcEndpointAwsService('monitoring');
InterfaceVpcEndpointAwsService.SAGEMAKER_API = new InterfaceVpcEndpointAwsService('sagemaker.api');
InterfaceVpcEndpointAwsService.SAGEMAKER_RUNTIME = new InterfaceVpcEndpointAwsService('sagemaker.runtime');
InterfaceVpcEndpointAwsService.SAGEMAKER_RUNTIME_FIPS = new InterfaceVpcEndpointAwsService('sagemaker.runtime-fips');
InterfaceVpcEndpointAwsService.SECRETS_MANAGER = new InterfaceVpcEndpointAwsService('secretsmanager');
InterfaceVpcEndpointAwsService.SERVICE_CATALOG = new InterfaceVpcEndpointAwsService('servicecatalog');
InterfaceVpcEndpointAwsService.SNS = new InterfaceVpcEndpointAwsService('sns');
InterfaceVpcEndpointAwsService.SQS = new InterfaceVpcEndpointAwsService('sqs');
InterfaceVpcEndpointAwsService.SSM = new InterfaceVpcEndpointAwsService('ssm');
InterfaceVpcEndpointAwsService.SSM_MESSAGES = new InterfaceVpcEndpointAwsService('ssmmessages');
InterfaceVpcEndpointAwsService.STS = new InterfaceVpcEndpointAwsService('sts');
InterfaceVpcEndpointAwsService.TRANSFER = new InterfaceVpcEndpointAwsService('transfer.server');
/**
 * A interface VPC endpoint.
 * @resource AWS::EC2::VPCEndpoint
 */
class InterfaceVpcEndpoint extends VpcEndpoint {
    constructor(scope, id, props) {
        super(scope, id);
        const securityGroup = new security_group_1.SecurityGroup(this, 'SecurityGroup', {
            vpc: props.vpc
        });
        this.securityGroupId = securityGroup.securityGroupId;
        this.connections = new connections_1.Connections({
            defaultPort: port_1.Port.tcp(props.service.port),
            securityGroups: [securityGroup]
        });
        const subnets = props.vpc.selectSubnets(Object.assign(Object.assign({}, props.subnets), { onePerAz: true }));
        const subnetIds = subnets.subnetIds;
        const endpoint = new ec2_generated_1.CfnVPCEndpoint(this, 'Resource', {
            privateDnsEnabled: props.privateDnsEnabled !== undefined ? props.privateDnsEnabled : true,
            policyDocument: core_1.Lazy.anyValue({ produce: () => this.policyDocument }),
            securityGroupIds: [this.securityGroupId],
            serviceName: props.service.name,
            vpcEndpointType: VpcEndpointType.INTERFACE,
            subnetIds,
            vpcId: props.vpc.vpcId
        });
        this.vpcEndpointId = endpoint.ref;
        this.vpcEndpointCreationTimestamp = endpoint.attrCreationTimestamp;
        this.vpcEndpointDnsEntries = endpoint.attrDnsEntries;
        this.vpcEndpointNetworkInterfaceIds = endpoint.attrNetworkInterfaceIds;
    }
    /**
     * Imports an existing interface VPC endpoint.
     */
    static fromInterfaceVpcEndpointAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.vpcEndpointId = attrs.vpcEndpointId;
                this.securityGroupId = attrs.securityGroupId;
                this.connections = new connections_1.Connections({
                    defaultPort: port_1.Port.tcp(attrs.port),
                    securityGroups: [security_group_1.SecurityGroup.fromSecurityGroupId(this, 'SecurityGroup', attrs.securityGroupId)],
                });
            }
        }
        return new Import(scope, id);
    }
}
exports.InterfaceVpcEndpoint = InterfaceVpcEndpoint;
//# sourceMappingURL=data:application/json;base64,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