"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const ec2_generated_1 = require("./ec2.generated");
/**
 * A NetworkAclBase that is not created in this template
 *
 * @experimental
 */
class NetworkAclBase extends core_1.Resource {
    /**
     * Add a new entry to the ACL
     */
    addEntry(id, options) {
        return new NetworkAclEntry(this, id, Object.assign({ networkAcl: this }, options));
    }
}
/**
 * Define a new custom network ACL
 *
 * By default, will deny all inbound and outbound traffic unless entries are
 * added explicitly allowing it.
 *
 * @experimental
 */
class NetworkAcl extends NetworkAclBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.networkAclName
        });
        this.vpc = props.vpc;
        this.networkAcl = new ec2_generated_1.CfnNetworkAcl(this, 'Resource', {
            vpcId: props.vpc.vpcId,
        });
        this.networkAclId = this.networkAcl.ref;
        this.networkAclVpcId = this.networkAcl.vpcId;
        if (props.subnetSelection !== undefined) {
            this.associateWithSubnet('DefaultAssociation', props.subnetSelection);
        }
    }
    /**
     * Import an existing NetworkAcl into this app.
     */
    static fromNetworkAclId(scope, id, networkAclId) {
        class Import extends NetworkAclBase {
            constructor() {
                super(...arguments);
                this.networkAclId = networkAclId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Associate the ACL with a given set of subnets
     */
    associateWithSubnet(id, selection) {
        const subnets = this.vpc.selectSubnets(selection);
        for (const subnet of subnets.subnets) {
            subnet.associateNetworkAcl(id, this);
        }
    }
}
exports.NetworkAcl = NetworkAcl;
/**
 * What action to apply to traffic matching the ACL
 *
 * @experimental
 */
var Action;
(function (Action) {
    /**
     * Allow the traffic
     */
    Action["ALLOW"] = "allow";
    /**
     * Deny the traffic
     */
    Action["DENY"] = "deny";
})(Action = exports.Action || (exports.Action = {}));
/**
 * Base class for NetworkAclEntries
 *
 * @experimental
 */
class NetworkAclEntryBase extends core_1.Resource {
}
/**
 * Direction of traffic the AclEntry applies to
 *
 * @experimental
 */
var TrafficDirection;
(function (TrafficDirection) {
    /**
     * Traffic leaving the subnet
     */
    TrafficDirection[TrafficDirection["EGRESS"] = 0] = "EGRESS";
    /**
     * Traffic entering the subnet
     */
    TrafficDirection[TrafficDirection["INGRESS"] = 1] = "INGRESS";
})(TrafficDirection = exports.TrafficDirection || (exports.TrafficDirection = {}));
/**
 * Define an entry in a Network ACL table
 *
 * @experimental
 */
class NetworkAclEntry extends NetworkAclEntryBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.networkAclEntryName
        });
        this.networkAcl = props.networkAcl;
        new ec2_generated_1.CfnNetworkAclEntry(this, 'Resource', Object.assign(Object.assign({ networkAclId: this.networkAcl.networkAclId, ruleNumber: props.ruleNumber, ruleAction: props.ruleAction !== undefined ? props.ruleAction : Action.ALLOW, egress: props.direction !== undefined ? props.direction === TrafficDirection.EGRESS : undefined }, props.traffic.toTrafficConfig()), props.cidr.toCidrConfig()));
    }
}
exports.NetworkAclEntry = NetworkAclEntry;
/**
 * Associate a network ACL with a subnet
 *
 * @experimental
 */
class SubnetNetworkAclAssociationBase extends core_1.Resource {
}
class SubnetNetworkAclAssociation extends SubnetNetworkAclAssociationBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.subnetNetworkAclAssociationName
        });
        this.association = new ec2_generated_1.CfnSubnetNetworkAclAssociation(this, 'Resource', {
            networkAclId: props.networkAcl.networkAclId,
            subnetId: props.subnet.subnetId
        });
        this.networkAcl = props.networkAcl;
        this.subnet = props.subnet;
        this.subnetNetworkAclAssociationAssociationId = this.association.attrAssociationId;
    }
    static fromSubnetNetworkAclAssociationAssociationId(scope, id, subnetNetworkAclAssociationAssociationId) {
        class Import extends SubnetNetworkAclAssociationBase {
            constructor() {
                super(...arguments);
                this.subnetNetworkAclAssociationAssociationId = subnetNetworkAclAssociationAssociationId;
            }
        }
        return new Import(scope, id);
    }
}
exports.SubnetNetworkAclAssociation = SubnetNetworkAclAssociation;
//# sourceMappingURL=data:application/json;base64,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