"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ssm = require("@aws-cdk/aws-ssm");
const core_1 = require("@aws-cdk/core");
/**
 * Select the latest version of the indicated Windows version
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 *
 * https://aws.amazon.com/blogs/mt/query-for-the-latest-windows-ami-using-systems-manager-parameter-store/
 */
class WindowsImage {
    constructor(version, props = {}) {
        this.version = version;
        this.props = props;
    }
    /**
     * Return the image to use in the given context
     */
    getImage(scope) {
        const parameterName = this.imageParameterName();
        const ami = ssm.StringParameter.valueForTypedStringParameter(scope, parameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
        return {
            imageId: ami,
            userData: this.props.userData,
            osType: OperatingSystemType.WINDOWS,
        };
    }
    /**
     * Construct the SSM parameter name for the given Windows image
     */
    imageParameterName() {
        return '/aws/service/ami-windows-latest/' + this.version;
    }
}
exports.WindowsImage = WindowsImage;
/**
 * Selects the latest version of Amazon Linux
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 */
class AmazonLinuxImage {
    constructor(props = {}) {
        this.props = props;
        this.generation = (props && props.generation) || AmazonLinuxGeneration.AMAZON_LINUX;
        this.edition = (props && props.edition) || AmazonLinuxEdition.STANDARD;
        this.virtualization = (props && props.virtualization) || AmazonLinuxVirt.HVM;
        this.storage = (props && props.storage) || AmazonLinuxStorage.GENERAL_PURPOSE;
    }
    /**
     * Return the image to use in the given context
     */
    getImage(scope) {
        const parts = [
            this.generation,
            'ami',
            this.edition !== AmazonLinuxEdition.STANDARD ? this.edition : undefined,
            this.virtualization,
            'x86_64',
            this.storage
        ].filter(x => x !== undefined); // Get rid of undefineds
        const parameterName = '/aws/service/ami-amazon-linux-latest/' + parts.join('-');
        const ami = ssm.StringParameter.valueForTypedStringParameter(scope, parameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
        return {
            imageId: ami,
            userData: this.props.userData,
            osType: OperatingSystemType.LINUX,
        };
    }
}
exports.AmazonLinuxImage = AmazonLinuxImage;
/**
 * What generation of Amazon Linux to use
 */
var AmazonLinuxGeneration;
(function (AmazonLinuxGeneration) {
    /**
     * Amazon Linux
     */
    AmazonLinuxGeneration["AMAZON_LINUX"] = "amzn";
    /**
     * Amazon Linux 2
     */
    AmazonLinuxGeneration["AMAZON_LINUX_2"] = "amzn2";
})(AmazonLinuxGeneration = exports.AmazonLinuxGeneration || (exports.AmazonLinuxGeneration = {}));
/**
 * Amazon Linux edition
 */
var AmazonLinuxEdition;
(function (AmazonLinuxEdition) {
    /**
     * Standard edition
     */
    AmazonLinuxEdition["STANDARD"] = "standard";
    /**
     * Minimal edition
     */
    AmazonLinuxEdition["MINIMAL"] = "minimal";
})(AmazonLinuxEdition = exports.AmazonLinuxEdition || (exports.AmazonLinuxEdition = {}));
/**
 * Virtualization type for Amazon Linux
 */
var AmazonLinuxVirt;
(function (AmazonLinuxVirt) {
    /**
     * HVM virtualization (recommended)
     */
    AmazonLinuxVirt["HVM"] = "hvm";
    /**
     * PV virtualization
     */
    AmazonLinuxVirt["PV"] = "pv";
})(AmazonLinuxVirt = exports.AmazonLinuxVirt || (exports.AmazonLinuxVirt = {}));
var AmazonLinuxStorage;
(function (AmazonLinuxStorage) {
    /**
     * EBS-backed storage
     */
    AmazonLinuxStorage["EBS"] = "ebs";
    /**
     * S3-backed storage
     */
    AmazonLinuxStorage["S3"] = "ebs";
    /**
     * General Purpose-based storage (recommended)
     */
    AmazonLinuxStorage["GENERAL_PURPOSE"] = "gp2";
})(AmazonLinuxStorage = exports.AmazonLinuxStorage || (exports.AmazonLinuxStorage = {}));
/**
 * Construct a Linux machine image from an AMI map
 *
 * Linux images IDs are not published to SSM parameter store yet, so you'll have to
 * manually specify an AMI map.
 */
class GenericLinuxImage {
    constructor(amiMap, props = {}) {
        this.amiMap = amiMap;
        this.props = props;
    }
    getImage(scope) {
        const region = core_1.Stack.of(scope).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error(`Unable to determine AMI from AMI map since stack is region-agnostic`);
        }
        const ami = region !== 'test-region' ? this.amiMap[region] : 'ami-12345';
        if (!ami) {
            throw new Error(`Unable to find AMI in AMI map: no AMI specified for region '${region}'`);
        }
        return {
            imageId: ami,
            userData: this.props.userData,
            osType: OperatingSystemType.LINUX,
        };
    }
}
exports.GenericLinuxImage = GenericLinuxImage;
/**
 * Construct a Windows machine image from an AMI map
 *
 * Allows you to create a generic Windows EC2 , manually specify an AMI map.
 */
class GenericWindowsImage {
    constructor(amiMap, props = {}) {
        this.amiMap = amiMap;
        this.props = props;
    }
    getImage(scope) {
        const region = core_1.Stack.of(scope).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error(`Unable to determine AMI from AMI map since stack is region-agnostic`);
        }
        const ami = region !== 'test-region' ? this.amiMap[region] : 'ami-12345';
        if (!ami) {
            throw new Error(`Unable to find AMI in AMI map: no AMI specified for region '${region}'`);
        }
        return {
            imageId: ami,
            userData: this.props.userData,
            osType: OperatingSystemType.WINDOWS,
        };
    }
}
exports.GenericWindowsImage = GenericWindowsImage;
/**
 * The OS type of a particular image
 */
var OperatingSystemType;
(function (OperatingSystemType) {
    OperatingSystemType[OperatingSystemType["LINUX"] = 0] = "LINUX";
    OperatingSystemType[OperatingSystemType["WINDOWS"] = 1] = "WINDOWS";
})(OperatingSystemType = exports.OperatingSystemType || (exports.OperatingSystemType = {}));
//# sourceMappingURL=data:application/json;base64,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