'''
Test the generation of uuid columns
'''

# Standard library imports
import unittest

# External library imports
import numpy as np
import pandas as pd

# Module under test
from exhibit.core.generate import uuids as tm

class uuidTests(unittest.TestCase):
    '''
    Testing only the generation of uuid columns. The formatting of uuids in spec
    and parsing of uuid-related CLI options are testing in the respective modules.
    '''

    def test_basic_uuid_column_generation(self):
        '''
        Generated UUIDs should match the hard-coded values for reproducibility
        '''

        freq_dist = [
            "frequency | probability_vector",
            "1         | 0.800",
            "2         | 0.200"
        ]
        
        result = tm.generate_uuid_column("id", 10, 0, freq_dist, 0)
        expected = pd.Series(
            data=[
            "eb1167b367a94378bc65c1e582e2e662",
            "b4862b21fb97443588561712e8e5216a",
            "e443df789558467f9ba91faf7a024204",
            "b4862b21fb97443588561712e8e5216a",
            "f728b4fa42484e3a8a5d2f346baa9455",
            "1846d424c17c4279a3c6612f48268673",
            "fcbd04c340214ef78ca5a5a19e4d6e3c",
            "f7c1bd874da5470994713d60c8a70639",
            "e3e70682c2094caca29f6fbed82c07cd",
            "23a7711a81334876b7ebdcd9e87a1613",
            ],
            name="id"
        )

        pd.testing.assert_series_equal(result, expected)

    def test_uuid_column_generation_with_missing_data(self):
        '''
        Adding missing probabilities will disrupt the frequencies
        '''

        freq_dist = [
            "frequency | probability_vector",
            "1         | 0.800",
            "2         | 0.200"
        ]
        
        result = tm.generate_uuid_column("id", 10, 0.5, freq_dist, 0)

        self.assertTrue(result.isna().any())

    def test_uuid_column_generation_with_more_uuids_than_requested(self):
        '''
        Due to rounding, there might be small differences in the number of rows
        generated by uuid function based on probabilties and frequencies.
        Also, probabilities must add up to 1 in the code.
        '''

        freq_dist = [
            "frequency | probability_vector",
            "1         | 0.820",
            "2         | 0.170",
        ]
        
        result = tm.generate_uuid_column("id", 100, 0, freq_dist, 0)

        self.assertEqual(len(result), 100)

if __name__ == "__main__" and __package__ is None:
    #overwrite __package__ builtin as per PEP 366
    __package__ = "exhibit"
    unittest.main(warnings="ignore")
