/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { Construct } from '@aws-cdk/core';
import { IPatchVersion, IVersion, PlatformInstallers } from './version-ref';
/**
 * Properties for the Deadline Version.
 */
export interface VersionQueryProps {
    /**
     * String containing the complete or partial deadline version.
     *
     * @default - the latest available version of deadline installer.
     */
    readonly version?: string;
}
/**
 * The abstract class for new or imported(custom) Deadline Version.
 */
declare abstract class VersionQueryBase extends Construct implements IVersion {
    /**
     *
     */
    protected static readonly INSTALLER_BUCKET = "thinkbox-installers";
    /**
     * @inheritdoc
     */
    abstract readonly majorVersion: number;
    /**
     * @inheritdoc
     */
    abstract readonly minorVersion: number;
    /**
     * @inheritdoc
     */
    abstract readonly releaseVersion: number;
    /**
     * @inheritdoc
     */
    abstract readonly linuxInstallers?: PlatformInstallers;
    /**
     * Construct the full version string for the linux patch release referenced in this version object.
     *
     * This is constructed by joining the major, minor,
     * release, and patch versions by dots.
     *
     * Will return undefined if the linuxInstallers property is undefined.
     *
     * @inheritdoc true
     */
    linuxFullVersionString(): string | undefined;
    /**
     *
     */
    protected fullVersionString(patchVersion: number): string;
    /**
     * Helper to concatenate the major, minor, and release version values into a release version string.
     */
    protected get releaseVersionString(): string;
}
/**
 * This class is reposonsible to do basic operations on version format.
 */
export declare class Version implements IPatchVersion {
    /**
     * This variable holds the value for minimum supported deadline version.
     */
    static readonly MINIMUM_SUPPORTED_DEADLINE_VERSION: Version;
    /**
     * This method parses the input string and returns the version object.
     *
     * @param version version string to parse.
     */
    static parse(version: string): Version;
    /**
     * This method validates the given string for a sequence '.' separated numbers.
     *
     * @param version the string to be validated.
     *
     * @returns true if the format is correct, else false.
     */
    private static validateVersionFormat;
    /**
     * Numeric components of version.
     */
    private readonly components;
    /**
     * The major version number.
     *
     * @inheritdoc true
     */
    get majorVersion(): number;
    /**
     * The minor version number.
     *
     * @inheritdoc true
     */
    get minorVersion(): number;
    /**
     * The release version number.
     *
     * @inheritdoc true
     */
    get releaseVersion(): number;
    /**
     * The patch version number.
     *
     * @inheritdoc true
     */
    get patchVersion(): number;
    /**
     *
     */
    constructor(components: number[]);
    /**
     * This method compares two version strings.
     *
     * @returns true if this version is greater than the provided version;
     * false if this version is less than or equal to the provided verison.
     */
    isGreaterThan(version: Version): boolean;
    /**
     * This method compares two version strings.
     *
     * @returns true if this version is less than the provided version;
     * false if this version is greater than or equal to the provided verison.
     */
    isLessThan(version: Version): boolean;
    /**
     * This method compares two version strings.
     *
     * @returns true if this version is equal to the provided version;
     * false otherwise.
     */
    isEqual(version: Version): boolean;
    /**
     * The method returns the version components in dot separated string format.
     */
    toString(): string;
    /**
     * This method compares 2 versions.
     *
     * @param version version to compare
     *
     * @returns negative value if this version is less than the provided version;
     * 0 if both the versions are equal;
     * positive value if this version is greater than the provided verison.
     */
    private compare;
}
/**
 * This class encapsulates information about a particular version of Thinkbox's Deadline software.
 *
 * Information such as the version number, and where to get installers for that version from Amazon S3.
 *
 * The version of an official release of Deadline is always four numeric version components separated by dots.
 * ex: 10.1.8.5. We refer to the components in this version, in order from left-to-right, as the
 * major, minor, release, and patch versions. For example, Deadline version 10.1.8.5 is majorVersion 10, minorVersion 1,
 * releaseVersion 8, and patchVersion 5.
 *
 * All of the installers provided by an instance of this class must be for the same Deadline release (ex: 10.1.8),
 * but the patch versions may differ between operating systems depending on the particulars of that release of Deadline.
 * This class provides a simple way to query a version of Deadline prior to or during deployment of a
 * CDK app.
 *
 * You pass an instance of this class to various Deadline constructs in this library to tell those
 * constructs which version of Deadline you want them to use, and be configured for.
 */
export declare class VersionQuery extends VersionQueryBase {
    /**
     * Specify a Deadline version from a fully-qualified Deadline patch version.
     *
     * This only provides the Linux repository installer based on its conventional
     * S3 object path.
     *
     * @param scope The parent scope.
     * @param id The construct ID.
     * @param versionComponents The individual components of the Deadline release version.
     * @remark Thinkbox reserves the right to revoke patch versions of Deadline and suceeed them with a new patch version.
     * For this reason, using this method may fail if Thinkbox revokes the specific patch version of the Deadline
     * installer in the event of a critical issue such as a security vulnerability.
     *
     * Use at your own risk.
     */
    static exact(scope: Construct, id: string, versionComponents: IPatchVersion): IVersion;
    /**
     * Specify Version from a fully-qualified Deadline release version string.
     *
     * This only provides the Linux repository installer based on its conventional
     * S3 object path.
     *
     * @param scope The parent scope.
     * @param id The construct ID.
     * @param versionString A fully qualified version string (e.g. 10.1.9.2).
     */
    static exactString(scope: Construct, id: string, versionString: string): IVersion;
    /**
     * The major version number.
     *
     * @inheritdoc true
     */
    readonly majorVersion: number;
    /**
     * The minor version number.
     *
     * @inheritdoc true
     */
    readonly minorVersion: number;
    /**
     * The release version number.
     *
     * @inheritdoc true
     */
    readonly releaseVersion: number;
    /**
     * The Linux installers for this version.
     *
     * @inheritdoc true
     */
    readonly linuxInstallers?: PlatformInstallers;
    /**
     *
     */
    constructor(scope: Construct, id: string, props?: VersionQueryProps);
}
export {};
