"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ThinkboxDockerRecipes = void 0;
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const core_1 = require("@aws-cdk/core");
/**
 * An enum that is associated with AWS Thinkbox managed recipes that are available in the stage manifest.
 */
var ThinkboxManagedDeadlineDockerRecipes;
(function (ThinkboxManagedDeadlineDockerRecipes) {
    /**
     * The Docker Image Asset for the Remote Connection Server.
     */
    ThinkboxManagedDeadlineDockerRecipes["RemoteConnectionServer"] = "rcs";
    /**
     * The Docker Image Asset for the License Forwarder.
     */
    ThinkboxManagedDeadlineDockerRecipes["LicenseForwarder"] = "license-forwarder";
})(ThinkboxManagedDeadlineDockerRecipes || (ThinkboxManagedDeadlineDockerRecipes = {}));
/**
 * An API for interacting with staged Deadline Docker recipes provided by AWS Thinkbox.
 *
 * This provides container images as required by RFDK's Deadline constructs such as
 *
 * * {@link @aws-rfdk/deadline#RenderQueue}
 * * {@link @aws-rfdk/deadline#UsageBasedLicensing}
 *
 * @example
 *
 * Construct a RenderQueue
 *
 * import { App, Stack, Vpc } from '@aws-rfdk/core';
 * import { RenderQueue, Repository, ThinkboxDockerRecipes } from '@aws-rfdk/deadline';
 * const app = new App();
 * const stack = new Stack(app, 'Stack');
 * const vpc = new Vpc(app, stack);
 * const recipes = new ThinkboxDockerRecipes(stack, 'Recipes', {
 * path: '/path/to/staged/recipes'
 * });
 * const repository = new Repository(stack, 'Repository', {
 * vpc,
 * version: recipes.version
 * });
 *
 * const renderQueue = new RenderQueue(stack, 'RenderQueue', {
 * images: recipes.renderQueueImages,
 * // ...
 * });
 */
class ThinkboxDockerRecipes extends core_1.Construct {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.version = props.stage.getVersion(this, 'Version');
        for (const recipe of [ThinkboxManagedDeadlineDockerRecipes.RemoteConnectionServer, ThinkboxManagedDeadlineDockerRecipes.LicenseForwarder]) {
            if (!props.stage.manifest.recipes[recipe]) {
                throw new Error(`Could not find ${recipe} recipe`);
            }
        }
        this.remoteConnectionServer = props.stage.imageFromRecipe(this, 'RemoteConnectionServer', ThinkboxManagedDeadlineDockerRecipes.RemoteConnectionServer.toString());
        this.licenseForwarder = props.stage.imageFromRecipe(this, 'LicenseForwarder', ThinkboxManagedDeadlineDockerRecipes.LicenseForwarder.toString());
        this.renderQueueImages = {
            remoteConnectionServer: aws_ecs_1.ContainerImage.fromDockerImageAsset(this.remoteConnectionServer),
        };
        this.ublImages = {
            licenseForwarder: aws_ecs_1.ContainerImage.fromDockerImageAsset(this.licenseForwarder),
        };
    }
}
exports.ThinkboxDockerRecipes = ThinkboxDockerRecipes;
//# sourceMappingURL=data:application/json;base64,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