"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.RenderQueueConnection = void 0;
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
/**
 * Helper class for connecting Clients to a render queue
 */
class RenderQueueConnection {
    /**
     * Create a RQ Connection for http traffic
     */
    static forHttp(options) {
        return new HTTPConnection(options);
    }
    /**
     * Create a RQ Connection for https traffic
     */
    static forHttps(options) {
        return new HTTPSConnection(options);
    }
    /**
     * Fetch the instance configuration python script
     * @param stack The stack that the asset should be created in
     */
    connectionAssetSingleton(stack) {
        var _a;
        const uuid = '3be2203f-fea1-4143-bc09-7ef079b4299c';
        const uniqueId = 'RenderQueueConnectionAsset' + uuid.replace(/[-]/g, '');
        return (_a = stack.node.tryFindChild(uniqueId)) !== null && _a !== void 0 ? _a : new aws_s3_assets_1.Asset(stack, uniqueId, {
            path: path.join(__dirname, '..', 'scripts', 'python', 'client-rq-connection.py'),
        });
    }
    /**
     * Executes connection asset singleton wth a given list of arguments
     */
    ExecuteConnectionAsset(host, args) {
        const hostStack = core_1.Stack.of(host);
        const connectionAsset = this.connectionAssetSingleton(hostStack);
        connectionAsset.grantRead(host);
        const configureScriptPath = host.userData.addS3DownloadCommand({
            bucket: connectionAsset.bucket,
            bucketKey: connectionAsset.s3ObjectKey,
        });
        const dlExtraCommands = [];
        if (args.tlsCaArn) {
            dlExtraCommands.push('--tls-ca', `"${args.tlsCaArn}"`);
        }
        if (host.osType === aws_ec2_1.OperatingSystemType.LINUX) {
            host.userData.addCommands('if [ -f "/etc/profile.d/deadlineclient.sh" ]; then', '  source "/etc/profile.d/deadlineclient.sh"', 'fi', `"\${DEADLINE_PATH}/deadlinecommand" -executeScriptNoGui "${configureScriptPath}" --render-queue "${args.address}" ${dlExtraCommands.join(' ')}`, 
            // Cleanup
            `rm -f "${configureScriptPath}"`, 'if service --status-all | grep -q "Deadline 10 Launcher"; then', '  service deadline10launcher restart', 'fi');
        }
        else if (host.osType === aws_ec2_1.OperatingSystemType.WINDOWS) {
            host.userData.addCommands('$ErrorActionPreference = "Stop"', '$DEADLINE_PATH = (get-item env:"DEADLINE_PATH").Value', `& "$DEADLINE_PATH/deadlinecommand.exe" -executeScriptNoGui "${configureScriptPath}" --render-queue "${args.address}" ${dlExtraCommands.join(' ')} 2>&1`, `Remove-Item -Path "${configureScriptPath}"`, 'If (Get-Service "deadline10launcherservice" -ErrorAction SilentlyContinue) {', '  Restart-Service "deadline10launcherservice"', '} Else {', '  & "$DEADLINE_PATH/deadlinelauncher.exe" -shutdownall 2>&1', '  & "$DEADLINE_PATH/deadlinelauncher.exe" 2>&1', '}');
        }
    }
}
exports.RenderQueueConnection = RenderQueueConnection;
/**
 * Specialization of {@link RenderQueueConnection} for HTTP Connections
 */
class HTTPConnection extends RenderQueueConnection {
    constructor(config) {
        super();
        this.config = config;
    }
    configureClientECS(params) {
        params.hosts.forEach(host => {
            host.connections.allowToDefaultPort(this.config.endpoint);
        });
        return {
            RENDER_QUEUE_URI: `http://${this.config.endpoint.socketAddress}`,
        };
    }
    configureClientInstance(params) {
        params.host.connections.allowToDefaultPort(this.config.endpoint);
        this.ExecuteConnectionAsset(params.host, {
            address: `http://${this.config.endpoint.socketAddress}`,
        });
    }
}
/**
 * Specialization of {@link RenderQueueConnection} for HTTPS Connections
 */
class HTTPSConnection extends RenderQueueConnection {
    constructor(config) {
        super();
        this.config = config;
    }
    configureClientECS(params) {
        params.hosts.forEach(host => {
            host.connections.allowToDefaultPort(this.config.endpoint);
        });
        this.config.caCert.grantRead(params.grantee);
        return {
            RENDER_QUEUE_URI: `https://${this.config.endpoint.socketAddress}`,
            RENDER_QUEUE_TLS_CA_CERT_URI: this.config.caCert.secretArn,
        };
    }
    configureClientInstance(params) {
        params.host.connections.allowToDefaultPort(this.config.endpoint);
        this.config.caCert.grantRead(params.host);
        this.ExecuteConnectionAsset(params.host, {
            address: `https://${this.config.endpoint.socketAddress}`,
            tlsCaArn: this.config.caCert.secretArn,
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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