/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { IDatabaseCluster } from '@aws-cdk/aws-docdb';
import { IConnectable } from '@aws-cdk/aws-ec2';
import { IGrantable } from '@aws-cdk/aws-iam';
import { ISecret } from '@aws-cdk/aws-secretsmanager';
import { IConstruct } from '@aws-cdk/core';
import { IMongoDb, IX509CertificatePkcs12 } from '../../core';
import { IHost } from './host-ref';
/**
 * Options when constructing UserData for Linux.
 */
export interface DocDBConnectionOptions {
    /**
     * The Document DB Cluster this connection is for.
     */
    readonly database: IDatabaseCluster;
    /**
     * A Secret that contains the login information for the database.
     *
     * This must be a secret containing a JSON document as follows:
     *      {
     *          "username": "<username>",
     *          "password": "<password for username>"
     *      }
     */
    readonly login: ISecret;
}
/**
 *
 */
export interface MongoDbInstanceConnectionOptions {
    /**
     * The MongoDB database to connect to.
     */
    readonly database: IMongoDb;
    /**
     * The client certificate to register in the database during install of the Deadline Repository, and for the Deadline Client to use to connect to the database.
     *
     * This **MUST** be signed by the same signing certificate as the MongoDB server's certificate.
     *
     * Note: A limitation of Deadline **requires** that this certificate be signed directly by your root certificate authority (CA).
     * Deadline will be unable to connect to MongoDB if it has been signed by an intermediate CA.
     */
    readonly clientCertificate: IX509CertificatePkcs12;
}
/**
 * Helper class for connecting Thinkbox's Deadline to a specific Database.
 */
export declare abstract class DatabaseConnection {
    /**
     * Creates a DatabaseConnection which allows Deadline to connect to Amazon DocumentDB.
     *
     * Resources Deployed
     * ------------------------
     * This construct does not deploy any resources
     */
    static forDocDB(options: DocDBConnectionOptions): DatabaseConnection;
    /**
     * Creates a DatabaseConnection which allows Deadline to connect to MongoDB.
     *
     * Resources Deployed
     * ------------------------
     * This construct does not deploy any resources
     */
    static forMongoDbInstance(options: MongoDbInstanceConnectionOptions): DatabaseConnection;
    /**
     * Returns the environment variables for configuring containers to connect to the database.
     */
    abstract readonly containerEnvironment: {
        [name: string]: string;
    };
    /**
     * Allow connections to the Database from the given connection peer.
     */
    abstract allowConnectionsFrom(other: IConnectable): void;
    /**
     * Adds commands to a UserData to build the argument list needed to install the Deadline Repository.
     *
     * The implementation must export a shell function called configure_database_installation_args(),
     * that takes no arguments. This function must define an array environment variable called
     * INSTALLER_DB_ARGS where each element of the array is a key-value pair of Deadline installer
     * option to option value. (ex: ["--dbuser"]=someusername).
     *
     * This implementation avoids secrets being leaked to the cloud-init logs.
     */
    abstract addInstallerDBArgs(host: IHost): void;
    /**
     * Adds commands to an Instance or Autoscaling groups User Data to configure the Deadline client so it can access the DB  Implementation must add commands to the instance userData that exports a function called configure_deadline_database() that accepts no arguments, and does what ever deadline-specific setup is required to allow Deadline to connect to the database.
     *
     * This implementation avoids secrets being leaked to the cloud-init logs.
     */
    abstract addConnectionDBArgs(host: IHost): void;
    /**
     * Grants permissions to the principal that allow it to use the Database as a typical Deadline user.
     */
    abstract grantRead(grantee: IGrantable): void;
    /**
     * Add an ordering dependency to another Construct.
     *
     * All constructs in the child's scope will be deployed after the database has been deployed.
     *
     * This can be used to ensure that the database is fully up and serving data before an instance attempts to connect to it.
     *
     * @param child The child to make dependent upon this database.
     */
    abstract addChildDependency(child: IConstruct): void;
}
