"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_kms_1 = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const x509_certificate_1 = require("../lib/x509-certificate");
test('Generate cert', () => {
    const stack = new core_1.Stack(undefined, 'Stack', { env: { region: 'us-west-2' } });
    const subject = { cn: 'testCN' };
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', {
        subject,
    });
    const certPassphraseID = stack.getLogicalId(cert.passphrase.node.defaultChild);
    // Expect the custom resource for cert generation
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_X509Generator', {
        DistinguishedName: {
            CN: 'testCN',
            O: 'AWS',
            OU: 'Thinkbox',
        },
    }));
    // Expect the resource for converting to PKCS #12 not to be created
    assert_1.expect(stack).notTo(assert_1.haveResource('Custom::RFDK_X509_PKCS12'));
    // Expect the DynamoDB table used for custom resource tracking
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table'));
    // Expect a Secret used to store the cert
    assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::Secret'));
    // Expect a policy that can interact with DynamoDB and SecretsManager
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'dynamodb:BatchGetItem',
                        'dynamodb:GetRecords',
                        'dynamodb:GetShardIterator',
                        'dynamodb:Query',
                        'dynamodb:GetItem',
                        'dynamodb:Scan',
                        'dynamodb:BatchWriteItem',
                        'dynamodb:PutItem',
                        'dynamodb:UpdateItem',
                        'dynamodb:DeleteItem',
                    ],
                },
                {
                    Action: 'dynamodb:DescribeTable',
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        Ref: certPassphraseID,
                    },
                },
                {
                    Action: [
                        'secretsmanager:CreateSecret',
                        'secretsmanager:DeleteSecret',
                        'secretsmanager:TagResource',
                        'secretsmanager:PutSecretValue',
                    ],
                    Condition: {
                        StringEquals: {
                            'secretsmanager:ResourceTag/X509SecretGrant-F53F5427': 'f53f5427b2e9eb4739661fcc0b249b6e',
                        },
                    },
                },
            ],
        },
    }));
    // Expect no KMS key for encryption
    assert_1.expect(stack).notTo(assert_1.haveResource('AWS::KMS::Key'));
    // Expect Lambda for doing the cert generation to use the generate() handler and openssl layer
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', (props, error) => {
        var _a, _b;
        if (!props.Handler || props.Handler !== 'x509-certificate.generate') {
            error.failureReason = 'x509-certificate.generate handler not found';
            error.resource = props.Handler;
            return false;
        }
        // Our test for the correct openssl lambda layer does not include the version, so we use a filter
        // function to do a partial match
        const filterOpensslArn = (value) => {
            return value.toString().includes('arn:aws:lambda:us-west-2:224375009292:layer:openssl-al2:');
        };
        if (!props.Layers
            || !Array.isArray(props.Layers)
            || Array.of(props.Layers).filter(filterOpensslArn).length === 0) {
            error.failureReason = 'openssl Lambda Layer missing';
            error.resource = props.Layers;
            return false;
        }
        if (!props.Environment
            || !props.Environment.Variables
            || !props.Environment.Variables.DATABASE) {
            error.failureReason = 'DATABASE environment variable not set';
            error.resource = (_b = (_a = props.Environment) === null || _a === void 0 ? void 0 : _a.Variables) === null || _b === void 0 ? void 0 : _b.DATABASE;
            return false;
        }
        return true;
    }));
});
test('Generate cert, all options set', () => {
    const stack = new core_1.Stack();
    const subject = {
        cn: 'testCN',
        o: 'testO',
        ou: 'testOu',
    };
    const encryptionKey = new aws_kms_1.Key(stack, 'Key');
    const signingCertificate = new x509_certificate_1.X509CertificatePem(stack, 'SigningCert', { subject });
    const signingCertPassphraseID = stack.getLogicalId(signingCertificate.passphrase.node.defaultChild);
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', {
        subject,
        encryptionKey,
        signingCertificate,
    });
    const certPassphraseID = stack.getLogicalId(cert.passphrase.node.defaultChild);
    // Expect the custom resource for cert generation
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_X509Generator', {
        DistinguishedName: {
            CN: 'testCN',
            O: 'testO',
            OU: 'testOu',
        },
        SigningCertificate: {
            Cert: {
                'Fn::GetAtt': [
                    'SigningCert',
                    'Cert',
                ],
            },
            Key: {
                'Fn::GetAtt': [
                    'SigningCert',
                    'Key',
                ],
            },
            Passphrase: {
                Ref: signingCertPassphraseID,
            },
            CertChain: '',
        },
    }));
    // Expect the resource for converting to PKCS #12 not to be created
    assert_1.expect(stack).notTo(assert_1.haveResource('Custom::RFDK_X509_PKCS12'));
    // Expect the DynamoDB table used for custom resource tracking
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table'));
    // Expect a Secret used to store the cert
    assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::Secret'));
    // Expect a KMS key for encryption
    assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key'));
    // Expect a policy that can interact with DynamoDB and SecretsManager for the signing cert's custom resource
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'dynamodb:BatchGetItem',
                        'dynamodb:GetRecords',
                        'dynamodb:GetShardIterator',
                        'dynamodb:Query',
                        'dynamodb:GetItem',
                        'dynamodb:Scan',
                        'dynamodb:BatchWriteItem',
                        'dynamodb:PutItem',
                        'dynamodb:UpdateItem',
                        'dynamodb:DeleteItem',
                    ],
                },
                {
                    Action: 'dynamodb:DescribeTable',
                },
                {
                    Action: [
                        'kms:Encrypt',
                        'kms:ReEncrypt*',
                        'kms:GenerateDataKey*',
                    ],
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        Ref: certPassphraseID,
                    },
                },
                {
                    Action: [
                        'secretsmanager:CreateSecret',
                        'secretsmanager:DeleteSecret',
                        'secretsmanager:TagResource',
                        'secretsmanager:PutSecretValue',
                    ],
                    Condition: {
                        StringEquals: {
                            'secretsmanager:ResourceTag/X509SecretGrant-B2B09A60': 'b2b09a6086e87fe14005f4e0b800e4f0',
                        },
                    },
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        'Fn::GetAtt': [
                            'SigningCert',
                            'Cert',
                        ],
                    },
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        'Fn::GetAtt': [
                            'SigningCert',
                            'Key',
                        ],
                    },
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        Ref: 'SigningCertPassphrase42F0BC4F',
                    },
                },
            ],
        },
    }));
    // Expect a policy that can interact with DynamoDB and SecretsManager for the cert's custom resource
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'dynamodb:BatchGetItem',
                        'dynamodb:GetRecords',
                        'dynamodb:GetShardIterator',
                        'dynamodb:Query',
                        'dynamodb:GetItem',
                        'dynamodb:Scan',
                        'dynamodb:BatchWriteItem',
                        'dynamodb:PutItem',
                        'dynamodb:UpdateItem',
                        'dynamodb:DeleteItem',
                    ],
                },
                {
                    Action: 'dynamodb:DescribeTable',
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        Ref: signingCertPassphraseID,
                    },
                },
                {
                    Action: [
                        'secretsmanager:CreateSecret',
                        'secretsmanager:DeleteSecret',
                        'secretsmanager:TagResource',
                        'secretsmanager:PutSecretValue',
                    ],
                    Condition: {
                        StringEquals: {
                            'secretsmanager:ResourceTag/X509SecretGrant-BA0FA489': 'ba0fa4898b2088c5b25f15075f605300',
                        },
                    },
                },
            ],
        },
    }));
    // Expect Lambda for doing the cert generation to use the generate() handler
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
        Handler: 'x509-certificate.generate',
    }));
});
test('Grant cert read', () => {
    const stack = new core_1.Stack();
    const grantable = new aws_iam_1.Role(stack, 'TestRole', {
        assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
    });
    const subject = { cn: 'testCN' };
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', {
        subject,
    });
    const certPassphraseID = stack.getLogicalId(cert.passphrase.node.defaultChild);
    cert.grantCertRead(grantable);
    // Expect the custom resource to be created
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_X509Generator', {
        DistinguishedName: {
            CN: 'testCN',
            O: 'AWS',
            OU: 'Thinkbox',
        },
    }));
    // Expect the grantCertRead() to add this policy
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Cert',
                        ],
                    },
                },
            ],
        },
    }));
    // Expect the grantCertRead() not to add this full read policy
    assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Cert',
                        ],
                    },
                },
                {
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Key',
                        ],
                    },
                },
                {
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        Ref: certPassphraseID,
                    },
                },
            ],
        },
    }));
    // Expect the PKCS #12 generator not to be created
    assert_1.expect(stack).notTo(assert_1.haveResource('Custom::RFDK_X509_PKCS12'));
});
test('Grant full read', () => {
    const stack = new core_1.Stack();
    const grantable = new aws_iam_1.Role(stack, 'TestRole', {
        assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
    });
    const subject = { cn: 'testCN' };
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', {
        subject,
    });
    const certPassphraseID = stack.getLogicalId(cert.passphrase.node.defaultChild);
    cert.grantFullRead(grantable);
    // Expect the custom resource to be created
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_X509Generator', {
        DistinguishedName: {
            CN: 'testCN',
            O: 'AWS',
            OU: 'Thinkbox',
        },
    }));
    // Expect the grantFullRead() to add this policy
    assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Cert',
                        ],
                    },
                },
                {
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Key',
                        ],
                    },
                },
                {
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        Ref: certPassphraseID,
                    },
                },
            ],
        },
    }));
    // Expect the PKCS #12 generator not to be created
    assert_1.expect(stack).notTo(assert_1.haveResource('Custom::RFDK_X509_PKCS12'));
});
test('Convert to PKCS #12', () => {
    const stack = new core_1.Stack();
    const subject = { cn: 'testCN' };
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', { subject });
    const certPassphraseID = stack.getLogicalId(cert.passphrase.node.defaultChild);
    const pkcs12Cert = new x509_certificate_1.X509CertificatePkcs12(stack, 'CertPkcs12', { sourceCertificate: cert });
    const pkcs12CertPassphraseID = stack.getLogicalId(pkcs12Cert.passphrase.node.defaultChild);
    // Expect the PKCS #12 custom resource
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_X509_PKCS12', {
        Passphrase: {
            Ref: 'CertPkcs12Passphrase1E3DF360',
        },
        Secret: {
            NamePrefix: 'Default/CertPkcs12',
            Description: 'Default/CertPkcs12',
            Tags: [
                {
                    Key: 'X509SecretGrant-71090F78',
                    Value: '71090f7809ce64f7c970cb645d4d473c',
                },
            ],
        },
        Certificate: {
            Cert: {
                'Fn::GetAtt': [
                    'Cert',
                    'Cert',
                ],
            },
            Key: {
                'Fn::GetAtt': [
                    'Cert',
                    'Key',
                ],
            },
            Passphrase: {
                Ref: certPassphraseID,
            },
        },
    }));
    // Expect the source certificate (custom resource)
    assert_1.expect(stack).to(assert_1.haveResource('Custom::RFDK_X509Generator'));
    // Expect the PKCS #12 to have a password secret
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SecretsManager::Secret', {
        Description: 'Passphrase for the private key of the X509Certificate CertPkcs12',
        GenerateSecretString: {
            ExcludeCharacters: '"()$\'',
            ExcludePunctuation: true,
            IncludeSpace: false,
            PasswordLength: 24,
            RequireEachIncludedType: true,
        },
    }));
    // Expect the PKCS #12 resource to have a policy with access to the X.509 resource
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'dynamodb:BatchGetItem',
                        'dynamodb:GetRecords',
                        'dynamodb:GetShardIterator',
                        'dynamodb:Query',
                        'dynamodb:GetItem',
                        'dynamodb:Scan',
                        'dynamodb:BatchWriteItem',
                        'dynamodb:PutItem',
                        'dynamodb:UpdateItem',
                        'dynamodb:DeleteItem',
                    ],
                },
                {
                    Action: 'dynamodb:DescribeTable',
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        Ref: pkcs12CertPassphraseID,
                    },
                },
                {
                    Action: [
                        'secretsmanager:CreateSecret',
                        'secretsmanager:DeleteSecret',
                        'secretsmanager:TagResource',
                        'secretsmanager:PutSecretValue',
                    ],
                    Condition: {
                        StringEquals: {
                            'secretsmanager:ResourceTag/X509SecretGrant-71090F78': '71090f7809ce64f7c970cb645d4d473c',
                        },
                    },
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Cert',
                        ],
                    },
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Key',
                        ],
                    },
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        Ref: certPassphraseID,
                    },
                },
            ],
        },
    }));
    // Expect no KMS key
    assert_1.expect(stack).notTo(assert_1.haveResource('AWS::KMS::Key'));
    // Expect the Lambda for converting the PEM to PKCS 12
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
        Handler: 'x509-certificate.convert',
    }));
});
test('Convert to PKCS #12, use KMS', () => {
    const stack = new core_1.Stack();
    const subject = { cn: 'testCN' };
    const sourceCertificate = new x509_certificate_1.X509CertificatePem(stack, 'Cert', { subject });
    const certPassphraseID = stack.getLogicalId(sourceCertificate.passphrase.node.defaultChild);
    const encryptionKey = new aws_kms_1.Key(stack, 'Key');
    new x509_certificate_1.X509CertificatePkcs12(stack, 'CertPkcs12', {
        sourceCertificate,
        encryptionKey,
    });
    // Expect the PKCS #12 custom resource
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_X509_PKCS12', {
        Passphrase: {
            Ref: 'CertPkcs12Passphrase1E3DF360',
        },
        Secret: {
            NamePrefix: 'Default/CertPkcs12',
            Description: 'Default/CertPkcs12',
            Tags: [
                {
                    Key: 'X509SecretGrant-71090F78',
                    Value: '71090f7809ce64f7c970cb645d4d473c',
                },
            ],
        },
        Certificate: {
            Cert: {
                'Fn::GetAtt': [
                    'Cert',
                    'Cert',
                ],
            },
            Key: {
                'Fn::GetAtt': [
                    'Cert',
                    'Key',
                ],
            },
            Passphrase: {
                Ref: certPassphraseID,
            },
        },
    }));
    // Expect the source certificate (custom resource)
    assert_1.expect(stack).to(assert_1.haveResource('Custom::RFDK_X509Generator'));
    // Expect the PKCS #12 to have a password secret
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SecretsManager::Secret', {
        Description: 'Passphrase for the private key of the X509Certificate CertPkcs12',
        GenerateSecretString: {
            ExcludeCharacters: '"()$\'',
            ExcludePunctuation: true,
            IncludeSpace: false,
            PasswordLength: 24,
            RequireEachIncludedType: true,
        },
    }));
    // Expect a KMS key for encryption
    assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key'));
    // Expect the Lambda for converting the PEM to PKCS #12
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
        Handler: 'x509-certificate.convert',
    }));
});
//# sourceMappingURL=data:application/json;base64,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