"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.testConstructTags = void 0;
const assert_1 = require("@aws-cdk/assert");
/**
 * The name of the tag that RFDK is expected to use
 */
const RFDK_TAG_NAME = 'aws-rfdk';
/**
 * The current version of the RFDK package
 */
const RFDK_VERSION = require('../../../package.json').version; // eslint-disable-line @typescript-eslint/no-require-imports
/**
 * Returns the resource properties for an expected RFDK tag for a given resource
 * type
 *
 * @param resourceType The L1 CloudFormation resource type (e.g. "AWS::EC2::Instance")
 * @param constructName The name of the L2 (or higher) RFDK construct class
 */
function getExpectedRfdkTagProperties(resourceType, constructName) {
    const expectedValue = `${RFDK_VERSION}:${constructName}`;
    if (resourceType === 'AWS::SSM::Parameter') {
        return {
            Tags: {
                [RFDK_TAG_NAME]: expectedValue,
            },
        };
    }
    else if (resourceType === 'AWS::AutoScaling::AutoScalingGroup') {
        return {
            Tags: assert_1.arrayWith({
                Key: RFDK_TAG_NAME,
                PropagateAtLaunch: true,
                Value: expectedValue,
            }),
        };
    }
    else {
        return {
            Tags: assert_1.arrayWith({
                Key: RFDK_TAG_NAME,
                Value: expectedValue,
            }),
        };
    }
}
/**
 * Creates tests for the specified resources created by a L2 (or higher construct) to ensure that the resources it
 * creates are created with the RFDK tagging convention.
 *
 * The convention is to create a tag named "aws-rfdk" whose value follows:
 *
 * <VERSION>:<CONSTRUCT_NAME>
 *
 * @param args Arguments to configure the creation of construct tagging tests
 */
function testConstructTags(args) {
    const { constructName, createConstruct, resourceTypeCounts } = args;
    const entries = Object.entries(resourceTypeCounts);
    test.each(entries)('tags %s x%d', (resourceType, expectedCount) => {
        // GIVEN
        const expectedProps = getExpectedRfdkTagProperties(resourceType, constructName);
        // WHEN
        const stack = createConstruct();
        // THEN
        assert_1.expect(stack).to(assert_1.countResourcesLike(resourceType, expectedCount, expectedProps));
    });
}
exports.testConstructTags = testConstructTags;
//# sourceMappingURL=data:application/json;base64,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