"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('Test StaticIpServer', () => {
    let stack;
    let vpc;
    beforeEach(() => {
        const app = new core_1.App();
        stack = new core_1.Stack(app, 'StackName');
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
    });
    test('basic setup', () => {
        // WHEN
        new lib_1.StaticPrivateIpServer(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
            Properties: {
                MinSize: '1',
                MaxSize: '1',
                LifecycleHookSpecificationList: assert_1.arrayWith(assert_1.objectLike({
                    DefaultResult: 'ABANDON',
                    HeartbeatTimeout: 120,
                    LifecycleHookName: 'NewStaticPrivateIpServer',
                    LifecycleTransition: 'autoscaling:EC2_INSTANCE_LAUNCHING',
                    NotificationMetadata: {
                        'Fn::Join': assert_1.arrayWith([
                            '{\"eniId\":\"',
                            {
                                Ref: 'InstanceEniA230F5FE',
                            },
                            '\"}',
                        ]),
                    },
                })),
                Tags: assert_1.arrayWith({
                    Key: 'RfdkStaticPrivateIpServerGrantConditionKey',
                    PropagateAtLaunch: true,
                    Value: 'StackNameAttachEniToInstance83a5dca5db544aa485d28d419cdf85ceF20CDF73',
                }),
            },
            DependsOn: assert_1.arrayWith('AttachEniToInstance83a5dca5db544aa485d28d419cdf85ceAttachEniNotificationTopicc8b1e9a6783c4954b191204dd5e3b9e0695D3E7F', // The SNS Topic Subscription; this is key.
            'InstanceEniA230F5FE'),
        }, assert_1.ResourcePart.CompleteDefinition));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::NetworkInterface', {
            Description: 'Static ENI for StackName/Instance',
            GroupSet: assert_1.arrayWith({
                'Fn::GetAtt': [
                    'InstanceAsgInstanceSecurityGroup2DB1DA8B',
                    'GroupId',
                ],
            }),
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
            Handler: 'index.handler',
            Runtime: 'nodejs12.x',
            Description: 'Created by RFDK StaticPrivateIpServer to process instance launch lifecycle events in stack \'StackName\'. This lambda attaches an ENI to newly launched instances.',
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SNS::Topic', {
            DisplayName: 'For RFDK instance-launch notifications for stack \'StackName\'',
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SNS::Subscription', {
            Protocol: 'lambda',
            TopicArn: {
                Ref: 'AttachEniNotificationTopicc8b1e9a6783c4954b191204dd5e3b9e0F5D22665',
            },
            Endpoint: {
                'Fn::GetAtt': [
                    'AttachEniToInstance83a5dca5db544aa485d28d419cdf85ce70724E62',
                    'Arn',
                ],
            },
        }));
        // The Lambda's policy should allow ENI attachment & condition-limited CompleteLifecycle.
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: assert_1.arrayWith(assert_1.objectLike({
                    Action: 'autoscaling:CompleteLifecycleAction',
                    Effect: 'Allow',
                    Condition: {
                        'ForAnyValue:StringEquals': {
                            'autoscaling:ResourceTag/RfdkStaticPrivateIpServerGrantConditionKey': 'StackNameAttachEniToInstance83a5dca5db544aa485d28d419cdf85ceF20CDF73',
                        },
                    },
                    Resource: {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':autoscaling:',
                                {
                                    Ref: 'AWS::Region',
                                },
                                ':',
                                {
                                    Ref: 'AWS::AccountId',
                                },
                                ':autoScalingGroup:*:autoScalingGroupName/*',
                            ],
                        ],
                    },
                }), assert_1.objectLike({
                    Effect: 'Allow',
                    Action: [
                        'ec2:DescribeNetworkInterfaces',
                        'ec2:AttachNetworkInterface',
                    ],
                    Resource: '*',
                })),
            },
        }));
        // Count singleton objects
        assert_1.expect(stack).to(assert_1.countResources('AWS::Lambda::Function', 2)); // Log retention & event handler.
        assert_1.expect(stack).to(assert_1.countResources('AWS::SNS::Topic', 1));
        assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::IAM::Role', 1, {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Effect: 'Allow',
                        Action: 'sts:AssumeRole',
                        Principal: {
                            Service: 'autoscaling.amazonaws.com',
                        },
                    },
                ],
            },
        }));
        assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::IAM::Policy', 1, {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'sns:Publish',
                        Effect: 'Allow',
                        Resource: {
                            Ref: 'AttachEniNotificationTopicc8b1e9a6783c4954b191204dd5e3b9e0F5D22665',
                        },
                    },
                ],
            },
        }));
    });
    test('creates singleton resources', () => {
        // WHEN
        new lib_1.StaticPrivateIpServer(stack, 'Instance1', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        new lib_1.StaticPrivateIpServer(stack, 'Instance2', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // THEN
        // Make sure both ASGs are tagged to allow CompleteLifeCycle by the singleton lambda.
        assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::AutoScaling::AutoScalingGroup', 2, {
            Tags: assert_1.arrayWith({
                Key: 'RfdkStaticPrivateIpServerGrantConditionKey',
                PropagateAtLaunch: true,
                Value: 'StackNameAttachEniToInstance83a5dca5db544aa485d28d419cdf85ceF20CDF73',
            }),
        }));
        // Count singleton objects
        assert_1.expect(stack).to(assert_1.countResources('AWS::Lambda::Function', 2)); // Log retention & event handler.
        assert_1.expect(stack).to(assert_1.countResources('AWS::SNS::Topic', 1));
        assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::IAM::Role', 1, {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Effect: 'Allow',
                        Action: 'sts:AssumeRole',
                        Principal: {
                            Service: 'autoscaling.amazonaws.com',
                        },
                    },
                ],
            },
        }));
    });
    test('throw exception when no available subnets', () => {
        // THEN
        expect(() => {
            new lib_1.StaticPrivateIpServer(stack, 'Instance', {
                vpc,
                instanceType: new aws_ec2_1.InstanceType('t3.small'),
                machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                vpcSubnets: {
                    subnetType: aws_ec2_1.SubnetType.PRIVATE,
                    availabilityZones: ['dummy zone'],
                },
            });
        }).toThrowError(/Did not find any subnets matching/);
    });
    test('resource signal count', () => {
        // WHEN
        new lib_1.StaticPrivateIpServer(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
            resourceSignalTimeout: core_1.Duration.hours(12),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
            CreationPolicy: {
                ResourceSignal: {
                    Count: 1,
                    Timeout: 'PT12H',
                },
            },
        }, assert_1.ResourcePart.CompleteDefinition));
        expect(() => {
            new lib_1.StaticPrivateIpServer(stack, 'InstanceFail', {
                vpc,
                instanceType: new aws_ec2_1.InstanceType('t3.small'),
                machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                resourceSignalTimeout: core_1.Duration.seconds(12 * 60 * 60 + 1),
            });
        }).toThrowError('Resource signal timeout cannot exceed 12 hours.');
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3RhdGljaXAtc2VydmVyLnRlc3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJzdGF0aWNpcC1zZXJ2ZXIudGVzdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7OztHQUdHOztBQUVILDRDQVF5QjtBQUN6Qiw4Q0FNMEI7QUFDMUIsd0NBSXVCO0FBQ3ZCLGdDQUE2QztBQUU3QyxRQUFRLENBQUMscUJBQXFCLEVBQUUsR0FBRyxFQUFFO0lBQ25DLElBQUksS0FBWSxDQUFDO0lBQ2pCLElBQUksR0FBUSxDQUFDO0lBRWIsVUFBVSxDQUFDLEdBQUcsRUFBRTtRQUNkLE1BQU0sR0FBRyxHQUFHLElBQUksVUFBRyxFQUFFLENBQUM7UUFDdEIsS0FBSyxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxXQUFXLENBQUMsQ0FBQztRQUNwQyxHQUFHLEdBQUcsSUFBSSxhQUFHLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQzlCLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLGFBQWEsRUFBRSxHQUFHLEVBQUU7UUFDdkIsT0FBTztRQUNQLElBQUksMkJBQXFCLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRTtZQUMzQyxHQUFHO1lBQ0gsWUFBWSxFQUFFLElBQUksc0JBQVksQ0FBQyxVQUFVLENBQUM7WUFDMUMsWUFBWSxFQUFFLHNCQUFZLENBQUMsaUJBQWlCLENBQUMsRUFBRSxVQUFVLEVBQUUsK0JBQXFCLENBQUMsY0FBYyxFQUFFLENBQUM7U0FDbkcsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsb0NBQW9DLEVBQUU7WUFDekUsVUFBVSxFQUFFO2dCQUNWLE9BQU8sRUFBRSxHQUFHO2dCQUNaLE9BQU8sRUFBRSxHQUFHO2dCQUNaLDhCQUE4QixFQUFFLGtCQUFTLENBQUMsbUJBQVUsQ0FBQztvQkFDbkQsYUFBYSxFQUFFLFNBQVM7b0JBQ3hCLGdCQUFnQixFQUFFLEdBQUc7b0JBQ3JCLGlCQUFpQixFQUFFLDBCQUEwQjtvQkFDN0MsbUJBQW1CLEVBQUUsb0NBQW9DO29CQUN6RCxvQkFBb0IsRUFBRTt3QkFDcEIsVUFBVSxFQUFFLGtCQUFTLENBQUM7NEJBQ3BCLGVBQWU7NEJBQ2Y7Z0NBQ0UsR0FBRyxFQUFFLHFCQUFxQjs2QkFDM0I7NEJBQ0QsS0FBSzt5QkFDTixDQUFDO3FCQUNIO2lCQUNGLENBQUMsQ0FBQztnQkFDSCxJQUFJLEVBQUUsa0JBQVMsQ0FBQztvQkFDZCxHQUFHLEVBQUUsNENBQTRDO29CQUNqRCxpQkFBaUIsRUFBRSxJQUFJO29CQUN2QixLQUFLLEVBQUUsc0VBQXNFO2lCQUM5RSxDQUFDO2FBQ0g7WUFDRCxTQUFTLEVBQUUsa0JBQVMsQ0FDbEIsdUhBQXVILEVBQUUsMkNBQTJDO1lBQ3BLLHFCQUFxQixDQUN0QjtTQUNGLEVBQUUscUJBQVksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUM7UUFFckMsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyw0QkFBNEIsRUFBRTtZQUNqRSxXQUFXLEVBQUUsbUNBQW1DO1lBQ2hELFFBQVEsRUFBRSxrQkFBUyxDQUFDO2dCQUNsQixZQUFZLEVBQUU7b0JBQ1osMENBQTBDO29CQUMxQyxTQUFTO2lCQUNWO2FBQ0YsQ0FBQztTQUNILENBQUMsQ0FBQyxDQUFDO1FBRUosZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyx1QkFBdUIsRUFBRTtZQUM1RCxPQUFPLEVBQUUsZUFBZTtZQUN4QixPQUFPLEVBQUUsWUFBWTtZQUNyQixXQUFXLEVBQUUsb0tBQW9LO1NBQ2xMLENBQUMsQ0FBQyxDQUFDO1FBRUosZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxpQkFBaUIsRUFBRTtZQUN0RCxXQUFXLEVBQUUsZ0VBQWdFO1NBQzlFLENBQUMsQ0FBQyxDQUFDO1FBRUosZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyx3QkFBd0IsRUFBRTtZQUM3RCxRQUFRLEVBQUUsUUFBUTtZQUNsQixRQUFRLEVBQUU7Z0JBQ1IsR0FBRyxFQUFFLG9FQUFvRTthQUMxRTtZQUNELFFBQVEsRUFBRTtnQkFDUixZQUFZLEVBQUU7b0JBQ1osNkRBQTZEO29CQUM3RCxLQUFLO2lCQUNOO2FBQ0Y7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLHlGQUF5RjtRQUN6RixlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLGtCQUFrQixFQUFFO1lBQ3ZELGNBQWMsRUFBRTtnQkFDZCxTQUFTLEVBQUUsa0JBQVMsQ0FDbEIsbUJBQVUsQ0FBQztvQkFDVCxNQUFNLEVBQUUscUNBQXFDO29CQUM3QyxNQUFNLEVBQUUsT0FBTztvQkFDZixTQUFTLEVBQUU7d0JBQ1QsMEJBQTBCLEVBQUU7NEJBQzFCLG9FQUFvRSxFQUFFLHNFQUFzRTt5QkFDN0k7cUJBQ0Y7b0JBQ0QsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRTs0QkFDVixFQUFFOzRCQUNGO2dDQUNFLE1BQU07Z0NBQ047b0NBQ0UsR0FBRyxFQUFFLGdCQUFnQjtpQ0FDdEI7Z0NBQ0QsZUFBZTtnQ0FDZjtvQ0FDRSxHQUFHLEVBQUUsYUFBYTtpQ0FDbkI7Z0NBQ0QsR0FBRztnQ0FDSDtvQ0FDRSxHQUFHLEVBQUUsZ0JBQWdCO2lDQUN0QjtnQ0FDRCw0Q0FBNEM7NkJBQzdDO3lCQUNGO3FCQUNGO2lCQUNGLENBQUMsRUFDRixtQkFBVSxDQUFDO29CQUNULE1BQU0sRUFBRSxPQUFPO29CQUNmLE1BQU0sRUFBRTt3QkFDTiwrQkFBK0I7d0JBQy9CLDRCQUE0QjtxQkFDN0I7b0JBQ0QsUUFBUSxFQUFFLEdBQUc7aUJBQ2QsQ0FBQyxDQUNIO2FBQ0Y7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLDBCQUEwQjtRQUMxQixlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHVCQUFjLENBQUMsdUJBQXVCLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLGlDQUFpQztRQUNsRyxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHVCQUFjLENBQUMsaUJBQWlCLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUMxRCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLDJCQUFrQixDQUFDLGdCQUFnQixFQUFFLENBQUMsRUFBRTtZQUMxRCx3QkFBd0IsRUFBRTtnQkFDeEIsU0FBUyxFQUFFO29CQUNUO3dCQUNFLE1BQU0sRUFBRSxPQUFPO3dCQUNmLE1BQU0sRUFBRSxnQkFBZ0I7d0JBQ3hCLFNBQVMsRUFBRTs0QkFDVCxPQUFPLEVBQUUsMkJBQTJCO3lCQUNyQztxQkFDRjtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFDSixlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLDJCQUFrQixDQUFDLGtCQUFrQixFQUFFLENBQUMsRUFBRTtZQUM1RCxjQUFjLEVBQUU7Z0JBQ2QsU0FBUyxFQUFFO29CQUNUO3dCQUNFLE1BQU0sRUFBRSxhQUFhO3dCQUNyQixNQUFNLEVBQUUsT0FBTzt3QkFDZixRQUFRLEVBQUU7NEJBQ1IsR0FBRyxFQUFFLG9FQUFvRTt5QkFDMUU7cUJBQ0Y7aUJBQ0Y7YUFDRjtTQUNGLENBQUMsQ0FBQyxDQUFDO0lBQ04sQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsNkJBQTZCLEVBQUUsR0FBRyxFQUFFO1FBQ3ZDLE9BQU87UUFDUCxJQUFJLDJCQUFxQixDQUFDLEtBQUssRUFBRSxXQUFXLEVBQUU7WUFDNUMsR0FBRztZQUNILFlBQVksRUFBRSxJQUFJLHNCQUFZLENBQUMsVUFBVSxDQUFDO1lBQzFDLFlBQVksRUFBRSxzQkFBWSxDQUFDLGlCQUFpQixDQUFDLEVBQUUsVUFBVSxFQUFFLCtCQUFxQixDQUFDLGNBQWMsRUFBRSxDQUFDO1NBQ25HLENBQUMsQ0FBQztRQUNILElBQUksMkJBQXFCLENBQUMsS0FBSyxFQUFFLFdBQVcsRUFBRTtZQUM1QyxHQUFHO1lBQ0gsWUFBWSxFQUFFLElBQUksc0JBQVksQ0FBQyxVQUFVLENBQUM7WUFDMUMsWUFBWSxFQUFFLHNCQUFZLENBQUMsaUJBQWlCLENBQUMsRUFBRSxVQUFVLEVBQUUsK0JBQXFCLENBQUMsY0FBYyxFQUFFLENBQUM7U0FDbkcsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLHFGQUFxRjtRQUNyRixlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLDJCQUFrQixDQUFDLG9DQUFvQyxFQUFFLENBQUMsRUFBRTtZQUM5RSxJQUFJLEVBQUUsa0JBQVMsQ0FBQztnQkFDZCxHQUFHLEVBQUUsNENBQTRDO2dCQUNqRCxpQkFBaUIsRUFBRSxJQUFJO2dCQUN2QixLQUFLLEVBQUUsc0VBQXNFO2FBQzlFLENBQUM7U0FDSCxDQUFDLENBQUMsQ0FBQztRQUVKLDBCQUEwQjtRQUMxQixlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHVCQUFjLENBQUMsdUJBQXVCLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLGlDQUFpQztRQUNsRyxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHVCQUFjLENBQUMsaUJBQWlCLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUMxRCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLDJCQUFrQixDQUFDLGdCQUFnQixFQUFFLENBQUMsRUFBRTtZQUMxRCx3QkFBd0IsRUFBRTtnQkFDeEIsU0FBUyxFQUFFO29CQUNUO3dCQUNFLE1BQU0sRUFBRSxPQUFPO3dCQUNmLE1BQU0sRUFBRSxnQkFBZ0I7d0JBQ3hCLFNBQVMsRUFBRTs0QkFDVCxPQUFPLEVBQUUsMkJBQTJCO3lCQUNyQztxQkFDRjtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDLENBQUM7SUFDTixDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQywyQ0FBMkMsRUFBRSxHQUFHLEVBQUU7UUFDckQsT0FBTztRQUNQLE1BQU0sQ0FBQyxHQUFHLEVBQUU7WUFDVixJQUFJLDJCQUFxQixDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7Z0JBQzNDLEdBQUc7Z0JBQ0gsWUFBWSxFQUFFLElBQUksc0JBQVksQ0FBQyxVQUFVLENBQUM7Z0JBQzFDLFlBQVksRUFBRSxzQkFBWSxDQUFDLGlCQUFpQixDQUFDLEVBQUUsVUFBVSxFQUFFLCtCQUFxQixDQUFDLGNBQWMsRUFBRSxDQUFDO2dCQUNsRyxVQUFVLEVBQUU7b0JBQ1YsVUFBVSxFQUFFLG9CQUFVLENBQUMsT0FBTztvQkFDOUIsaUJBQWlCLEVBQUUsQ0FBQyxZQUFZLENBQUM7aUJBQ2xDO2FBQ0YsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUMsWUFBWSxDQUFDLG1DQUFtQyxDQUFDLENBQUM7SUFDdkQsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsdUJBQXVCLEVBQUUsR0FBRyxFQUFFO1FBQ2pDLE9BQU87UUFDUCxJQUFJLDJCQUFxQixDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7WUFDM0MsR0FBRztZQUNILFlBQVksRUFBRSxJQUFJLHNCQUFZLENBQUMsVUFBVSxDQUFDO1lBQzFDLFlBQVksRUFBRSxzQkFBWSxDQUFDLGlCQUFpQixDQUFDLEVBQUUsVUFBVSxFQUFFLCtCQUFxQixDQUFDLGNBQWMsRUFBRSxDQUFDO1lBQ2xHLHFCQUFxQixFQUFFLGVBQVEsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDO1NBQzFDLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLG9DQUFvQyxFQUFFO1lBQ3pFLGNBQWMsRUFBRTtnQkFDZCxjQUFjLEVBQUU7b0JBQ2QsS0FBSyxFQUFFLENBQUM7b0JBQ1IsT0FBTyxFQUFFLE9BQU87aUJBQ2pCO2FBQ0Y7U0FDRixFQUFFLHFCQUFZLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDO1FBQ3JDLE1BQU0sQ0FBQyxHQUFHLEVBQUU7WUFDVixJQUFJLDJCQUFxQixDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUU7Z0JBQy9DLEdBQUc7Z0JBQ0gsWUFBWSxFQUFFLElBQUksc0JBQVksQ0FBQyxVQUFVLENBQUM7Z0JBQzFDLFlBQVksRUFBRSxzQkFBWSxDQUFDLGlCQUFpQixDQUFDLEVBQUUsVUFBVSxFQUFFLCtCQUFxQixDQUFDLGNBQWMsRUFBRSxDQUFDO2dCQUNsRyxxQkFBcUIsRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLENBQUMsQ0FBQzthQUMxRCxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQyxZQUFZLENBQUMsaURBQWlELENBQUMsQ0FBQztJQUNyRSxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBDb3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4gKi9cblxuaW1wb3J0IHtcbiAgYXJyYXlXaXRoLFxuICBjb3VudFJlc291cmNlcyxcbiAgY291bnRSZXNvdXJjZXNMaWtlLFxuICBleHBlY3QgYXMgY2RrRXhwZWN0LFxuICBoYXZlUmVzb3VyY2VMaWtlLFxuICBvYmplY3RMaWtlLFxuICBSZXNvdXJjZVBhcnQsXG59IGZyb20gJ0Bhd3MtY2RrL2Fzc2VydCc7XG5pbXBvcnQge1xuICBBbWF6b25MaW51eEdlbmVyYXRpb24sXG4gIEluc3RhbmNlVHlwZSxcbiAgTWFjaGluZUltYWdlLFxuICBTdWJuZXRUeXBlLFxuICBWcGMsXG59IGZyb20gJ0Bhd3MtY2RrL2F3cy1lYzInO1xuaW1wb3J0IHtcbiAgQXBwLFxuICBEdXJhdGlvbixcbiAgU3RhY2ssXG59IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHtTdGF0aWNQcml2YXRlSXBTZXJ2ZXJ9IGZyb20gJy4uL2xpYic7XG5cbmRlc2NyaWJlKCdUZXN0IFN0YXRpY0lwU2VydmVyJywgKCkgPT4ge1xuICBsZXQgc3RhY2s6IFN0YWNrO1xuICBsZXQgdnBjOiBWcGM7XG5cbiAgYmVmb3JlRWFjaCgoKSA9PiB7XG4gICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICAgIHN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ1N0YWNrTmFtZScpO1xuICAgIHZwYyA9IG5ldyBWcGMoc3RhY2ssICdWcGMnKTtcbiAgfSk7XG5cbiAgdGVzdCgnYmFzaWMgc2V0dXAnLCAoKSA9PiB7XG4gICAgLy8gV0hFTlxuICAgIG5ldyBTdGF0aWNQcml2YXRlSXBTZXJ2ZXIoc3RhY2ssICdJbnN0YW5jZScsIHtcbiAgICAgIHZwYyxcbiAgICAgIGluc3RhbmNlVHlwZTogbmV3IEluc3RhbmNlVHlwZSgndDMuc21hbGwnKSxcbiAgICAgIG1hY2hpbmVJbWFnZTogTWFjaGluZUltYWdlLmxhdGVzdEFtYXpvbkxpbnV4KHsgZ2VuZXJhdGlvbjogQW1hem9uTGludXhHZW5lcmF0aW9uLkFNQVpPTl9MSU5VWF8yIH0pLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGNka0V4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpBdXRvU2NhbGluZzo6QXV0b1NjYWxpbmdHcm91cCcsIHtcbiAgICAgIFByb3BlcnRpZXM6IHtcbiAgICAgICAgTWluU2l6ZTogJzEnLFxuICAgICAgICBNYXhTaXplOiAnMScsXG4gICAgICAgIExpZmVjeWNsZUhvb2tTcGVjaWZpY2F0aW9uTGlzdDogYXJyYXlXaXRoKG9iamVjdExpa2Uoe1xuICAgICAgICAgIERlZmF1bHRSZXN1bHQ6ICdBQkFORE9OJyxcbiAgICAgICAgICBIZWFydGJlYXRUaW1lb3V0OiAxMjAsXG4gICAgICAgICAgTGlmZWN5Y2xlSG9va05hbWU6ICdOZXdTdGF0aWNQcml2YXRlSXBTZXJ2ZXInLFxuICAgICAgICAgIExpZmVjeWNsZVRyYW5zaXRpb246ICdhdXRvc2NhbGluZzpFQzJfSU5TVEFOQ0VfTEFVTkNISU5HJyxcbiAgICAgICAgICBOb3RpZmljYXRpb25NZXRhZGF0YToge1xuICAgICAgICAgICAgJ0ZuOjpKb2luJzogYXJyYXlXaXRoKFtcbiAgICAgICAgICAgICAgJ3tcXFwiZW5pSWRcXFwiOlxcXCInLFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgUmVmOiAnSW5zdGFuY2VFbmlBMjMwRjVGRScsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICdcXFwifScsXG4gICAgICAgICAgICBdKSxcbiAgICAgICAgICB9LFxuICAgICAgICB9KSksXG4gICAgICAgIFRhZ3M6IGFycmF5V2l0aCh7XG4gICAgICAgICAgS2V5OiAnUmZka1N0YXRpY1ByaXZhdGVJcFNlcnZlckdyYW50Q29uZGl0aW9uS2V5JyxcbiAgICAgICAgICBQcm9wYWdhdGVBdExhdW5jaDogdHJ1ZSxcbiAgICAgICAgICBWYWx1ZTogJ1N0YWNrTmFtZUF0dGFjaEVuaVRvSW5zdGFuY2U4M2E1ZGNhNWRiNTQ0YWE0ODVkMjhkNDE5Y2RmODVjZUYyMENERjczJyxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgICAgRGVwZW5kc09uOiBhcnJheVdpdGgoXG4gICAgICAgICdBdHRhY2hFbmlUb0luc3RhbmNlODNhNWRjYTVkYjU0NGFhNDg1ZDI4ZDQxOWNkZjg1Y2VBdHRhY2hFbmlOb3RpZmljYXRpb25Ub3BpY2M4YjFlOWE2NzgzYzQ5NTRiMTkxMjA0ZGQ1ZTNiOWUwNjk1RDNFN0YnLCAvLyBUaGUgU05TIFRvcGljIFN1YnNjcmlwdGlvbjsgdGhpcyBpcyBrZXkuXG4gICAgICAgICdJbnN0YW5jZUVuaUEyMzBGNUZFJywgLy8gVGhlIE5ldFdvcmtJbnRlcmZhY2UuIEFsc28ga2V5LlxuICAgICAgKSxcbiAgICB9LCBSZXNvdXJjZVBhcnQuQ29tcGxldGVEZWZpbml0aW9uKSk7XG5cbiAgICBjZGtFeHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RUMyOjpOZXR3b3JrSW50ZXJmYWNlJywge1xuICAgICAgRGVzY3JpcHRpb246ICdTdGF0aWMgRU5JIGZvciBTdGFja05hbWUvSW5zdGFuY2UnLFxuICAgICAgR3JvdXBTZXQ6IGFycmF5V2l0aCh7XG4gICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICdJbnN0YW5jZUFzZ0luc3RhbmNlU2VjdXJpdHlHcm91cDJEQjFEQThCJyxcbiAgICAgICAgICAnR3JvdXBJZCcsXG4gICAgICAgIF0sXG4gICAgICB9KSxcbiAgICB9KSk7XG5cbiAgICBjZGtFeHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6TGFtYmRhOjpGdW5jdGlvbicsIHtcbiAgICAgIEhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJyxcbiAgICAgIFJ1bnRpbWU6ICdub2RlanMxMi54JyxcbiAgICAgIERlc2NyaXB0aW9uOiAnQ3JlYXRlZCBieSBSRkRLIFN0YXRpY1ByaXZhdGVJcFNlcnZlciB0byBwcm9jZXNzIGluc3RhbmNlIGxhdW5jaCBsaWZlY3ljbGUgZXZlbnRzIGluIHN0YWNrIFxcJ1N0YWNrTmFtZVxcJy4gVGhpcyBsYW1iZGEgYXR0YWNoZXMgYW4gRU5JIHRvIG5ld2x5IGxhdW5jaGVkIGluc3RhbmNlcy4nLFxuICAgIH0pKTtcblxuICAgIGNka0V4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpTTlM6OlRvcGljJywge1xuICAgICAgRGlzcGxheU5hbWU6ICdGb3IgUkZESyBpbnN0YW5jZS1sYXVuY2ggbm90aWZpY2F0aW9ucyBmb3Igc3RhY2sgXFwnU3RhY2tOYW1lXFwnJyxcbiAgICB9KSk7XG5cbiAgICBjZGtFeHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6U05TOjpTdWJzY3JpcHRpb24nLCB7XG4gICAgICBQcm90b2NvbDogJ2xhbWJkYScsXG4gICAgICBUb3BpY0Fybjoge1xuICAgICAgICBSZWY6ICdBdHRhY2hFbmlOb3RpZmljYXRpb25Ub3BpY2M4YjFlOWE2NzgzYzQ5NTRiMTkxMjA0ZGQ1ZTNiOWUwRjVEMjI2NjUnLFxuICAgICAgfSxcbiAgICAgIEVuZHBvaW50OiB7XG4gICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICdBdHRhY2hFbmlUb0luc3RhbmNlODNhNWRjYTVkYjU0NGFhNDg1ZDI4ZDQxOWNkZjg1Y2U3MDcyNEU2MicsXG4gICAgICAgICAgJ0FybicsXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgIH0pKTtcblxuICAgIC8vIFRoZSBMYW1iZGEncyBwb2xpY3kgc2hvdWxkIGFsbG93IEVOSSBhdHRhY2htZW50ICYgY29uZGl0aW9uLWxpbWl0ZWQgQ29tcGxldGVMaWZlY3ljbGUuXG4gICAgY2RrRXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgU3RhdGVtZW50OiBhcnJheVdpdGgoXG4gICAgICAgICAgb2JqZWN0TGlrZSh7XG4gICAgICAgICAgICBBY3Rpb246ICdhdXRvc2NhbGluZzpDb21wbGV0ZUxpZmVjeWNsZUFjdGlvbicsXG4gICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICBDb25kaXRpb246IHtcbiAgICAgICAgICAgICAgJ0ZvckFueVZhbHVlOlN0cmluZ0VxdWFscyc6IHtcbiAgICAgICAgICAgICAgICAnYXV0b3NjYWxpbmc6UmVzb3VyY2VUYWcvUmZka1N0YXRpY1ByaXZhdGVJcFNlcnZlckdyYW50Q29uZGl0aW9uS2V5JzogJ1N0YWNrTmFtZUF0dGFjaEVuaVRvSW5zdGFuY2U4M2E1ZGNhNWRiNTQ0YWE0ODVkMjhkNDE5Y2RmODVjZUYyMENERjczJyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBSZXNvdXJjZToge1xuICAgICAgICAgICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICAgICAgICAgJycsXG4gICAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgICAgJ2FybjonLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBSZWY6ICdBV1M6OlBhcnRpdGlvbicsXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgJzphdXRvc2NhbGluZzonLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBSZWY6ICdBV1M6OlJlZ2lvbicsXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgJzonLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBSZWY6ICdBV1M6OkFjY291bnRJZCcsXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgJzphdXRvU2NhbGluZ0dyb3VwOio6YXV0b1NjYWxpbmdHcm91cE5hbWUvKicsXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSksXG4gICAgICAgICAgb2JqZWN0TGlrZSh7XG4gICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICBBY3Rpb246IFtcbiAgICAgICAgICAgICAgJ2VjMjpEZXNjcmliZU5ldHdvcmtJbnRlcmZhY2VzJyxcbiAgICAgICAgICAgICAgJ2VjMjpBdHRhY2hOZXR3b3JrSW50ZXJmYWNlJyxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBSZXNvdXJjZTogJyonLFxuICAgICAgICAgIH0pLFxuICAgICAgICApLFxuICAgICAgfSxcbiAgICB9KSk7XG5cbiAgICAvLyBDb3VudCBzaW5nbGV0b24gb2JqZWN0c1xuICAgIGNka0V4cGVjdChzdGFjaykudG8oY291bnRSZXNvdXJjZXMoJ0FXUzo6TGFtYmRhOjpGdW5jdGlvbicsIDIpKTsgLy8gTG9nIHJldGVudGlvbiAmIGV2ZW50IGhhbmRsZXIuXG4gICAgY2RrRXhwZWN0KHN0YWNrKS50byhjb3VudFJlc291cmNlcygnQVdTOjpTTlM6OlRvcGljJywgMSkpO1xuICAgIGNka0V4cGVjdChzdGFjaykudG8oY291bnRSZXNvdXJjZXNMaWtlKCdBV1M6OklBTTo6Um9sZScsIDEsIHtcbiAgICAgIEFzc3VtZVJvbGVQb2xpY3lEb2N1bWVudDoge1xuICAgICAgICBTdGF0ZW1lbnQ6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICBBY3Rpb246ICdzdHM6QXNzdW1lUm9sZScsXG4gICAgICAgICAgICBQcmluY2lwYWw6IHtcbiAgICAgICAgICAgICAgU2VydmljZTogJ2F1dG9zY2FsaW5nLmFtYXpvbmF3cy5jb20nLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9KSk7XG4gICAgY2RrRXhwZWN0KHN0YWNrKS50byhjb3VudFJlc291cmNlc0xpa2UoJ0FXUzo6SUFNOjpQb2xpY3knLCAxLCB7XG4gICAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgICBTdGF0ZW1lbnQ6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBBY3Rpb246ICdzbnM6UHVibGlzaCcsXG4gICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICBSZXNvdXJjZToge1xuICAgICAgICAgICAgICBSZWY6ICdBdHRhY2hFbmlOb3RpZmljYXRpb25Ub3BpY2M4YjFlOWE2NzgzYzQ5NTRiMTkxMjA0ZGQ1ZTNiOWUwRjVEMjI2NjUnLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9KSk7XG4gIH0pO1xuXG4gIHRlc3QoJ2NyZWF0ZXMgc2luZ2xldG9uIHJlc291cmNlcycsICgpID0+IHtcbiAgICAvLyBXSEVOXG4gICAgbmV3IFN0YXRpY1ByaXZhdGVJcFNlcnZlcihzdGFjaywgJ0luc3RhbmNlMScsIHtcbiAgICAgIHZwYyxcbiAgICAgIGluc3RhbmNlVHlwZTogbmV3IEluc3RhbmNlVHlwZSgndDMuc21hbGwnKSxcbiAgICAgIG1hY2hpbmVJbWFnZTogTWFjaGluZUltYWdlLmxhdGVzdEFtYXpvbkxpbnV4KHsgZ2VuZXJhdGlvbjogQW1hem9uTGludXhHZW5lcmF0aW9uLkFNQVpPTl9MSU5VWF8yIH0pLFxuICAgIH0pO1xuICAgIG5ldyBTdGF0aWNQcml2YXRlSXBTZXJ2ZXIoc3RhY2ssICdJbnN0YW5jZTInLCB7XG4gICAgICB2cGMsXG4gICAgICBpbnN0YW5jZVR5cGU6IG5ldyBJbnN0YW5jZVR5cGUoJ3QzLnNtYWxsJyksXG4gICAgICBtYWNoaW5lSW1hZ2U6IE1hY2hpbmVJbWFnZS5sYXRlc3RBbWF6b25MaW51eCh7IGdlbmVyYXRpb246IEFtYXpvbkxpbnV4R2VuZXJhdGlvbi5BTUFaT05fTElOVVhfMiB9KSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICAvLyBNYWtlIHN1cmUgYm90aCBBU0dzIGFyZSB0YWdnZWQgdG8gYWxsb3cgQ29tcGxldGVMaWZlQ3ljbGUgYnkgdGhlIHNpbmdsZXRvbiBsYW1iZGEuXG4gICAgY2RrRXhwZWN0KHN0YWNrKS50byhjb3VudFJlc291cmNlc0xpa2UoJ0FXUzo6QXV0b1NjYWxpbmc6OkF1dG9TY2FsaW5nR3JvdXAnLCAyLCB7XG4gICAgICBUYWdzOiBhcnJheVdpdGgoe1xuICAgICAgICBLZXk6ICdSZmRrU3RhdGljUHJpdmF0ZUlwU2VydmVyR3JhbnRDb25kaXRpb25LZXknLFxuICAgICAgICBQcm9wYWdhdGVBdExhdW5jaDogdHJ1ZSxcbiAgICAgICAgVmFsdWU6ICdTdGFja05hbWVBdHRhY2hFbmlUb0luc3RhbmNlODNhNWRjYTVkYjU0NGFhNDg1ZDI4ZDQxOWNkZjg1Y2VGMjBDREY3MycsXG4gICAgICB9KSxcbiAgICB9KSk7XG5cbiAgICAvLyBDb3VudCBzaW5nbGV0b24gb2JqZWN0c1xuICAgIGNka0V4cGVjdChzdGFjaykudG8oY291bnRSZXNvdXJjZXMoJ0FXUzo6TGFtYmRhOjpGdW5jdGlvbicsIDIpKTsgLy8gTG9nIHJldGVudGlvbiAmIGV2ZW50IGhhbmRsZXIuXG4gICAgY2RrRXhwZWN0KHN0YWNrKS50byhjb3VudFJlc291cmNlcygnQVdTOjpTTlM6OlRvcGljJywgMSkpO1xuICAgIGNka0V4cGVjdChzdGFjaykudG8oY291bnRSZXNvdXJjZXNMaWtlKCdBV1M6OklBTTo6Um9sZScsIDEsIHtcbiAgICAgIEFzc3VtZVJvbGVQb2xpY3lEb2N1bWVudDoge1xuICAgICAgICBTdGF0ZW1lbnQ6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICBBY3Rpb246ICdzdHM6QXNzdW1lUm9sZScsXG4gICAgICAgICAgICBQcmluY2lwYWw6IHtcbiAgICAgICAgICAgICAgU2VydmljZTogJ2F1dG9zY2FsaW5nLmFtYXpvbmF3cy5jb20nLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9KSk7XG4gIH0pO1xuXG4gIHRlc3QoJ3Rocm93IGV4Y2VwdGlvbiB3aGVuIG5vIGF2YWlsYWJsZSBzdWJuZXRzJywgKCkgPT4ge1xuICAgIC8vIFRIRU5cbiAgICBleHBlY3QoKCkgPT4ge1xuICAgICAgbmV3IFN0YXRpY1ByaXZhdGVJcFNlcnZlcihzdGFjaywgJ0luc3RhbmNlJywge1xuICAgICAgICB2cGMsXG4gICAgICAgIGluc3RhbmNlVHlwZTogbmV3IEluc3RhbmNlVHlwZSgndDMuc21hbGwnKSxcbiAgICAgICAgbWFjaGluZUltYWdlOiBNYWNoaW5lSW1hZ2UubGF0ZXN0QW1hem9uTGludXgoeyBnZW5lcmF0aW9uOiBBbWF6b25MaW51eEdlbmVyYXRpb24uQU1BWk9OX0xJTlVYXzIgfSksXG4gICAgICAgIHZwY1N1Ym5ldHM6IHtcbiAgICAgICAgICBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBSSVZBVEUsXG4gICAgICAgICAgYXZhaWxhYmlsaXR5Wm9uZXM6IFsnZHVtbXkgem9uZSddLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfSkudG9UaHJvd0Vycm9yKC9EaWQgbm90IGZpbmQgYW55IHN1Ym5ldHMgbWF0Y2hpbmcvKTtcbiAgfSk7XG5cbiAgdGVzdCgncmVzb3VyY2Ugc2lnbmFsIGNvdW50JywgKCkgPT4ge1xuICAgIC8vIFdIRU5cbiAgICBuZXcgU3RhdGljUHJpdmF0ZUlwU2VydmVyKHN0YWNrLCAnSW5zdGFuY2UnLCB7XG4gICAgICB2cGMsXG4gICAgICBpbnN0YW5jZVR5cGU6IG5ldyBJbnN0YW5jZVR5cGUoJ3QzLnNtYWxsJyksXG4gICAgICBtYWNoaW5lSW1hZ2U6IE1hY2hpbmVJbWFnZS5sYXRlc3RBbWF6b25MaW51eCh7IGdlbmVyYXRpb246IEFtYXpvbkxpbnV4R2VuZXJhdGlvbi5BTUFaT05fTElOVVhfMiB9KSxcbiAgICAgIHJlc291cmNlU2lnbmFsVGltZW91dDogRHVyYXRpb24uaG91cnMoMTIpLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGNka0V4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpBdXRvU2NhbGluZzo6QXV0b1NjYWxpbmdHcm91cCcsIHtcbiAgICAgIENyZWF0aW9uUG9saWN5OiB7XG4gICAgICAgIFJlc291cmNlU2lnbmFsOiB7XG4gICAgICAgICAgQ291bnQ6IDEsXG4gICAgICAgICAgVGltZW91dDogJ1BUMTJIJyxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSwgUmVzb3VyY2VQYXJ0LkNvbXBsZXRlRGVmaW5pdGlvbikpO1xuICAgIGV4cGVjdCgoKSA9PiB7XG4gICAgICBuZXcgU3RhdGljUHJpdmF0ZUlwU2VydmVyKHN0YWNrLCAnSW5zdGFuY2VGYWlsJywge1xuICAgICAgICB2cGMsXG4gICAgICAgIGluc3RhbmNlVHlwZTogbmV3IEluc3RhbmNlVHlwZSgndDMuc21hbGwnKSxcbiAgICAgICAgbWFjaGluZUltYWdlOiBNYWNoaW5lSW1hZ2UubGF0ZXN0QW1hem9uTGludXgoeyBnZW5lcmF0aW9uOiBBbWF6b25MaW51eEdlbmVyYXRpb24uQU1BWk9OX0xJTlVYXzIgfSksXG4gICAgICAgIHJlc291cmNlU2lnbmFsVGltZW91dDogRHVyYXRpb24uc2Vjb25kcygxMiAqIDYwICogNjAgKyAxKSxcbiAgICAgIH0pO1xuICAgIH0pLnRvVGhyb3dFcnJvcignUmVzb3VyY2Ugc2lnbmFsIHRpbWVvdXQgY2Fubm90IGV4Y2VlZCAxMiBob3Vycy4nKTtcbiAgfSk7XG59KTtcbiJdfQ==