"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const imported_acm_certificate_1 = require("../lib/imported-acm-certificate");
const x509_certificate_1 = require("../lib/x509-certificate");
test('Import cert', () => {
    const stack = new core_1.Stack(undefined, 'Stack', { env: { region: 'us-west-2' } });
    const secretCert = new x509_certificate_1.X509CertificatePem(stack, 'Pem', {
        subject: { cn: 'Server' },
    });
    const certPassphraseID = stack.getLogicalId(secretCert.passphrase.node.defaultChild);
    new imported_acm_certificate_1.ImportedAcmCertificate(stack, 'AcmCert', {
        cert: secretCert.cert,
        certChain: secretCert.certChain,
        key: secretCert.key,
        passphrase: secretCert.passphrase,
    });
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_AcmImportedCertificate', {
        X509CertificatePem: {
            Cert: {
                'Fn::GetAtt': [
                    'Pem',
                    'Cert',
                ],
            },
            Key: {
                'Fn::GetAtt': [
                    'Pem',
                    'Key',
                ],
            },
            Passphrase: {
                Ref: certPassphraseID,
            },
            CertChain: '',
        },
        Tags: [
            {
                Key: 'AcmCertImport-F4E2ABF9',
                Value: 'f4e2abf974443234fdb095fafcfa9ee2',
            },
            {
                Key: 'Name',
                Value: 'f4e2abf974443234fdb095fafcfa9ee2',
            },
        ],
    }));
    assert_1.expect(stack).to(assert_1.countResources('AWS::DynamoDB::Table', 2));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'dynamodb:BatchGetItem',
                        'dynamodb:GetRecords',
                        'dynamodb:GetShardIterator',
                        'dynamodb:Query',
                        'dynamodb:GetItem',
                        'dynamodb:Scan',
                        'dynamodb:BatchWriteItem',
                        'dynamodb:PutItem',
                        'dynamodb:UpdateItem',
                        'dynamodb:DeleteItem',
                    ],
                },
                {
                    Action: 'dynamodb:DescribeTable',
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        'Fn::GetAtt': [
                            'Pem',
                            'Cert',
                        ],
                    },
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        'Fn::GetAtt': [
                            'Pem',
                            'Key',
                        ],
                    },
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        Ref: certPassphraseID,
                    },
                },
                {
                    Action: [
                        'acm:AddTagsToCertificate',
                        'acm:ImportCertificate',
                    ],
                    Condition: {
                        StringEquals: {
                            'aws:RequestTag/AcmCertImport-F4E2ABF9': 'f4e2abf974443234fdb095fafcfa9ee2',
                        },
                    },
                    Resource: '*',
                },
                {
                    Action: [
                        'acm:DeleteCertificate',
                        'acm:DescribeCertificate',
                        'acm:GetCertificate',
                    ],
                    Resource: '*',
                },
            ],
        },
    }));
    // Expect Lambda for doing the cert importation to use the importCert() handler, openssl layer, and set DATABASE
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', (props, error) => {
        var _a, _b;
        if (!props.Handler || props.Handler !== 'x509-certificate.importCert') {
            error.failureReason = 'x509-certificate.importCert handler not found';
            error.resource = props.Handler;
            return false;
        }
        // Our test for the correct openssl lambda layer does not include the version, so we use a filter
        // function to do a partial match
        const filterOpensslArn = (value) => {
            return value.toString().includes('arn:aws:lambda:us-west-2:224375009292:layer:openssl-al2:');
        };
        if (!props.Layers
            || !Array.isArray(props.Layers)
            || Array.of(props.Layers).filter(filterOpensslArn).length === 0) {
            error.failureReason = 'openssl Lambda Layer missing';
            error.resource = props.Layers;
            return false;
        }
        if (!props.Environment
            || !props.Environment.Variables
            || !props.Environment.Variables.DATABASE) {
            error.failureReason = 'DATABASE environment variable not set';
            error.resource = (_b = (_a = props.Environment) === null || _a === void 0 ? void 0 : _a.Variables) === null || _b === void 0 ? void 0 : _b.DATABASE;
            return false;
        }
        return true;
    }));
});
//# sourceMappingURL=data:application/json;base64,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