"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.StaticPrivateIpServer = void 0;
const path = require("path");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_sns_1 = require("@aws-cdk/aws-sns");
const aws_sns_subscriptions_1 = require("@aws-cdk/aws-sns-subscriptions");
const core_1 = require("@aws-cdk/core");
/**
 * This construct provides a single instance, provided by an Auto Scaling Group (ASG), that has an attached Elastic Network Interface (ENI) that is providing a private ip address.
 *
 * This ENI is automatically re-attached to the instance if the instance is replaced
 * by the ASG.
 *
 * The ENI provides an unchanging private IP address that can always be used to connect
 * to the instance regardless of how many times the instance has been replaced. Furthermore,
 * the ENI has a MAC address that remains unchanged unless the ENI is destroyed.
 *
 * Essentially, this provides an instance with an unchanging private IP address that will
 * automatically recover from termination. This instance is suitable for use as an application server,
 * such as a license server, that must always be reachable by the same IP address.
 *
 * Resources Deployed
 * ------------------------
 * - Auto Scaling Group (ASG) with min & max capacity of 1 instance.
 * - Elastic Network Interface (ENI).
 * - Security Group for the ASG.
 * - Instance Role and corresponding IAM Policy.
 * - SNS Topic & Role for instance-launch lifecycle events -- max one of each per stack.
 * - Lambda function, with role, to attach the ENI in response to instance-launch lifecycle events -- max one per stack.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The AWS Lambda that is deployed through this construct has broad IAM permissions to attach any Elastic
 *    Network Interface (ENI) to any instance. You should not grant any additional actors/principals the ability
 *    to modify or execute this Lambda.
 * - The SNS Topic that is deployed through this construct controls the execution of the Lambda discussed above.
 *    Principals that can publish messages to this SNS Topic will be able to trigger the Lambda to run. You should
 *    not allow any additional principals to publish messages to this SNS Topic.
 */
class StaticPrivateIpServer extends core_1.Construct {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        const { subnets } = props.vpc.selectSubnets(props.vpcSubnets);
        if (subnets.length === 0) {
            throw new Error(`Did not find any subnets matching ${JSON.stringify(props.vpcSubnets)}. Please use a different selection.`);
        }
        const subnet = subnets[0];
        if (props.resourceSignalTimeout && props.resourceSignalTimeout.toSeconds() > (12 * 60 * 60)) {
            throw new Error('Resource signal timeout cannot exceed 12 hours.');
        }
        this.autoscalingGroup = new aws_autoscaling_1.AutoScalingGroup(this, 'Asg', {
            minCapacity: 1,
            maxCapacity: 1,
            vpc: props.vpc,
            instanceType: props.instanceType,
            machineImage: props.machineImage,
            vpcSubnets: { subnets: [subnet] },
            blockDevices: props.blockDevices,
            keyName: props.keyName,
            resourceSignalCount: props.resourceSignalTimeout ? 1 : undefined,
            resourceSignalTimeout: props.resourceSignalTimeout,
            role: props.role,
            securityGroup: props.securityGroup,
            userData: props.userData,
        });
        this.connections = this.autoscalingGroup.connections;
        this.grantPrincipal = this.autoscalingGroup.grantPrincipal;
        this.osType = this.autoscalingGroup.osType;
        this.role = this.autoscalingGroup.role;
        this.userData = this.autoscalingGroup.userData;
        const scopePath = this.node.scopes.map(construct => construct.node.id).slice(1); // Slice to remove the unnamed <root> scope.
        const eni = new aws_ec2_1.CfnNetworkInterface(this, 'Eni', {
            subnetId: subnet.subnetId,
            description: `Static ENI for ${scopePath.join('/')}`,
            groupSet: core_1.Lazy.listValue({ produce: () => this.connections.securityGroups.map(sg => sg.securityGroupId) }),
            privateIpAddress: props.privateIpAddress,
        });
        this.privateIpAddress = eni.attrPrimaryPrivateIpAddress;
        // We need to be sure that the ENI is created before the instance would be brought up; otherwise, we cannot attach it.
        this.autoscalingGroup.node.defaultChild.addDependsOn(eni);
        this.attachEniLifecyleTarget(eni);
        this.node.defaultChild = this.autoscalingGroup.node.defaultChild;
    }
    /**
     * Set up an instance launch lifecycle action that will attach the eni to the single instance in this construct's AutoScalingGroup when a new instance is launched.
     */
    attachEniLifecyleTarget(eni) {
        // Note: The design of AutoScalingGroup life cycle notifications in CDK v1.49.1 is such that
        // using the provided AutoScalingGroup.addLifecycleHook() will result in a setup that misses
        // launch notifications for instances created when the ASG is created. This is because
        // it uses the separate CfnLifecycleHook resource to do it, and that resource references the
        // ASG ARN; i.e. it must be created after the ASG has an ARN... thus it can miss instance launches
        // when the ASG is first created.
        //
        // We work around this by using an escape-hatch to the L1 ASG to create our own notification from scratch.
        const eventHandler = this.setupLifecycleEventHandlerFunction();
        const { topic, role } = this.setupLifecycleNotificationTopic(eventHandler);
        // Ensure no race conditions that might prevent the lambda from being able to perform its required functions by making
        // the ASG depend on the creation of the SNS Subscription.
        // Note: The topic subscriptions are children of the lambda, and are given an id equal to the Topic's id.
        this.autoscalingGroup.node.defaultChild.node.addDependency(eventHandler.node.findChild(topic.node.id));
        this.autoscalingGroup.node.defaultChild.lifecycleHookSpecificationList = [
            {
                defaultResult: aws_autoscaling_1.DefaultResult.ABANDON,
                heartbeatTimeout: 120,
                lifecycleHookName: 'NewStaticPrivateIpServer',
                lifecycleTransition: aws_autoscaling_1.LifecycleTransition.INSTANCE_LAUNCHING,
                notificationTargetArn: topic.topicArn,
                roleArn: role.roleArn,
                notificationMetadata: JSON.stringify({ eniId: eni.ref }),
            },
        ];
    }
    /**
     * Create, or fetch, the lambda function that will process instance-start lifecycle events from this construct.
     */
    setupLifecycleEventHandlerFunction() {
        const stack = core_1.Stack.of(this);
        // The SingletonFunction does not tell us when it's newly created vs. finding a pre-existing
        // one. So, we do our own singleton Function so that we know when it's the first creation, and, thus,
        // we must attach one-time permissions.
        const functionUniqueId = 'AttachEniToInstance' + this.removeHyphens('83a5dca5-db54-4aa4-85d2-8d419cdf85ce');
        let singletonPreExists = true;
        let eventHandler = stack.node.tryFindChild(functionUniqueId);
        if (!eventHandler) {
            const handlerCode = aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', 'lambdas', 'nodejs', 'asg-attach-eni'), {
                exclude: ['**/*', '!index*'],
            });
            eventHandler = new aws_lambda_1.Function(stack, functionUniqueId, {
                code: handlerCode,
                handler: 'index.handler',
                runtime: aws_lambda_1.Runtime.NODEJS_12_X,
                description: `Created by RFDK StaticPrivateIpServer to process instance launch lifecycle events in stack '${stack.stackName}'. This lambda attaches an ENI to newly launched instances.`,
                logRetention: aws_logs_1.RetentionDays.THREE_DAYS,
            });
            singletonPreExists = false;
        }
        // Note: We **cannot** reference the ASG's ARN in the lambda's policy. It would create a deadlock at deployment:
        //  Lambda policy waiting on ASG completion to get ARN
        //  -> lambda waiting on policy to be created
        //  -> ASG waiting on lambda to signal lifecycle continue for instance start
        //  -> back to the start of the cycle.
        // Instead we use resourcetags condition to limit the scope of the lambda.
        const tagKey = 'RfdkStaticPrivateIpServerGrantConditionKey';
        const tagValue = eventHandler.node.uniqueId;
        const grantCondition = {};
        grantCondition[`autoscaling:ResourceTag/${tagKey}`] = tagValue;
        core_1.Tags.of(this.autoscalingGroup).add(tagKey, tagValue);
        // Allow the lambda to complete the lifecycle action for only tagged ASGs.
        const iamCompleteLifecycle = new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'autoscaling:CompleteLifecycleAction',
            ],
            resources: [
                `arn:${stack.partition}:autoscaling:${stack.region}:${stack.account}:autoScalingGroup:*:autoScalingGroupName/*`,
            ],
            conditions: {
                'ForAnyValue:StringEquals': grantCondition,
            },
        });
        eventHandler.role.addToPolicy(iamCompleteLifecycle);
        if (!singletonPreExists) {
            // Allow the lambda to attach the ENI to the instance that was created.
            // Referencing: https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2.html
            // Last-Accessed: July 2020
            // The ec2:DescribeNetworkInterfaces, and ec2:AttachNetworkInterface operations
            // do not support conditions, and do not support resource restriction.
            // So, we only attach the policy to the lambda function once; when we first create it.
            const iamEniAttach = new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: [
                    'ec2:DescribeNetworkInterfaces',
                    'ec2:AttachNetworkInterface',
                ],
                resources: ['*'],
            });
            eventHandler.role.addToPolicy(iamEniAttach);
        }
        return eventHandler;
    }
    /**
     * Create, or fetch, an SNS Topic to which we'll direct the ASG's instance-start lifecycle hook events.
     *
     * Also creates, or fetches,
     * the accompanying role that allows the lifecycle events to be published to the SNS Topic.
     *
     * @param lambdaHandler The lambda singleton that will be processing the lifecycle events.
     * @returns : Topic, role: Role }
     */
    setupLifecycleNotificationTopic(lambdaHandler) {
        const stack = core_1.Stack.of(this);
        // We only need to have a single SNS topic & subscription set up to handle lifecycle events for *all* instances of this class.
        // We have to be careful, however, to ensure that our initial setup only happens once when we first add the topic and such
        // to this stack; otherwise, we will not be able to deploy more than one of these constructs in a stack.
        const notificationRoleUniqueId = 'AttachEniNotificationRole' + this.removeHyphens('a0376ff8-248e-4534-bf42-58c6ffa4d5b4');
        const notificationTopicUniqueId = 'AttachEniNotificationTopic' + this.removeHyphens('c8b1e9a6-783c-4954-b191-204dd5e3b9e0');
        let notificationTopic = stack.node.tryFindChild(notificationTopicUniqueId);
        let notificationRole;
        if (!notificationTopic) {
            // First time creating the singleton Topic in this stack. Set it all up...
            notificationRole = new aws_iam_1.Role(stack, notificationRoleUniqueId, {
                assumedBy: new aws_iam_1.ServicePrincipal('autoscaling.amazonaws.com'),
            });
            notificationTopic = new aws_sns_1.Topic(stack, notificationTopicUniqueId, {
                displayName: `For RFDK instance-launch notifications for stack '${stack.stackName}'`,
            });
            notificationTopic.addSubscription(new aws_sns_subscriptions_1.LambdaSubscription(lambdaHandler));
            notificationTopic.grantPublish(notificationRole);
        }
        else {
            notificationRole = stack.node.findChild(notificationRoleUniqueId);
        }
        return {
            topic: notificationTopic,
            role: notificationRole,
        };
    }
    /**
     * Convert a UUID into a string that's usable in a construct id.
     */
    removeHyphens(x) {
        return x.replace(/[-]/g, '');
    }
}
exports.StaticPrivateIpServer = StaticPrivateIpServer;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3RhdGljaXAtc2VydmVyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsic3RhdGljaXAtc2VydmVyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQTs7O0dBR0c7OztBQUVILDZCQUE2QjtBQUM3Qiw4REFNa0M7QUFDbEMsOENBVzBCO0FBQzFCLDhDQVEwQjtBQUMxQixvREFJNkI7QUFDN0IsZ0RBRTJCO0FBQzNCLDhDQUUwQjtBQUMxQiwwRUFFd0M7QUFDeEMsd0NBT3VCOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7QUE4SHZCLE1BQWEscUJBQXNCLFNBQVEsZ0JBQVM7Ozs7SUF1Q2xELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBaUM7UUFDekUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixNQUFNLEVBQUUsT0FBTyxFQUFFLEdBQUcsS0FBSyxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQzlELElBQUksT0FBTyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDeEIsTUFBTSxJQUFJLEtBQUssQ0FBQyxxQ0FBcUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLHFDQUFxQyxDQUFDLENBQUM7U0FDN0g7UUFDRCxNQUFNLE1BQU0sR0FBRyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFFMUIsSUFBSSxLQUFLLENBQUMscUJBQXFCLElBQUksS0FBSyxDQUFDLHFCQUFxQixDQUFDLFNBQVMsRUFBRSxHQUFHLENBQUMsRUFBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLENBQUMsRUFBRTtZQUMzRixNQUFNLElBQUksS0FBSyxDQUFDLGlEQUFpRCxDQUFDLENBQUM7U0FDcEU7UUFFRCxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsSUFBSSxrQ0FBZ0IsQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFO1lBQ3hELFdBQVcsRUFBRSxDQUFDO1lBQ2QsV0FBVyxFQUFFLENBQUM7WUFDZCxHQUFHLEVBQUUsS0FBSyxDQUFDLEdBQUc7WUFDZCxZQUFZLEVBQUUsS0FBSyxDQUFDLFlBQVk7WUFDaEMsWUFBWSxFQUFFLEtBQUssQ0FBQyxZQUFZO1lBQ2hDLFVBQVUsRUFBRSxFQUFFLE9BQU8sRUFBRSxDQUFDLE1BQU0sQ0FBQyxFQUFFO1lBQ2pDLFlBQVksRUFBRSxLQUFLLENBQUMsWUFBWTtZQUNoQyxPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU87WUFDdEIsbUJBQW1CLEVBQUUsS0FBSyxDQUFDLHFCQUFxQixDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDaEUscUJBQXFCLEVBQUUsS0FBSyxDQUFDLHFCQUFxQjtZQUNsRCxJQUFJLEVBQUUsS0FBSyxDQUFDLElBQUk7WUFDaEIsYUFBYSxFQUFFLEtBQUssQ0FBQyxhQUFhO1lBQ2xDLFFBQVEsRUFBRSxLQUFLLENBQUMsUUFBUTtTQUN6QixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsV0FBVyxHQUFHLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxXQUFXLENBQUM7UUFDckQsSUFBSSxDQUFDLGNBQWMsR0FBRyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsY0FBYyxDQUFDO1FBQzNELElBQUksQ0FBQyxNQUFNLEdBQUcsSUFBSSxDQUFDLGdCQUFnQixDQUFDLE1BQU0sQ0FBQztRQUMzQyxJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUM7UUFDdkMsSUFBSSxDQUFDLFFBQVEsR0FBRyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsUUFBUSxDQUFDO1FBRS9DLE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxTQUFTLENBQUMsRUFBRSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsNENBQTRDO1FBQzdILE1BQU0sR0FBRyxHQUFHLElBQUksNkJBQW1CLENBQUMsSUFBSSxFQUFFLEtBQUssRUFBRTtZQUMvQyxRQUFRLEVBQUUsTUFBTSxDQUFDLFFBQVE7WUFDekIsV0FBVyxFQUFFLGtCQUFrQixTQUFTLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFO1lBQ3BELFFBQVEsRUFBRSxXQUFJLENBQUMsU0FBUyxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsRUFBRSxDQUFDO1lBQzFHLGdCQUFnQixFQUFFLEtBQUssQ0FBQyxnQkFBZ0I7U0FDekMsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLGdCQUFnQixHQUFHLEdBQUcsQ0FBQywyQkFBMkIsQ0FBQztRQUV4RCxzSEFBc0g7UUFDckgsSUFBSSxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxZQUE0QixDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUUzRSxJQUFJLENBQUMsdUJBQXVCLENBQUMsR0FBRyxDQUFDLENBQUM7UUFFbEMsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLEdBQUcsSUFBSSxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxZQUFZLENBQUM7SUFDbkUsQ0FBQzs7OztJQU1TLHVCQUF1QixDQUFDLEdBQXdCO1FBQ3hELDRGQUE0RjtRQUM1Riw0RkFBNEY7UUFDNUYsc0ZBQXNGO1FBQ3RGLDRGQUE0RjtRQUM1RixrR0FBa0c7UUFDbEcsaUNBQWlDO1FBQ2pDLEVBQUU7UUFDRiwwR0FBMEc7UUFFMUcsTUFBTSxZQUFZLEdBQUcsSUFBSSxDQUFDLGtDQUFrQyxFQUFFLENBQUM7UUFDL0QsTUFBTSxFQUFFLEtBQUssRUFBRSxJQUFJLEVBQUUsR0FBRyxJQUFJLENBQUMsK0JBQStCLENBQUMsWUFBWSxDQUFDLENBQUM7UUFFM0Usc0hBQXNIO1FBQ3RILDBEQUEwRDtRQUMxRCx5R0FBeUc7UUFDekcsSUFBSSxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxZQUFhLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFFdkcsSUFBSSxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxZQUFvQyxDQUFDLDhCQUE4QixHQUFHO1lBQ2hHO2dCQUNFLGFBQWEsRUFBRSwrQkFBYSxDQUFDLE9BQU87Z0JBQ3BDLGdCQUFnQixFQUFFLEdBQUc7Z0JBQ3JCLGlCQUFpQixFQUFFLDBCQUEwQjtnQkFDN0MsbUJBQW1CLEVBQUUscUNBQW1CLENBQUMsa0JBQWtCO2dCQUMzRCxxQkFBcUIsRUFBRSxLQUFLLENBQUMsUUFBUTtnQkFDckMsT0FBTyxFQUFFLElBQUksQ0FBQyxPQUFPO2dCQUNyQixvQkFBb0IsRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLEVBQUUsS0FBSyxFQUFFLEdBQUcsQ0FBQyxHQUFHLEVBQUUsQ0FBQzthQUN6RDtTQUNGLENBQUM7SUFDSixDQUFDOzs7O0lBS1Msa0NBQWtDO1FBQzFDLE1BQU0sS0FBSyxHQUFHLFlBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFN0IsNEZBQTRGO1FBQzVGLHFHQUFxRztRQUNyRyx1Q0FBdUM7UUFDdkMsTUFBTSxnQkFBZ0IsR0FBRyxxQkFBcUIsR0FBRyxJQUFJLENBQUMsYUFBYSxDQUFDLHNDQUFzQyxDQUFDLENBQUM7UUFDNUcsSUFBSSxrQkFBa0IsR0FBWSxJQUFJLENBQUM7UUFDdkMsSUFBSSxZQUFZLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsZ0JBQWdCLENBQW1CLENBQUM7UUFDL0UsSUFBSSxDQUFDLFlBQVksRUFBRTtZQUNqQixNQUFNLFdBQVcsR0FBRyxpQkFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxJQUFJLEVBQUUsU0FBUyxFQUFFLFFBQVEsRUFBRSxnQkFBZ0IsQ0FBQyxFQUFFO2dCQUNwRyxPQUFPLEVBQUUsQ0FBQyxNQUFNLEVBQUUsU0FBUyxDQUFDO2FBQzdCLENBQUMsQ0FBQztZQUNILFlBQVksR0FBRyxJQUFJLHFCQUFjLENBQUMsS0FBSyxFQUFFLGdCQUFnQixFQUFFO2dCQUN6RCxJQUFJLEVBQUUsV0FBVztnQkFDakIsT0FBTyxFQUFFLGVBQWU7Z0JBQ3hCLE9BQU8sRUFBRSxvQkFBTyxDQUFDLFdBQVc7Z0JBQzVCLFdBQVcsRUFBRSwrRkFBK0YsS0FBSyxDQUFDLFNBQVMsNkRBQTZEO2dCQUN4TCxZQUFZLEVBQUUsd0JBQWEsQ0FBQyxVQUFVO2FBQ3ZDLENBQUMsQ0FBQztZQUNILGtCQUFrQixHQUFHLEtBQUssQ0FBQztTQUM1QjtRQUVELGdIQUFnSDtRQUNoSCxzREFBc0Q7UUFDdEQsNkNBQTZDO1FBQzdDLDRFQUE0RTtRQUM1RSxzQ0FBc0M7UUFDdEMsMEVBQTBFO1FBQzFFLE1BQU0sTUFBTSxHQUFHLDRDQUE0QyxDQUFDO1FBQzVELE1BQU0sUUFBUSxHQUFHLFlBQVksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDO1FBQzVDLE1BQU0sY0FBYyxHQUE4QixFQUFFLENBQUM7UUFDckQsY0FBYyxDQUFDLDJCQUEyQixNQUFNLEVBQUUsQ0FBQyxHQUFHLFFBQVEsQ0FBQztRQUMvRCxXQUFJLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxNQUFNLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFFckQsMEVBQTBFO1FBQzFFLE1BQU0sb0JBQW9CLEdBQUcsSUFBSSx5QkFBZSxDQUFDO1lBQy9DLE1BQU0sRUFBRSxnQkFBTSxDQUFDLEtBQUs7WUFDcEIsT0FBTyxFQUFFO2dCQUNQLHFDQUFxQzthQUN0QztZQUNELFNBQVMsRUFBRTtnQkFDVCxPQUFPLEtBQUssQ0FBQyxTQUFTLGdCQUFnQixLQUFLLENBQUMsTUFBTSxJQUFJLEtBQUssQ0FBQyxPQUFPLDRDQUE0QzthQUNoSDtZQUNELFVBQVUsRUFBRTtnQkFDViwwQkFBMEIsRUFBRSxjQUFjO2FBQzNDO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsWUFBWSxDQUFDLElBQUssQ0FBQyxXQUFXLENBQUMsb0JBQW9CLENBQUMsQ0FBQztRQUVyRCxJQUFJLENBQUMsa0JBQWtCLEVBQUU7WUFDdkIsdUVBQXVFO1lBQ3ZFLG9GQUFvRjtZQUNwRiwyQkFBMkI7WUFDM0IsK0VBQStFO1lBQy9FLHNFQUFzRTtZQUN0RSxzRkFBc0Y7WUFDdEYsTUFBTSxZQUFZLEdBQUcsSUFBSSx5QkFBZSxDQUFDO2dCQUN2QyxNQUFNLEVBQUUsZ0JBQU0sQ0FBQyxLQUFLO2dCQUNwQixPQUFPLEVBQUU7b0JBQ1AsK0JBQStCO29CQUMvQiw0QkFBNEI7aUJBQzdCO2dCQUNELFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQzthQUNqQixDQUFDLENBQUM7WUFDSCxZQUFZLENBQUMsSUFBSyxDQUFDLFdBQVcsQ0FBQyxZQUFZLENBQUMsQ0FBQztTQUM5QztRQUVELE9BQU8sWUFBWSxDQUFDO0lBQ3RCLENBQUM7Ozs7Ozs7Ozs7SUFRUywrQkFBK0IsQ0FBQyxhQUE2QjtRQUNyRSxNQUFNLEtBQUssR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzdCLDhIQUE4SDtRQUM5SCwwSEFBMEg7UUFDMUgsd0dBQXdHO1FBRXhHLE1BQU0sd0JBQXdCLEdBQUcsMkJBQTJCLEdBQUcsSUFBSSxDQUFDLGFBQWEsQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDO1FBQzFILE1BQU0seUJBQXlCLEdBQUcsNEJBQTRCLEdBQUcsSUFBSSxDQUFDLGFBQWEsQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDO1FBQzVILElBQUksaUJBQWlCLEdBQVcsS0FBSyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMseUJBQXlCLENBQVcsQ0FBQztRQUM3RixJQUFJLGdCQUFzQixDQUFDO1FBQzNCLElBQUksQ0FBQyxpQkFBaUIsRUFBRTtZQUN0QiwwRUFBMEU7WUFFMUUsZ0JBQWdCLEdBQUcsSUFBSSxjQUFJLENBQUMsS0FBSyxFQUFFLHdCQUF3QixFQUFFO2dCQUMzRCxTQUFTLEVBQUUsSUFBSSwwQkFBZ0IsQ0FBQywyQkFBMkIsQ0FBQzthQUM3RCxDQUFDLENBQUM7WUFFSCxpQkFBaUIsR0FBRyxJQUFJLGVBQUssQ0FBQyxLQUFLLEVBQUUseUJBQXlCLEVBQUU7Z0JBQzlELFdBQVcsRUFBRSxxREFBcUQsS0FBSyxDQUFDLFNBQVMsR0FBRzthQUNyRixDQUFDLENBQUM7WUFFSCxpQkFBaUIsQ0FBQyxlQUFlLENBQUMsSUFBSSwwQ0FBa0IsQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDO1lBQ3pFLGlCQUFpQixDQUFDLFlBQVksQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1NBQ2xEO2FBQU07WUFDTCxnQkFBZ0IsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyx3QkFBd0IsQ0FBUyxDQUFDO1NBQzNFO1FBRUQsT0FBTztZQUNMLEtBQUssRUFBRSxpQkFBaUI7WUFDeEIsSUFBSSxFQUFFLGdCQUFnQjtTQUN2QixDQUFDO0lBQ0osQ0FBQztJQUVEOztPQUVHO0lBQ0ssYUFBYSxDQUFDLENBQVM7UUFDN0IsT0FBTyxDQUFDLENBQUMsT0FBTyxDQUFDLE1BQU0sRUFBRSxFQUFFLENBQUMsQ0FBQztJQUMvQixDQUFDO0NBQ0Y7QUFwUEQsc0RBb1BDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBDb3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4gKi9cblxuaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCB7XG4gIEF1dG9TY2FsaW5nR3JvdXAsXG4gIEJsb2NrRGV2aWNlLFxuICBDZm5BdXRvU2NhbGluZ0dyb3VwLFxuICBEZWZhdWx0UmVzdWx0LFxuICBMaWZlY3ljbGVUcmFuc2l0aW9uLFxufSBmcm9tICdAYXdzLWNkay9hd3MtYXV0b3NjYWxpbmcnO1xuaW1wb3J0IHtcbiAgQ2ZuTmV0d29ya0ludGVyZmFjZSxcbiAgQ29ubmVjdGlvbnMsXG4gIElDb25uZWN0YWJsZSxcbiAgSU1hY2hpbmVJbWFnZSxcbiAgSW5zdGFuY2VUeXBlLFxuICBJU2VjdXJpdHlHcm91cCxcbiAgSVZwYyxcbiAgT3BlcmF0aW5nU3lzdGVtVHlwZSxcbiAgU3VibmV0U2VsZWN0aW9uLFxuICBVc2VyRGF0YSxcbn0gZnJvbSAnQGF3cy1jZGsvYXdzLWVjMic7XG5pbXBvcnQge1xuICBFZmZlY3QsXG4gIElHcmFudGFibGUsXG4gIElQcmluY2lwYWwsXG4gIElSb2xlLFxuICBQb2xpY3lTdGF0ZW1lbnQsXG4gIFJvbGUsXG4gIFNlcnZpY2VQcmluY2lwYWwsXG59IGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0IHtcbiAgQ29kZSxcbiAgRnVuY3Rpb24gYXMgTGFtYmRhRnVuY3Rpb24sXG4gIFJ1bnRpbWUsXG59IGZyb20gJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnO1xuaW1wb3J0IHtcbiAgUmV0ZW50aW9uRGF5cyxcbn0gZnJvbSAnQGF3cy1jZGsvYXdzLWxvZ3MnO1xuaW1wb3J0IHtcbiAgVG9waWMsXG59IGZyb20gJ0Bhd3MtY2RrL2F3cy1zbnMnO1xuaW1wb3J0IHtcbiAgTGFtYmRhU3Vic2NyaXB0aW9uLFxufSBmcm9tICdAYXdzLWNkay9hd3Mtc25zLXN1YnNjcmlwdGlvbnMnO1xuaW1wb3J0IHtcbiAgQ2ZuUmVzb3VyY2UsXG4gIENvbnN0cnVjdCxcbiAgRHVyYXRpb24sXG4gIExhenksXG4gIFN0YWNrLFxuICBUYWdzLFxufSBmcm9tICdAYXdzLWNkay9jb3JlJztcblxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBTdGF0aWNQcml2YXRlSXBTZXJ2ZXJQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB2cGM6IElWcGM7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGluc3RhbmNlVHlwZTogSW5zdGFuY2VUeXBlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBtYWNoaW5lSW1hZ2U6IElNYWNoaW5lSW1hZ2U7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGJsb2NrRGV2aWNlcz86IEJsb2NrRGV2aWNlW107XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGtleU5hbWU/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBwcml2YXRlSXBBZGRyZXNzPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcmVzb3VyY2VTaWduYWxUaW1lb3V0PzogRHVyYXRpb247XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcm9sZT86IElSb2xlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHNlY3VyaXR5R3JvdXA/OiBJU2VjdXJpdHlHcm91cDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdXNlckRhdGE/OiBVc2VyRGF0YTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB2cGNTdWJuZXRzPzogU3VibmV0U2VsZWN0aW9uO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIFN0YXRpY1ByaXZhdGVJcFNlcnZlciBleHRlbmRzIENvbnN0cnVjdCBpbXBsZW1lbnRzIElDb25uZWN0YWJsZSwgSUdyYW50YWJsZSB7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgYXV0b3NjYWxpbmdHcm91cDogQXV0b1NjYWxpbmdHcm91cDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBjb25uZWN0aW9uczogQ29ubmVjdGlvbnM7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBncmFudFByaW5jaXBhbDogSVByaW5jaXBhbDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBvc1R5cGU6IE9wZXJhdGluZ1N5c3RlbVR5cGU7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBwcml2YXRlSXBBZGRyZXNzOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSByb2xlOiBJUm9sZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSB1c2VyRGF0YTogVXNlckRhdGE7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFN0YXRpY1ByaXZhdGVJcFNlcnZlclByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGNvbnN0IHsgc3VibmV0cyB9ID0gcHJvcHMudnBjLnNlbGVjdFN1Ym5ldHMocHJvcHMudnBjU3VibmV0cyk7XG4gICAgaWYgKHN1Ym5ldHMubGVuZ3RoID09PSAwKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYERpZCBub3QgZmluZCBhbnkgc3VibmV0cyBtYXRjaGluZyAke0pTT04uc3RyaW5naWZ5KHByb3BzLnZwY1N1Ym5ldHMpfS4gUGxlYXNlIHVzZSBhIGRpZmZlcmVudCBzZWxlY3Rpb24uYCk7XG4gICAgfVxuICAgIGNvbnN0IHN1Ym5ldCA9IHN1Ym5ldHNbMF07XG5cbiAgICBpZiAocHJvcHMucmVzb3VyY2VTaWduYWxUaW1lb3V0ICYmIHByb3BzLnJlc291cmNlU2lnbmFsVGltZW91dC50b1NlY29uZHMoKSA+ICgxMiAqIDYwICogNjApKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1Jlc291cmNlIHNpZ25hbCB0aW1lb3V0IGNhbm5vdCBleGNlZWQgMTIgaG91cnMuJyk7XG4gICAgfVxuXG4gICAgdGhpcy5hdXRvc2NhbGluZ0dyb3VwID0gbmV3IEF1dG9TY2FsaW5nR3JvdXAodGhpcywgJ0FzZycsIHtcbiAgICAgIG1pbkNhcGFjaXR5OiAxLFxuICAgICAgbWF4Q2FwYWNpdHk6IDEsXG4gICAgICB2cGM6IHByb3BzLnZwYyxcbiAgICAgIGluc3RhbmNlVHlwZTogcHJvcHMuaW5zdGFuY2VUeXBlLFxuICAgICAgbWFjaGluZUltYWdlOiBwcm9wcy5tYWNoaW5lSW1hZ2UsXG4gICAgICB2cGNTdWJuZXRzOiB7IHN1Ym5ldHM6IFtzdWJuZXRdIH0sXG4gICAgICBibG9ja0RldmljZXM6IHByb3BzLmJsb2NrRGV2aWNlcyxcbiAgICAgIGtleU5hbWU6IHByb3BzLmtleU5hbWUsXG4gICAgICByZXNvdXJjZVNpZ25hbENvdW50OiBwcm9wcy5yZXNvdXJjZVNpZ25hbFRpbWVvdXQgPyAxIDogdW5kZWZpbmVkLFxuICAgICAgcmVzb3VyY2VTaWduYWxUaW1lb3V0OiBwcm9wcy5yZXNvdXJjZVNpZ25hbFRpbWVvdXQsXG4gICAgICByb2xlOiBwcm9wcy5yb2xlLFxuICAgICAgc2VjdXJpdHlHcm91cDogcHJvcHMuc2VjdXJpdHlHcm91cCxcbiAgICAgIHVzZXJEYXRhOiBwcm9wcy51c2VyRGF0YSxcbiAgICB9KTtcbiAgICB0aGlzLmNvbm5lY3Rpb25zID0gdGhpcy5hdXRvc2NhbGluZ0dyb3VwLmNvbm5lY3Rpb25zO1xuICAgIHRoaXMuZ3JhbnRQcmluY2lwYWwgPSB0aGlzLmF1dG9zY2FsaW5nR3JvdXAuZ3JhbnRQcmluY2lwYWw7XG4gICAgdGhpcy5vc1R5cGUgPSB0aGlzLmF1dG9zY2FsaW5nR3JvdXAub3NUeXBlO1xuICAgIHRoaXMucm9sZSA9IHRoaXMuYXV0b3NjYWxpbmdHcm91cC5yb2xlO1xuICAgIHRoaXMudXNlckRhdGEgPSB0aGlzLmF1dG9zY2FsaW5nR3JvdXAudXNlckRhdGE7XG5cbiAgICBjb25zdCBzY29wZVBhdGggPSB0aGlzLm5vZGUuc2NvcGVzLm1hcChjb25zdHJ1Y3QgPT4gY29uc3RydWN0Lm5vZGUuaWQpLnNsaWNlKDEpOyAvLyBTbGljZSB0byByZW1vdmUgdGhlIHVubmFtZWQgPHJvb3Q+IHNjb3BlLlxuICAgIGNvbnN0IGVuaSA9IG5ldyBDZm5OZXR3b3JrSW50ZXJmYWNlKHRoaXMsICdFbmknLCB7XG4gICAgICBzdWJuZXRJZDogc3VibmV0LnN1Ym5ldElkLFxuICAgICAgZGVzY3JpcHRpb246IGBTdGF0aWMgRU5JIGZvciAke3Njb3BlUGF0aC5qb2luKCcvJyl9YCxcbiAgICAgIGdyb3VwU2V0OiBMYXp5Lmxpc3RWYWx1ZSh7IHByb2R1Y2U6ICgpID0+IHRoaXMuY29ubmVjdGlvbnMuc2VjdXJpdHlHcm91cHMubWFwKHNnID0+IHNnLnNlY3VyaXR5R3JvdXBJZCkgfSksXG4gICAgICBwcml2YXRlSXBBZGRyZXNzOiBwcm9wcy5wcml2YXRlSXBBZGRyZXNzLFxuICAgIH0pO1xuICAgIHRoaXMucHJpdmF0ZUlwQWRkcmVzcyA9IGVuaS5hdHRyUHJpbWFyeVByaXZhdGVJcEFkZHJlc3M7XG5cbiAgICAvLyBXZSBuZWVkIHRvIGJlIHN1cmUgdGhhdCB0aGUgRU5JIGlzIGNyZWF0ZWQgYmVmb3JlIHRoZSBpbnN0YW5jZSB3b3VsZCBiZSBicm91Z2h0IHVwOyBvdGhlcndpc2UsIHdlIGNhbm5vdCBhdHRhY2ggaXQuXG4gICAgKHRoaXMuYXV0b3NjYWxpbmdHcm91cC5ub2RlLmRlZmF1bHRDaGlsZCBhcyBDZm5SZXNvdXJjZSkuYWRkRGVwZW5kc09uKGVuaSk7XG5cbiAgICB0aGlzLmF0dGFjaEVuaUxpZmVjeWxlVGFyZ2V0KGVuaSk7XG5cbiAgICB0aGlzLm5vZGUuZGVmYXVsdENoaWxkID0gdGhpcy5hdXRvc2NhbGluZ0dyb3VwLm5vZGUuZGVmYXVsdENoaWxkO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHByb3RlY3RlZCBhdHRhY2hFbmlMaWZlY3lsZVRhcmdldChlbmk6IENmbk5ldHdvcmtJbnRlcmZhY2UpIHtcbiAgICAvLyBOb3RlOiBUaGUgZGVzaWduIG9mIEF1dG9TY2FsaW5nR3JvdXAgbGlmZSBjeWNsZSBub3RpZmljYXRpb25zIGluIENESyB2MS40OS4xIGlzIHN1Y2ggdGhhdFxuICAgIC8vIHVzaW5nIHRoZSBwcm92aWRlZCBBdXRvU2NhbGluZ0dyb3VwLmFkZExpZmVjeWNsZUhvb2soKSB3aWxsIHJlc3VsdCBpbiBhIHNldHVwIHRoYXQgbWlzc2VzXG4gICAgLy8gbGF1bmNoIG5vdGlmaWNhdGlvbnMgZm9yIGluc3RhbmNlcyBjcmVhdGVkIHdoZW4gdGhlIEFTRyBpcyBjcmVhdGVkLiBUaGlzIGlzIGJlY2F1c2VcbiAgICAvLyBpdCB1c2VzIHRoZSBzZXBhcmF0ZSBDZm5MaWZlY3ljbGVIb29rIHJlc291cmNlIHRvIGRvIGl0LCBhbmQgdGhhdCByZXNvdXJjZSByZWZlcmVuY2VzIHRoZVxuICAgIC8vIEFTRyBBUk47IGkuZS4gaXQgbXVzdCBiZSBjcmVhdGVkIGFmdGVyIHRoZSBBU0cgaGFzIGFuIEFSTi4uLiB0aHVzIGl0IGNhbiBtaXNzIGluc3RhbmNlIGxhdW5jaGVzXG4gICAgLy8gd2hlbiB0aGUgQVNHIGlzIGZpcnN0IGNyZWF0ZWQuXG4gICAgLy9cbiAgICAvLyBXZSB3b3JrIGFyb3VuZCB0aGlzIGJ5IHVzaW5nIGFuIGVzY2FwZS1oYXRjaCB0byB0aGUgTDEgQVNHIHRvIGNyZWF0ZSBvdXIgb3duIG5vdGlmaWNhdGlvbiBmcm9tIHNjcmF0Y2guXG5cbiAgICBjb25zdCBldmVudEhhbmRsZXIgPSB0aGlzLnNldHVwTGlmZWN5Y2xlRXZlbnRIYW5kbGVyRnVuY3Rpb24oKTtcbiAgICBjb25zdCB7IHRvcGljLCByb2xlIH0gPSB0aGlzLnNldHVwTGlmZWN5Y2xlTm90aWZpY2F0aW9uVG9waWMoZXZlbnRIYW5kbGVyKTtcblxuICAgIC8vIEVuc3VyZSBubyByYWNlIGNvbmRpdGlvbnMgdGhhdCBtaWdodCBwcmV2ZW50IHRoZSBsYW1iZGEgZnJvbSBiZWluZyBhYmxlIHRvIHBlcmZvcm0gaXRzIHJlcXVpcmVkIGZ1bmN0aW9ucyBieSBtYWtpbmdcbiAgICAvLyB0aGUgQVNHIGRlcGVuZCBvbiB0aGUgY3JlYXRpb24gb2YgdGhlIFNOUyBTdWJzY3JpcHRpb24uXG4gICAgLy8gTm90ZTogVGhlIHRvcGljIHN1YnNjcmlwdGlvbnMgYXJlIGNoaWxkcmVuIG9mIHRoZSBsYW1iZGEsIGFuZCBhcmUgZ2l2ZW4gYW4gaWQgZXF1YWwgdG8gdGhlIFRvcGljJ3MgaWQuXG4gICAgdGhpcy5hdXRvc2NhbGluZ0dyb3VwLm5vZGUuZGVmYXVsdENoaWxkIS5ub2RlLmFkZERlcGVuZGVuY3koZXZlbnRIYW5kbGVyLm5vZGUuZmluZENoaWxkKHRvcGljLm5vZGUuaWQpKTtcblxuICAgICh0aGlzLmF1dG9zY2FsaW5nR3JvdXAubm9kZS5kZWZhdWx0Q2hpbGQgYXMgQ2ZuQXV0b1NjYWxpbmdHcm91cCkubGlmZWN5Y2xlSG9va1NwZWNpZmljYXRpb25MaXN0ID0gW1xuICAgICAge1xuICAgICAgICBkZWZhdWx0UmVzdWx0OiBEZWZhdWx0UmVzdWx0LkFCQU5ET04sXG4gICAgICAgIGhlYXJ0YmVhdFRpbWVvdXQ6IDEyMCxcbiAgICAgICAgbGlmZWN5Y2xlSG9va05hbWU6ICdOZXdTdGF0aWNQcml2YXRlSXBTZXJ2ZXInLFxuICAgICAgICBsaWZlY3ljbGVUcmFuc2l0aW9uOiBMaWZlY3ljbGVUcmFuc2l0aW9uLklOU1RBTkNFX0xBVU5DSElORyxcbiAgICAgICAgbm90aWZpY2F0aW9uVGFyZ2V0QXJuOiB0b3BpYy50b3BpY0FybixcbiAgICAgICAgcm9sZUFybjogcm9sZS5yb2xlQXJuLFxuICAgICAgICBub3RpZmljYXRpb25NZXRhZGF0YTogSlNPTi5zdHJpbmdpZnkoeyBlbmlJZDogZW5pLnJlZiB9KSxcbiAgICAgIH0sXG4gICAgXTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwcm90ZWN0ZWQgc2V0dXBMaWZlY3ljbGVFdmVudEhhbmRsZXJGdW5jdGlvbigpOiBMYW1iZGFGdW5jdGlvbiB7XG4gICAgY29uc3Qgc3RhY2sgPSBTdGFjay5vZih0aGlzKTtcblxuICAgIC8vIFRoZSBTaW5nbGV0b25GdW5jdGlvbiBkb2VzIG5vdCB0ZWxsIHVzIHdoZW4gaXQncyBuZXdseSBjcmVhdGVkIHZzLiBmaW5kaW5nIGEgcHJlLWV4aXN0aW5nXG4gICAgLy8gb25lLiBTbywgd2UgZG8gb3VyIG93biBzaW5nbGV0b24gRnVuY3Rpb24gc28gdGhhdCB3ZSBrbm93IHdoZW4gaXQncyB0aGUgZmlyc3QgY3JlYXRpb24sIGFuZCwgdGh1cyxcbiAgICAvLyB3ZSBtdXN0IGF0dGFjaCBvbmUtdGltZSBwZXJtaXNzaW9ucy5cbiAgICBjb25zdCBmdW5jdGlvblVuaXF1ZUlkID0gJ0F0dGFjaEVuaVRvSW5zdGFuY2UnICsgdGhpcy5yZW1vdmVIeXBoZW5zKCc4M2E1ZGNhNS1kYjU0LTRhYTQtODVkMi04ZDQxOWNkZjg1Y2UnKTtcbiAgICBsZXQgc2luZ2xldG9uUHJlRXhpc3RzOiBib29sZWFuID0gdHJ1ZTtcbiAgICBsZXQgZXZlbnRIYW5kbGVyID0gc3RhY2subm9kZS50cnlGaW5kQ2hpbGQoZnVuY3Rpb25VbmlxdWVJZCkgYXMgTGFtYmRhRnVuY3Rpb247XG4gICAgaWYgKCFldmVudEhhbmRsZXIpIHtcbiAgICAgIGNvbnN0IGhhbmRsZXJDb2RlID0gQ29kZS5mcm9tQXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJy4uJywgJ2xhbWJkYXMnLCAnbm9kZWpzJywgJ2FzZy1hdHRhY2gtZW5pJyksIHtcbiAgICAgICAgZXhjbHVkZTogWycqKi8qJywgJyFpbmRleConXSxcbiAgICAgIH0pO1xuICAgICAgZXZlbnRIYW5kbGVyID0gbmV3IExhbWJkYUZ1bmN0aW9uKHN0YWNrLCBmdW5jdGlvblVuaXF1ZUlkLCB7XG4gICAgICAgIGNvZGU6IGhhbmRsZXJDb2RlLFxuICAgICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICAgIHJ1bnRpbWU6IFJ1bnRpbWUuTk9ERUpTXzEyX1gsXG4gICAgICAgIGRlc2NyaXB0aW9uOiBgQ3JlYXRlZCBieSBSRkRLIFN0YXRpY1ByaXZhdGVJcFNlcnZlciB0byBwcm9jZXNzIGluc3RhbmNlIGxhdW5jaCBsaWZlY3ljbGUgZXZlbnRzIGluIHN0YWNrICcke3N0YWNrLnN0YWNrTmFtZX0nLiBUaGlzIGxhbWJkYSBhdHRhY2hlcyBhbiBFTkkgdG8gbmV3bHkgbGF1bmNoZWQgaW5zdGFuY2VzLmAsXG4gICAgICAgIGxvZ1JldGVudGlvbjogUmV0ZW50aW9uRGF5cy5USFJFRV9EQVlTLFxuICAgICAgfSk7XG4gICAgICBzaW5nbGV0b25QcmVFeGlzdHMgPSBmYWxzZTtcbiAgICB9XG5cbiAgICAvLyBOb3RlOiBXZSAqKmNhbm5vdCoqIHJlZmVyZW5jZSB0aGUgQVNHJ3MgQVJOIGluIHRoZSBsYW1iZGEncyBwb2xpY3kuIEl0IHdvdWxkIGNyZWF0ZSBhIGRlYWRsb2NrIGF0IGRlcGxveW1lbnQ6XG4gICAgLy8gIExhbWJkYSBwb2xpY3kgd2FpdGluZyBvbiBBU0cgY29tcGxldGlvbiB0byBnZXQgQVJOXG4gICAgLy8gIC0+IGxhbWJkYSB3YWl0aW5nIG9uIHBvbGljeSB0byBiZSBjcmVhdGVkXG4gICAgLy8gIC0+IEFTRyB3YWl0aW5nIG9uIGxhbWJkYSB0byBzaWduYWwgbGlmZWN5Y2xlIGNvbnRpbnVlIGZvciBpbnN0YW5jZSBzdGFydFxuICAgIC8vICAtPiBiYWNrIHRvIHRoZSBzdGFydCBvZiB0aGUgY3ljbGUuXG4gICAgLy8gSW5zdGVhZCB3ZSB1c2UgcmVzb3VyY2V0YWdzIGNvbmRpdGlvbiB0byBsaW1pdCB0aGUgc2NvcGUgb2YgdGhlIGxhbWJkYS5cbiAgICBjb25zdCB0YWdLZXkgPSAnUmZka1N0YXRpY1ByaXZhdGVJcFNlcnZlckdyYW50Q29uZGl0aW9uS2V5JztcbiAgICBjb25zdCB0YWdWYWx1ZSA9IGV2ZW50SGFuZGxlci5ub2RlLnVuaXF1ZUlkO1xuICAgIGNvbnN0IGdyYW50Q29uZGl0aW9uOiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9ID0ge307XG4gICAgZ3JhbnRDb25kaXRpb25bYGF1dG9zY2FsaW5nOlJlc291cmNlVGFnLyR7dGFnS2V5fWBdID0gdGFnVmFsdWU7XG4gICAgVGFncy5vZih0aGlzLmF1dG9zY2FsaW5nR3JvdXApLmFkZCh0YWdLZXksIHRhZ1ZhbHVlKTtcblxuICAgIC8vIEFsbG93IHRoZSBsYW1iZGEgdG8gY29tcGxldGUgdGhlIGxpZmVjeWNsZSBhY3Rpb24gZm9yIG9ubHkgdGFnZ2VkIEFTR3MuXG4gICAgY29uc3QgaWFtQ29tcGxldGVMaWZlY3ljbGUgPSBuZXcgUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGVmZmVjdDogRWZmZWN0LkFMTE9XLFxuICAgICAgYWN0aW9uczogW1xuICAgICAgICAnYXV0b3NjYWxpbmc6Q29tcGxldGVMaWZlY3ljbGVBY3Rpb24nLFxuICAgICAgXSxcbiAgICAgIHJlc291cmNlczogW1xuICAgICAgICBgYXJuOiR7c3RhY2sucGFydGl0aW9ufTphdXRvc2NhbGluZzoke3N0YWNrLnJlZ2lvbn06JHtzdGFjay5hY2NvdW50fTphdXRvU2NhbGluZ0dyb3VwOio6YXV0b1NjYWxpbmdHcm91cE5hbWUvKmAsXG4gICAgICBdLFxuICAgICAgY29uZGl0aW9uczoge1xuICAgICAgICAnRm9yQW55VmFsdWU6U3RyaW5nRXF1YWxzJzogZ3JhbnRDb25kaXRpb24sXG4gICAgICB9LFxuICAgIH0pO1xuICAgIGV2ZW50SGFuZGxlci5yb2xlIS5hZGRUb1BvbGljeShpYW1Db21wbGV0ZUxpZmVjeWNsZSk7XG5cbiAgICBpZiAoIXNpbmdsZXRvblByZUV4aXN0cykge1xuICAgICAgLy8gQWxsb3cgdGhlIGxhbWJkYSB0byBhdHRhY2ggdGhlIEVOSSB0byB0aGUgaW5zdGFuY2UgdGhhdCB3YXMgY3JlYXRlZC5cbiAgICAgIC8vIFJlZmVyZW5jaW5nOiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hbWF6b25lYzIuaHRtbFxuICAgICAgLy8gTGFzdC1BY2Nlc3NlZDogSnVseSAyMDIwXG4gICAgICAvLyBUaGUgZWMyOkRlc2NyaWJlTmV0d29ya0ludGVyZmFjZXMsIGFuZCBlYzI6QXR0YWNoTmV0d29ya0ludGVyZmFjZSBvcGVyYXRpb25zXG4gICAgICAvLyBkbyBub3Qgc3VwcG9ydCBjb25kaXRpb25zLCBhbmQgZG8gbm90IHN1cHBvcnQgcmVzb3VyY2UgcmVzdHJpY3Rpb24uXG4gICAgICAvLyBTbywgd2Ugb25seSBhdHRhY2ggdGhlIHBvbGljeSB0byB0aGUgbGFtYmRhIGZ1bmN0aW9uIG9uY2U7IHdoZW4gd2UgZmlyc3QgY3JlYXRlIGl0LlxuICAgICAgY29uc3QgaWFtRW5pQXR0YWNoID0gbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGVmZmVjdDogRWZmZWN0LkFMTE9XLFxuICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgJ2VjMjpEZXNjcmliZU5ldHdvcmtJbnRlcmZhY2VzJyxcbiAgICAgICAgICAnZWMyOkF0dGFjaE5ldHdvcmtJbnRlcmZhY2UnLFxuICAgICAgICBdLFxuICAgICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgICAgfSk7XG4gICAgICBldmVudEhhbmRsZXIucm9sZSEuYWRkVG9Qb2xpY3koaWFtRW5pQXR0YWNoKTtcbiAgICB9XG5cbiAgICByZXR1cm4gZXZlbnRIYW5kbGVyO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHJvdGVjdGVkIHNldHVwTGlmZWN5Y2xlTm90aWZpY2F0aW9uVG9waWMobGFtYmRhSGFuZGxlcjogTGFtYmRhRnVuY3Rpb24pOiB7IFtrZXk6IHN0cmluZ106IGFueSB9IHtcbiAgICBjb25zdCBzdGFjayA9IFN0YWNrLm9mKHRoaXMpO1xuICAgIC8vIFdlIG9ubHkgbmVlZCB0byBoYXZlIGEgc2luZ2xlIFNOUyB0b3BpYyAmIHN1YnNjcmlwdGlvbiBzZXQgdXAgdG8gaGFuZGxlIGxpZmVjeWNsZSBldmVudHMgZm9yICphbGwqIGluc3RhbmNlcyBvZiB0aGlzIGNsYXNzLlxuICAgIC8vIFdlIGhhdmUgdG8gYmUgY2FyZWZ1bCwgaG93ZXZlciwgdG8gZW5zdXJlIHRoYXQgb3VyIGluaXRpYWwgc2V0dXAgb25seSBoYXBwZW5zIG9uY2Ugd2hlbiB3ZSBmaXJzdCBhZGQgdGhlIHRvcGljIGFuZCBzdWNoXG4gICAgLy8gdG8gdGhpcyBzdGFjazsgb3RoZXJ3aXNlLCB3ZSB3aWxsIG5vdCBiZSBhYmxlIHRvIGRlcGxveSBtb3JlIHRoYW4gb25lIG9mIHRoZXNlIGNvbnN0cnVjdHMgaW4gYSBzdGFjay5cblxuICAgIGNvbnN0IG5vdGlmaWNhdGlvblJvbGVVbmlxdWVJZCA9ICdBdHRhY2hFbmlOb3RpZmljYXRpb25Sb2xlJyArIHRoaXMucmVtb3ZlSHlwaGVucygnYTAzNzZmZjgtMjQ4ZS00NTM0LWJmNDItNThjNmZmYTRkNWI0Jyk7XG4gICAgY29uc3Qgbm90aWZpY2F0aW9uVG9waWNVbmlxdWVJZCA9ICdBdHRhY2hFbmlOb3RpZmljYXRpb25Ub3BpYycgKyB0aGlzLnJlbW92ZUh5cGhlbnMoJ2M4YjFlOWE2LTc4M2MtNDk1NC1iMTkxLTIwNGRkNWUzYjllMCcpO1xuICAgIGxldCBub3RpZmljYXRpb25Ub3BpYzogVG9waWMgPSAoc3RhY2subm9kZS50cnlGaW5kQ2hpbGQobm90aWZpY2F0aW9uVG9waWNVbmlxdWVJZCkgYXMgVG9waWMpO1xuICAgIGxldCBub3RpZmljYXRpb25Sb2xlOiBSb2xlO1xuICAgIGlmICghbm90aWZpY2F0aW9uVG9waWMpIHtcbiAgICAgIC8vIEZpcnN0IHRpbWUgY3JlYXRpbmcgdGhlIHNpbmdsZXRvbiBUb3BpYyBpbiB0aGlzIHN0YWNrLiBTZXQgaXQgYWxsIHVwLi4uXG5cbiAgICAgIG5vdGlmaWNhdGlvblJvbGUgPSBuZXcgUm9sZShzdGFjaywgbm90aWZpY2F0aW9uUm9sZVVuaXF1ZUlkLCB7XG4gICAgICAgIGFzc3VtZWRCeTogbmV3IFNlcnZpY2VQcmluY2lwYWwoJ2F1dG9zY2FsaW5nLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgIH0pO1xuXG4gICAgICBub3RpZmljYXRpb25Ub3BpYyA9IG5ldyBUb3BpYyhzdGFjaywgbm90aWZpY2F0aW9uVG9waWNVbmlxdWVJZCwge1xuICAgICAgICBkaXNwbGF5TmFtZTogYEZvciBSRkRLIGluc3RhbmNlLWxhdW5jaCBub3RpZmljYXRpb25zIGZvciBzdGFjayAnJHtzdGFjay5zdGFja05hbWV9J2AsXG4gICAgICB9KTtcblxuICAgICAgbm90aWZpY2F0aW9uVG9waWMuYWRkU3Vic2NyaXB0aW9uKG5ldyBMYW1iZGFTdWJzY3JpcHRpb24obGFtYmRhSGFuZGxlcikpO1xuICAgICAgbm90aWZpY2F0aW9uVG9waWMuZ3JhbnRQdWJsaXNoKG5vdGlmaWNhdGlvblJvbGUpO1xuICAgIH0gZWxzZSB7XG4gICAgICBub3RpZmljYXRpb25Sb2xlID0gc3RhY2subm9kZS5maW5kQ2hpbGQobm90aWZpY2F0aW9uUm9sZVVuaXF1ZUlkKSBhcyBSb2xlO1xuICAgIH1cblxuICAgIHJldHVybiB7XG4gICAgICB0b3BpYzogbm90aWZpY2F0aW9uVG9waWMsXG4gICAgICByb2xlOiBub3RpZmljYXRpb25Sb2xlLFxuICAgIH07XG4gIH1cblxuICAvKipcbiAgICogQ29udmVydCBhIFVVSUQgaW50byBhIHN0cmluZyB0aGF0J3MgdXNhYmxlIGluIGEgY29uc3RydWN0IGlkLlxuICAgKi9cbiAgcHJpdmF0ZSByZW1vdmVIeXBoZW5zKHg6IHN0cmluZyk6IHN0cmluZyB7XG4gICAgcmV0dXJuIHgucmVwbGFjZSgvWy1dL2csICcnKTtcbiAgfVxufVxuIl19