"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.MongoDbInstance = void 0;
const path = require("path");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const _1 = require("./");
const runtime_info_1 = require("./runtime-info");
/**
 * This construct provides a {@link StaticPrivateIpServer} that is hosting MongoDB.
 *
 * The data for this MongoDB database
 * is stored in an Amazon Elastic Block Storage (EBS) Volume that is automatically attached to the instance when it is
 * launched, and is separate from the instance's root volume; it is recommended that you set up a backup schedule for
 * this volume.
 *
 * When this instance is first launched, or relaunched after an instance replacement, it will:
 * 1. Attach an EBS volume to /var/lib/mongo upon which the MongoDB data is stored;
 * 2. Automatically install the specified version of MongoDB, from the official Mongo Inc. sources;
 * 3. Create an admin user in that database if one has not yet been created -- the credentials for this user
 * can be provided by you, or randomly generated;
 * 4. Configure MongoDB to require authentication, and only allow encrypted connections over TLS.
 *
 * The instance's launch logs and MongoDB logs will be automatically stored in Amazon CloudWatch logs; the
 * default log group name is: /renderfarm/<this construct ID>
 *
 * Resources Deployed
 * ------------------------
 * - {@link StaticPrivateIpServer} that hosts MongoDB.
 * - An A-Record in the provided PrivateHostedZone to create a DNS entry for this server's static private IP.
 * - A Secret in AWS SecretsManager that contains the administrator credentials for MongoDB.
 * - An encrypted Amazon Elastic Block Store (EBS) Volume on which the MongoDB data is stored.
 * - Amazon CloudWatch log group that contains instance-launch and MongoDB application logs.
 *
 * Security Considerations
 * ------------------------
 * - The administrator credentials for MongoDB are stored in a Secret within AWS SecretsManager. You must strictly limit
 *    access to this secret to only entities that require it.
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The EBS Volume that is created by, or provided to, this construct is used to store the contents of your MongoDB data. To
 *    protect the sensitive data in your database, you should not grant access to this EBS Volume to any principal or instance
 *    other than the instance created by this construct. Furthermore, we recommend that you ensure that the volume that is
 *    used for this purpose is encrypted at rest.
 * - This construct uses this package's {@link StaticPrivateIpServer}, {@link MongoDbInstaller}, {@link CloudWatchAgent},
 *    {@link ExportingLogGroup}, and {@link MountableBlockVolume}. Security considerations that are outlined by the documentation
 *    for those constructs should also be taken into account.
 */
class MongoDbInstance extends core_1.Construct {
    /**
     *
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e;
        super(scope, id);
        this.version = props.mongoDb.version;
        // Select the subnet for this instance.
        const { subnets } = props.vpc.selectSubnets(props.vpcSubnets);
        if (subnets.length === 0) {
            throw new Error(`Did not find any subnets matching ${JSON.stringify(props.vpcSubnets)}. Please use a different selection.`);
        }
        const subnet = subnets[0];
        this.server = new _1.StaticPrivateIpServer(this, 'Server', {
            vpc: props.vpc,
            vpcSubnets: { subnets: [subnet] },
            instanceType: (_a = props.instanceType) !== null && _a !== void 0 ? _a : new aws_ec2_1.InstanceType('r5.large'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
            blockDevices: [
                {
                    deviceName: '/dev/xvda',
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(MongoDbInstance.ROOT_DEVICE_SIZE.toGibibytes(), { encrypted: true }),
                },
            ],
            keyName: props.keyName,
            resourceSignalTimeout: core_1.Duration.minutes(5),
            role: props.role,
            securityGroup: props.securityGroup,
        });
        new aws_route53_1.ARecord(this, 'ARecord', {
            target: aws_route53_1.RecordTarget.fromIpAddresses(this.server.privateIpAddress),
            zone: props.mongoDb.dnsZone,
            recordName: props.mongoDb.hostname,
        });
        this.adminUser = (_b = props.mongoDb.adminUser) !== null && _b !== void 0 ? _b : new aws_secretsmanager_1.Secret(this, 'AdminUser', {
            description: `Admin credentials for the MongoDB database ${this.node.uniqueId}`,
            generateSecretString: {
                excludeCharacters: '"()$\'',
                excludePunctuation: true,
                includeSpace: false,
                passwordLength: 24,
                requireEachIncludedType: true,
                generateStringKey: 'password',
                secretStringTemplate: JSON.stringify({ username: 'admin' }),
            },
        });
        this.mongoDataVolume = (_d = (_c = props.mongoDb.mongoDataVolume) === null || _c === void 0 ? void 0 : _c.volume) !== null && _d !== void 0 ? _d : new aws_ec2_1.Volume(this, 'MongoDbData', {
            size: MongoDbInstance.DEFAULT_MONGO_DEVICE_SIZE,
            ...(_e = props.mongoDb.mongoDataVolume) === null || _e === void 0 ? void 0 : _e.volumeProps,
            availabilityZone: subnet.availabilityZone,
            encrypted: true,
        });
        const volumeMount = new _1.MountableBlockVolume(this, {
            blockVolume: this.mongoDataVolume,
            volumeFormat: _1.BlockVolumeFormat.XFS,
        });
        const mongoInstaller = new _1.MongoDbInstaller(this, {
            version: props.mongoDb.version,
            userSsplAcceptance: props.mongoDb.userSsplAcceptance,
        });
        // Set up the server's UserData.
        this.server.userData.addCommands('set -xefuo pipefail');
        this.server.userData.addSignalOnExitCommand(this.server.autoscalingGroup);
        this.configureCloudWatchLogStreams(this.server, id, props.logGroupProps); // MUST BE FIRST
        volumeMount.mountToLinuxInstance(this.server, {
            location: MongoDbInstance.MONGO_DEVICE_MOUNT_POINT,
        });
        mongoInstaller.installOnLinuxInstance(this.server);
        this.configureMongoDb(this.server, props.mongoDb);
        this.certificateChain = props.mongoDb.serverCertificate.certChain;
        this.connections = this.server.connections;
        this.grantPrincipal = this.server.grantPrincipal;
        this.port = 27017;
        this.role = this.server.role;
        this.userData = this.server.userData;
        this.fullHostname = `${props.mongoDb.hostname}.${props.mongoDb.dnsZone.zoneName}`;
        this.node.defaultChild = this.server;
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Adds UserData commands to install & configure the CloudWatch Agent onto the instance.
     *
     * The commands configure the agent to stream the following logs to a new CloudWatch log group:
     *      - The cloud-init log
     *      - The MongoDB application log.
     *
     * @param host The instance/host to setup the CloudWatchAgent upon.
     * @param groupName Name to append to the log group prefix when forming the log group name.
     * @param logGroupProps Properties for the log group.
     */
    configureCloudWatchLogStreams(host, groupName, logGroupProps) {
        var _a;
        const prefix = (_a = logGroupProps === null || logGroupProps === void 0 ? void 0 : logGroupProps.logGroupPrefix) !== null && _a !== void 0 ? _a : MongoDbInstance.DEFAULT_LOG_GROUP_PREFIX;
        const defaultedLogGroupProps = {
            ...logGroupProps,
            logGroupPrefix: prefix,
        };
        const logGroup = _1.LogGroupFactory.createOrFetch(this, 'MongoDbInstanceLogGroupWrapper', groupName, defaultedLogGroupProps);
        logGroup.grantWrite(host.grantPrincipal);
        const cloudWatchConfigurationBuilder = new _1.CloudWatchConfigBuilder(MongoDbInstance.CLOUDWATCH_LOG_FLUSH_INTERVAL);
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'cloud-init-output', '/var/log/cloud-init-output.log');
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'MongoDB', '/var/log/mongodb/mongod.log');
        new _1.CloudWatchAgent(this, 'MongoDbInstanceLogsConfig', {
            cloudWatchConfig: cloudWatchConfigurationBuilder.generateCloudWatchConfiguration(),
            host,
        });
    }
    /**
     * Adds commands to the userData of the instance to install MongoDB, create an admin user if one does not exist, and to to start mongod running.
     */
    configureMongoDb(instance, settings) {
        const scriptsAsset = new aws_s3_assets_1.Asset(this, 'MongoSetup', {
            path: path.join(__dirname, '..', 'scripts', 'mongodb', settings.version),
        });
        scriptsAsset.grantRead(instance.grantPrincipal);
        const scriptZipfile = instance.userData.addS3DownloadCommand({
            bucket: scriptsAsset.bucket,
            bucketKey: scriptsAsset.s3ObjectKey,
        });
        instance.userData.addCommands(
        // Ensure mongod is installed and stopped before we go any further
        'which mongod && test -f /etc/mongod.conf', 'sudo service mongod stop', 
        // We're going to make a temporary RAM filesystem for the mongo setup files.
        // This will let us write sensitive data to "disk" without worrying about it
        // being persisted in any physical disk, even temporarily.
        'MONGO_SETUP_DIR=$(mktemp -d)', 'mkdir -p "${MONGO_SETUP_DIR}"', 'sudo mount -t tmpfs -o size=50M tmpfs "${MONGO_SETUP_DIR}"', 'pushd "${MONGO_SETUP_DIR}"', `unzip ${scriptZipfile}`, 
        // Backup mongod.conf for now
        'cp /etc/mongod.conf .');
        const cert = settings.serverCertificate;
        instance.userData.addCommands(`bash serverCertFromSecrets.sh "${cert.cert.secretArn}" "${cert.certChain.secretArn}" "${cert.key.secretArn}" "${cert.passphrase.secretArn}"`);
        cert.cert.grantRead(instance.grantPrincipal);
        cert.certChain.grantRead(instance.grantPrincipal);
        cert.key.grantRead(instance.grantPrincipal);
        cert.passphrase.grantRead(instance.grantPrincipal);
        const certsDirectory = '/etc/mongod_certs';
        instance.userData.addCommands(
        // Move the certificates into place
        `sudo mkdir -p ${certsDirectory}`, `sudo mv ./ca.crt ./key.pem ${certsDirectory}`, 'sudo chown root.mongod -R /etc/mongod_certs/', // Something weird about shell interpretation. Can't use '*' on this or next line.
        'sudo chmod 640 -R /etc/mongod_certs/', 'sudo chmod 750 /etc/mongod_certs/', // Directory needs to be executable.
        // mongod user id might, potentially change on reboot. Make sure we own all mongo data
        `sudo chown mongod.mongod -R ${MongoDbInstance.MONGO_DEVICE_MOUNT_POINT}`, 
        // Configure mongod
        'bash ./setMongoLimits.sh', `bash ./setStoragePath.sh "${MongoDbInstance.MONGO_DEVICE_MOUNT_POINT}"`, 'bash ./setMongoNoAuth.sh', 'sudo service mongod start', `bash ./setAdminCredentials.sh "${this.adminUser.secretArn}"`);
        this.adminUser.grantRead(instance.grantPrincipal);
        instance.userData.addCommands(
        // Setup for live deployment, and start mongod
        'sudo service mongod stop', 'bash ./setLiveConfiguration.sh', 'sudo systemctl enable mongod', // Enable restart on reboot
        'sudo service mongod start', 'popd');
        instance.userData.addOnExitCommands(
        // Clean up the temporary RAM filesystem
        'test "${MONGO_SETUP_DIR} != "" && sudo umount "${MONGO_SETUP_DIR}');
    }
}
exports.MongoDbInstance = MongoDbInstance;
// How often Cloudwatch logs will be flushed.
MongoDbInstance.CLOUDWATCH_LOG_FLUSH_INTERVAL = core_1.Duration.seconds(15);
// Default prefix for a LogGroup if one isn't provided in the props.
MongoDbInstance.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
// Size of the EBS volume for MongoDB data, if we create one.
MongoDbInstance.DEFAULT_MONGO_DEVICE_SIZE = core_1.Size.gibibytes(20);
// Mount point for the MongoDB data volume.
MongoDbInstance.MONGO_DEVICE_MOUNT_POINT = '/var/lib/mongo';
// Size of the root device volume on the instance.
MongoDbInstance.ROOT_DEVICE_SIZE = core_1.Size.gibibytes(10);
//# sourceMappingURL=data:application/json;base64,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