/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { ICertificate } from '@aws-cdk/aws-certificatemanager';
import { Table } from '@aws-cdk/aws-dynamodb';
import { IKey } from '@aws-cdk/aws-kms';
import { ISecret } from '@aws-cdk/aws-secretsmanager';
import { Construct, ResourceEnvironment, Stack, Tag } from '@aws-cdk/core';
/**
 * Properties for importing a Certificate from Secrets into ACM.
 */
export interface ImportedAcmCertificateProps {
    /**
     * A Secret that contains the Certificate data.
     */
    readonly cert: ISecret;
    /**
     * A Secret that contains the encrypted Private Key data.
     */
    readonly key: ISecret;
    /**
     * A Secret that contains the passphrase of the encrypted Private Key.
     */
    readonly passphrase: ISecret;
    /**
     * A Secret that contains the chain of Certificates used to sign this Certificate.
     *
     * @default : No certificate chain is used, signifying a self-signed Certificate
     */
    readonly certChain?: ISecret;
    /**
     * The KMS Key used to encrypt the secrets.
     *
     * The Custom Resource to import the Certificate to ACM will be granted
     * permission to decrypt Secrets using this Key.
     *
     * @default : If the account's default CMK was used to encrypt the Secrets, no special permissions need to be given
     */
    readonly encryptionKey?: IKey;
}
/**
 * A Construct that creates an AWS CloudFormation Custom Resource that models a certificate that is imported into AWS Certificate Manager (ACM).
 *
 * It uses an AWS Lambda Function to extract the certificate from Secrets in AWS SecretsManager
 * and then import it into ACM. The interface is intended to be used with the {@link X509CertificatePem} Construct.
 *
 * Resources Deployed
 * ------------------------
 * - DynamoDB Table - Used for tracking resources created by the Custom Resource.
 * - An AWS Lambda Function, with IAM Role - Used to create/update/delete the Custom Resource.
 * - AWS Certificate Manager Certificate - Created by the Custom Resource.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The AWS Lambda for this construct also has broad IAM permissions to delete any Certificate that is stored
 *    in AWS Certificate Manager. You should not grant any additional actors/principals the ability to modify or
 *    execute this Lambda.
 */
export declare class ImportedAcmCertificate extends Construct implements ICertificate {
    private static IMPORTER_UUID;
    /**
     * The ARN for the Certificate that was imported into ACM.
     */
    readonly certificateArn: string;
    /**
     * The stack in which this resource is defined.
     */
    readonly stack: Stack;
    /**
     * The environment this resource belongs to.
     *
     * For resources that are created and managed by the CDK
     * (generally, those created by creating new class instances like Role, Bucket, etc.),
     * this is always the same as the environment of the stack they belong to;
     * however, for imported resources
     * (those obtained from static methods like fromRoleArn, fromBucketName, etc.),
     * that might be different than the stack they were imported into.
     */
    readonly env: ResourceEnvironment;
    /**
     *
     */
    protected readonly database: Table;
    /**
     *
     */
    protected readonly uniqueTag: Tag;
    /**
     *
     */
    constructor(scope: Construct, id: string, props: ImportedAcmCertificateProps);
}
