"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ExportingLogGroup = void 0;
const path = require("path");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
/**
 * This construct takes the name of a CloudWatch LogGroup and will either create it if it doesn't already exist, or reuse the existing one.
 *
 * It also creates a regularly scheduled lambda that will export LogEvents to S3
 * before they expire in CloudWatch.
 *
 * It's used for cost-reduction, as it is more economical to archive logs in S3 than CloudWatch when
 * retaining them for more than a week.
 * Note, that it isn't economical to export logs to S3 if you plan on storing them for less than
 * 7 days total (CloudWatch and S3 combined).
 *
 * Resources Deployed
 * ------------------------
 * - The Lambda SingletonFunction that checks for the existence of the LogGroup.
 * - The CloudWatch LogGroup (if it didn't exist already).
 * - The CloudWatch Alarm watching log exportation failures.
 * - The CloudWatch Event Rule to schedule log exportation.
 * - The Lambda SingletonFunction, with role, to export log groups to S3 by schedule.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 */
class ExportingLogGroup extends core_1.Construct {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * UUID needed to identify the SingletonFunction for the log exporter.
         */
        this.LOG_EXPORTER_UUID = '6382448c-e4b2-42e9-b14f-a0a9ccdb198e';
        /**
         * Duration of time between export task Lambda runs.
         */
        this.EXPORT_TASK_FREQUENCY = core_1.Duration.hours(1);
        /**
         * Default value for the number of days to retain logs in CloudWatch for.
         */
        this.RETENTION_DEFAULT = aws_logs_1.RetentionDays.THREE_DAYS;
        const retentionInDays = props.retention ? props.retention : this.RETENTION_DEFAULT;
        const exportLogsFunction = this.setupLogsExporter();
        this.exportErrorAlarm = exportLogsFunction.metricErrors().createAlarm(this, 'LogExporterFailureAlarm', {
            evaluationPeriods: 1,
            threshold: 1,
        });
        const logRetention = new aws_lambda_1.LogRetention(this, 'LogRetention', {
            logGroupName: props.logGroupName,
            retention: retentionInDays,
        });
        this.logGroup = aws_logs_1.LogGroup.fromLogGroupArn(scope, `${props.logGroupName}LogGroup`, logRetention.logGroupArn);
        this.logGroup.grant(exportLogsFunction, 'logs:CreateExportTask');
        const scheduledLogExportation = new aws_events_1.Rule(this, 'LogExporterRule', {
            schedule: aws_events_1.Schedule.rate(this.EXPORT_TASK_FREQUENCY),
        });
        scheduledLogExportation.addTarget(new aws_events_targets_1.LambdaFunction(exportLogsFunction, {
            event: aws_events_1.RuleTargetInput.fromObject({
                BucketName: props.bucketName,
                ExportFrequencyInHours: this.EXPORT_TASK_FREQUENCY.toHours(),
                LogGroupName: props.logGroupName,
                RetentionInHours: retentionInDays.valueOf() * 24,
            }),
        }));
    }
    setupLogsExporter() {
        const exportLogsFunction = new aws_lambda_1.SingletonFunction(this, 'LogExporterFunction', {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '../lambdas/nodejs/export-logs')),
            handler: 'index.handler',
            lambdaPurpose: 'LogGroupExporter',
            logRetention: aws_logs_1.RetentionDays.ONE_DAY,
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            uuid: this.LOG_EXPORTER_UUID,
        });
        exportLogsFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['logs:DescribeExportTasks'],
            effect: aws_iam_1.Effect.ALLOW,
            resources: ['*'],
        }));
        return exportLogsFunction;
    }
}
exports.ExportingLogGroup = ExportingLogGroup;
//# sourceMappingURL=data:application/json;base64,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