"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.importCert = exports.AcmCertificateImporter = void 0;
/* eslint-disable no-console */
const crypto = require("crypto");
// eslint-disable-next-line import/no-extraneous-dependencies
const aws_sdk_1 = require("aws-sdk");
const custom_resource_1 = require("../lib/custom-resource");
const x509_certs_1 = require("../lib/x509-certs");
const types_1 = require("./types");
const defaultSleep = function (ms) {
    return new Promise(resolve => setTimeout(resolve, ms));
};
const ACM_VERSION = '2015-12-08';
const DYNAMODB_VERSION = '2012-08-10';
const SECRETS_MANAGER_VERSION = '2017-10-17';
class AcmCertificateImporter extends custom_resource_1.DynamoBackedCustomResource {
    constructor(acmClient, dynamoDbClient, secretsManagerClient) {
        super(dynamoDbClient);
        this.acmClient = acmClient;
        this.secretsManagerClient = secretsManagerClient;
    }
    validateInput(data) {
        return types_1.implementsIAcmImportCertProps(data);
    }
    async doCreate(physicalId, resourceProperties) {
        const resourceTable = await this.getResourceTable();
        await Promise.all([
            this.databasePermissionsCheck(resourceTable),
        ]);
        const cert = await this.getSecretString(resourceProperties.X509CertificatePem.Cert);
        const certChainArn = resourceProperties.X509CertificatePem.CertChain;
        const certChain = certChainArn.length > 0 ? await this.getSecretString(certChainArn) : undefined;
        const key = await this.getSecretString(resourceProperties.X509CertificatePem.Key);
        const passphrase = await this.getSecretString(resourceProperties.X509CertificatePem.Passphrase);
        const decryptedKey = await x509_certs_1.Certificate.decryptKey(key, passphrase);
        const tags = resourceProperties.Tags;
        const certificateArn = await this.importAndStoreCertificate({
            resourceTable,
            key: decryptedKey,
            cert,
            certChain,
            physicalId,
            tags,
        });
        return { CertificateArn: certificateArn };
    }
    async doDelete(physicalId) {
        const resourceTable = await this.getResourceTable();
        await Promise.all([
            this.databasePermissionsCheck(resourceTable),
        ]);
        const resources = await resourceTable.query(physicalId);
        const maxAttempts = 10;
        for (const [key, resource] of Object.entries(resources)) {
            const arn = resource.ARN;
            let inUseByResources = [];
            for (let attempt = 0; attempt < maxAttempts; attempt++) {
                const { Certificate: cert } = await this.acmClient.describeCertificate({
                    CertificateArn: arn,
                }).promise();
                inUseByResources = cert.InUseBy || [];
                if (inUseByResources.length) {
                    // Exponential backoff with jitter based on 200ms base
                    // component of backoff fixed to ensure minimum total wait time on
                    // slow targets.
                    const base = Math.pow(2, attempt);
                    await defaultSleep(Math.random() * base * 50 + base * 150);
                }
                else {
                    break;
                }
            }
            if (inUseByResources.length) {
                throw new Error(`Response from describeCertificate did not contain an empty InUseBy list after ${maxAttempts} attempts.`);
            }
            console.log(`Deleting resource for '${key}'`);
            try {
                await this.acmClient.deleteCertificate({ CertificateArn: arn }).promise();
            }
            catch (e) {
                // AccessDeniedException can happen if either:
                //  a) We do not have the required permission to delete the Certificate (unlikely)
                //  b) The Certificate has already been deleted (more likely)
                if (e.message.indexOf('AccessDeniedException')) {
                    console.warn(`Could not delete Certificate ${arn}. Please ensure it has been deleted.`);
                }
                throw e; // Rethrow so the custom resource handler will error-out.
            }
            await resourceTable.deleteItem({
                primaryKeyValue: physicalId,
                sortKeyValue: key,
            });
        }
    }
    async importAndStoreCertificate(args) {
        let certificateArn;
        const sortKey = crypto.createHash('md5').update(args.cert).digest('hex');
        const existingItem = await args.resourceTable.getItem({
            primaryKeyValue: args.physicalId,
            sortKeyValue: sortKey,
        });
        if (existingItem) {
            if (!existingItem.ARN) {
                throw Error("Database Item missing 'ARN' attribute");
            }
            certificateArn = existingItem.ARN;
            const certificate = await this.acmClient.getCertificate({ CertificateArn: certificateArn }).promise();
            // If the cert already existed, we will updating it by performing an import again, with the new values.
            if (certificate.Certificate) {
                const importCertRequest = {
                    CertificateArn: certificateArn,
                    Certificate: args.cert,
                    CertificateChain: args.certChain,
                    PrivateKey: args.key,
                    Tags: args.tags,
                };
                await this.acmClient.importCertificate(importCertRequest).promise();
            }
            else {
                throw Error(`Database entry ${existingItem.ARN} could not be found in ACM.`);
            }
        }
        else {
            const importCertRequest = {
                Certificate: args.cert,
                CertificateChain: args.certChain,
                PrivateKey: args.key,
                Tags: args.tags,
            };
            const resp = await this.acmClient.importCertificate(importCertRequest).promise();
            if (!resp.CertificateArn) {
                throw new Error(`CertificateArn was not properly populated after attempt to import ${args.cert}`);
            }
            certificateArn = resp.CertificateArn;
            await args.resourceTable.putItem({
                primaryKeyValue: args.physicalId,
                sortKeyValue: sortKey,
                attributes: {
                    ARN: certificateArn,
                },
                allow_overwrite: false,
            });
        }
        return certificateArn;
    }
    async getSecretString(SecretId) {
        console.debug(`Retrieving secret: ${SecretId}`);
        const resp = await this.secretsManagerClient.getSecretValue({ SecretId }).promise();
        if (!resp.SecretString) {
            throw new Error(`Secret ${SecretId} did not contain a SecretString as expected`);
        }
        return resp.SecretString;
    }
}
exports.AcmCertificateImporter = AcmCertificateImporter;
/**
 * The handler used to import an X.509 certificate to ACM from a Secret
 */
async function importCert(event, context) {
    const handler = new AcmCertificateImporter(new aws_sdk_1.ACM({ apiVersion: ACM_VERSION }), new aws_sdk_1.DynamoDB({ apiVersion: DYNAMODB_VERSION }), new aws_sdk_1.SecretsManager({ apiVersion: SECRETS_MANAGER_VERSION }));
    return await handler.handler(event, context);
}
exports.importCert = importCert;
//# sourceMappingURL=data:application/json;base64,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