"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.configureMongo = exports.MongoDbConfigure = void 0;
/* eslint-disable no-console */
const child_process_1 = require("child_process");
const util_1 = require("util");
// eslint-disable-next-line import/no-extraneous-dependencies
const aws_sdk_1 = require("aws-sdk");
const custom_resource_1 = require("../lib/custom-resource");
const filesystem_1 = require("../lib/filesystem");
const secrets_manager_1 = require("../lib/secrets-manager");
const types_1 = require("./types");
const exec = util_1.promisify(child_process_1.exec);
class MongoDbConfigure extends custom_resource_1.SimpleCustomResource {
    constructor(secretsManagerClient) {
        super();
        this.secretsManagerClient = secretsManagerClient;
    }
    /**
     * @inheritdoc
     */
    /* istanbul ignore next */ // @ts-ignore
    validateInput(data) {
        return types_1.implementsIMongoDbConfigureResource(data);
    }
    /**
     * @inheritdoc
     */
    // @ts-ignore  -- we do not use the physicalId
    async doCreate(physicalId, resourceProperties) {
        const mongoDbDriver = this.installMongoDbDriver();
        const mongoClient = await this.mongoLogin(mongoDbDriver, resourceProperties.Connection);
        try {
            if (resourceProperties.PasswordAuthUsers) {
                const adminDb = mongoClient.db('admin');
                for (const userArn of resourceProperties.PasswordAuthUsers) {
                    if (!await this.createPasswordAuthUser(adminDb, userArn)) {
                        // Note: Intentionally not including the data as part of this message. It may contain secrets, and including it will leak those secrets.
                        console.log(`User in '${userArn}' already exists in the MongoDB. No action performed for this user.`);
                    }
                }
            }
            if (resourceProperties.X509AuthUsers) {
                const externalDb = mongoClient.db('$external');
                for (const x509User of resourceProperties.X509AuthUsers) {
                    if (!await this.createX509AuthUser(externalDb, x509User)) {
                        // Note: Intentionally not including the data as part of this message. It may contain secrets, and including it will leak those secrets.
                        console.log(`User in '${x509User.Certificate}' already exists in the MongoDB. No action performed for this user.`);
                    }
                }
            }
        }
        finally {
            console.log('Closing Mongo connection');
            await mongoClient.close();
        }
        return undefined;
    }
    /**
     * @inheritdoc
     */
    /* istanbul ignore next */ // @ts-ignore
    async doDelete(physicalId, resourceProperties) {
        // Nothing to do -- we don't modify any existing DB contents.
        return;
    }
    /**
     * Installs the official NodeJS MongoDB driver into /tmp, and returns the module object for it.
     */
    /* istanbul ignore next */
    installMongoDbDriver() {
        console.log('Installing latest MongoDB Driver for NodeJS from npmjs.org');
        // Both HOME and --prefix are needed here because /tmp is the only writable location
        child_process_1.execSync('HOME=/tmp npm install mongodb@3 --production --no-package-lock --no-save --prefix /tmp');
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        return require('/tmp/node_modules/mongodb');
    }
    /**
     * Login to MongoDB and return the MongoClient connection object.
     * @param mongoDbDriver
     * @param options
     */
    async mongoLogin(mongoDbDriver, options) {
        // Get the CA cert.
        const caData = await this.readCertificateData(options.CaCertificate);
        await filesystem_1.writeAsciiFile('/tmp/ca.crt', caData);
        // Next, the login credentials
        const credentials = await this.readLoginCredentials(options.Credentials);
        // Login to MongoDB
        const mongoUri = `mongodb://${options.Hostname}:${options.Port}`;
        console.log(`Connecting to: ${mongoUri}`);
        // Reference: http://mongodb.github.io/node-mongodb-native/3.5/api/MongoClient.html#.connect
        return await mongoDbDriver.MongoClient.connect(mongoUri, {
            tls: true,
            tlsInsecure: false,
            tlsCAFile: '/tmp/ca.crt',
            auth: {
                user: credentials.username,
                password: credentials.password,
            },
            useUnifiedTopology: true,
        });
    }
    /**
     * Retrieve CA certificate data from the Secret with the given ARN.
     * @param certificateArn
     */
    async readCertificateData(certificateArn) {
        const data = await secrets_manager_1.Secret.fromArn(certificateArn, this.secretsManagerClient).getValue();
        if (Buffer.isBuffer(data) || !/BEGIN CERTIFICATE/.test(data)) {
            throw new Error(`CA Certificate Secret (${certificateArn}) must contain a Certificate in PEM format.`);
        }
        return data;
    }
    /**
     * Use openssl to retrieve the subject, in RFC2253 format, of the given certificate.
     * @param certificateData
     */
    async retrieveRfc2253Subject(certificateData) {
        await filesystem_1.writeAsciiFile('/tmp/client.crt', certificateData);
        const subject = await exec('openssl x509 -in /tmp/client.crt -noout -subject -nameopt RFC2253');
        return subject.stdout.replace(/subject= /, '').trim();
    }
    /**
     * Retrieve the credentials of the user that we're to login to the DB with.
     * @param credentialsArn
     */
    async readLoginCredentials(credentialsArn) {
        const data = await secrets_manager_1.Secret.fromArn(credentialsArn, this.secretsManagerClient).getValue();
        if (Buffer.isBuffer(data) || !data) {
            throw new Error(`Login credentials, in Secret ${credentialsArn}, for MongoDB must be a JSON encoded string`);
        }
        let credentials;
        try {
            credentials = JSON.parse(data);
        }
        catch (e) {
            // Note: Intentionally not including the data as part of this error message. It may contain secrets, and including it will leak those secrets.
            throw new Error(`Failed to parse JSON in MongoDB login credentials Secret (${credentialsArn}). Please ensure that the Secret contains properly formatted JSON.`);
        }
        for (const key of ['username', 'password']) {
            if (!(key in credentials)) {
                throw new Error(`Login credentials Secret (${credentialsArn}) is missing: ${key}`);
            }
        }
        return credentials;
    }
    /**
     * Read, from the given Secret, the information for a password-authenticated user to be created
     * in the DB.
     * @param userArn
     */
    async readPasswordAuthUserInfo(userArn) {
        const data = await secrets_manager_1.Secret.fromArn(userArn, this.secretsManagerClient).getValue();
        if (Buffer.isBuffer(data) || !data) {
            throw new Error(`Password-auth user credentials, in Secret ${userArn}, for MongoDB must be a JSON encoded string`);
        }
        let userCreds;
        try {
            userCreds = JSON.parse(data);
        }
        catch (e) {
            // Note: Intentionally not including the data as part of this error message. It may contain secrets, and including it will leak those secrets.
            throw new Error(`Failed to parse JSON for password-auth user Secret (${userArn}). Please ensure that the Secret contains properly formatted JSON.`);
        }
        for (const key of ['username', 'password', 'roles']) {
            if (!(key in userCreds)) {
                // Note: Intentionally not including the data as part of this error message. It may contain secrets, and including it will leak those secrets.
                throw new Error(`User credentials Secret '${userArn}' is missing: ${key}`);
            }
        }
        return userCreds;
    }
    /**
     * Query the given DB to determine whether or not there is a user with the given username.
     * @param db
     * @param username
     */
    async userExists(db, username) {
        const result = await db.command({ usersInfo: username });
        if (result.ok !== 1) {
            throw new Error(`MongoDB error checking whether user exists \'${username}\' -- ${JSON.stringify(result)}`);
        }
        return result.users.length > 0;
    }
    /**
     * Add a user to the database. This must only be called if you know that the user does not
     * already exist.
     * @param db
     * @param credentials
     */
    async createUser(db, credentials) {
        console.log(`Creating user: ${credentials.username}`);
        const request = {
            createUser: credentials.username,
            roles: credentials.roles,
        };
        // It is an error to include a pwd field with undefined value, and our password
        // will be absent/undefined for x.509 authenticated users in the $external DB.
        if (credentials.password) {
            request.pwd = credentials.password;
        }
        const result = await db.command(request);
        if (result.ok !== 1) {
            throw new Error(`MongoDB error when adding user \'${credentials.username}\' -- ${JSON.stringify(result)}`);
        }
    }
    /**
     * Create a user in the admin DB if it does not already exist. If it does exist, then do nothing.
     * @param db
     * @param userArn
     */
    async createPasswordAuthUser(db, userArn) {
        const credentials = await this.readPasswordAuthUserInfo(userArn);
        if (await this.userExists(db, credentials.username)) {
            return false;
        }
        await this.createUser(db, credentials);
        return true;
    }
    /**
     * Create a user in the $external DB if it does not already exist. If it does exist, then do nothing.
     * @param db
     * @param user
     */
    async createX509AuthUser(db, user) {
        const userCertData = await this.readCertificateData(user.Certificate);
        const username = await this.retrieveRfc2253Subject(userCertData);
        if (await this.userExists(db, username)) {
            return false;
        }
        const credentials = {
            username,
            // Note: No need to check for parse-errors. It's already been vetted twice. Once by the typescript code, and once by the
            // input verifier of this handler.
            roles: JSON.parse(user.Roles),
        };
        await this.createUser(db, credentials);
        return true;
    }
}
exports.MongoDbConfigure = MongoDbConfigure;
/**
 * The lambda handler that is used to log in to MongoDB and perform some configuration actions.
 */
/* istanbul ignore next */
async function configureMongo(event, context) {
    const handler = new MongoDbConfigure(new aws_sdk_1.SecretsManager());
    return await handler.handler(event, context);
}
exports.configureMongo = configureMongo;
//# sourceMappingURL=data:application/json;base64,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