"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable no-console */
const child_process = require("child_process");
const fs = require("fs");
const os = require("os");
const path = require("path");
const util_1 = require("util");
const filesystem_1 = require("../../filesystem");
const certificate_1 = require("../certificate");
const distinguished_name_1 = require("../distinguished-name");
const exec = util_1.promisify(child_process.exec);
let tmpDir;
// Enable/disable debugging statements.
const DEBUG = false;
if (!DEBUG) {
    console.debug = () => { };
}
beforeAll(async () => {
    tmpDir = await fs.promises.mkdtemp(path.join(os.tmpdir(), 'tmp.'));
});
afterAll(async () => {
    const unlinks = [];
    const filenames = await fs.promises.readdir(tmpDir);
    for (const file of filenames) {
        unlinks.push(fs.promises.unlink(path.join(tmpDir, file)));
    }
    await Promise.all(unlinks);
    await fs.promises.rmdir(tmpDir);
});
test('generate self-signed', async () => {
    // GIVEN
    const name = new distinguished_name_1.DistinguishedName({
        CN: 'TestCN',
        O: 'TestO',
        OU: 'TestOU',
    });
    const passphrase = 'test_passphrase';
    // WHEN
    const certificate = await certificate_1.Certificate.fromGenerated(name, passphrase);
    const crtFileName = path.join(tmpDir, 'ss-ca.crt');
    const keyFileName = path.join(tmpDir, 'ss-ca.key');
    await filesystem_1.writeAsciiFile(crtFileName, certificate.cert);
    await filesystem_1.writeAsciiFile(keyFileName, certificate.key);
    const certOut = await exec(`openssl x509 -noout -text -in ${crtFileName}`);
    const certVerification = certOut.stdout;
    const keyOut = await exec(`openssl rsa -noout -text -check -passin env:PW -in ${keyFileName}`, { env: { PW: passphrase } });
    const keyVerification = keyOut.stdout;
    // THEN
    expect(certificate.cert).toContain('-----BEGIN CERTIFICATE-----');
    expect(certificate.cert).toContain('-----END CERTIFICATE-----');
    expect(certificate.key).toContain('-----BEGIN ENCRYPTED PRIVATE KEY-----');
    expect(certificate.key).toContain('-----END ENCRYPTED PRIVATE KEY-----');
    expect(certificate.passphrase).toBe(passphrase);
    expect(certificate.certChain).toEqual('');
    expect(certVerification).toContain('Issuer: CN=TestCN, O=TestO, OU=TestOU');
    expect(certVerification).toContain('Subject: CN=TestCN, O=TestO, OU=TestOU');
    expect(certVerification).toContain('Version: 3 (0x2)');
    expect(certVerification).toContain('Public-Key: (2048 bit)');
    expect(keyVerification).toContain('RSA key ok');
    expect(keyVerification).toContain('Private-Key: (2048 bit)');
});
test('generate signed certificate', async () => {
    // GIVEN
    const caName = new distinguished_name_1.DistinguishedName({
        CN: 'TestCN',
        O: 'TestO',
        OU: 'TestOU',
    });
    const certName = new distinguished_name_1.DistinguishedName({
        CN: 'CertCN',
        O: 'CertO',
        OU: 'CertOU',
    });
    const ca = await certificate_1.Certificate.fromGenerated(caName, 'signing_passphrase');
    const passphrase = 'test_passphrase';
    // WHEN
    const certificate = await certificate_1.Certificate.fromGenerated(certName, passphrase, ca);
    const crtFileName = path.join(tmpDir, 'signed.crt');
    const crtChainFileName = path.join(tmpDir, 'chain.crt');
    const keyFileName = path.join(tmpDir, 'signed.key');
    await filesystem_1.writeAsciiFile(crtFileName, certificate.cert);
    if (certificate.certChain) {
        await filesystem_1.writeAsciiFile(crtChainFileName, certificate.certChain);
    }
    await filesystem_1.writeAsciiFile(keyFileName, certificate.key);
    const certOut = await exec(`openssl x509 -noout -text -in ${crtFileName}`);
    const certVerification = certOut.stdout;
    const certChainOut = await exec(`openssl x509 -noout -text -in ${crtChainFileName}`);
    const certChainVerification = certChainOut.stdout;
    const keyOut = await exec(`openssl rsa -noout -text -check -passin env:PW -in ${keyFileName}`, { env: { PATH: process.env.PATH, PW: passphrase } });
    const keyVerification = keyOut.stdout;
    // THEN
    expect(certificate.cert).toContain('-----BEGIN CERTIFICATE-----');
    expect(certificate.cert).toContain('-----END CERTIFICATE-----');
    // The cert chain should contain the signing cert
    expect(certificate.certChain).toContain(ca.cert);
    // The cert should not contain any chain
    expect(certificate.cert.indexOf('-----BEGIN CERTIFICATE-----')).toBe(certificate.cert.lastIndexOf('-----BEGIN CERTIFICATE-----'));
    expect(certificate.certChain).toContain('-----BEGIN CERTIFICATE-----');
    expect(certificate.certChain).toContain('-----END CERTIFICATE-----');
    expect(certificate.key).toContain('-----BEGIN ENCRYPTED PRIVATE KEY-----');
    expect(certificate.key).toContain('-----END ENCRYPTED PRIVATE KEY-----');
    expect(certificate.passphrase).toBe(passphrase);
    expect(certVerification).toContain('Issuer: CN=TestCN, O=TestO, OU=TestOU');
    expect(certVerification).toContain('Subject: CN=CertCN, O=CertO, OU=CertOU');
    expect(certVerification).toContain('Public-Key: (2048 bit)');
    expect(certChainVerification).toContain('Issuer: CN=TestCN, O=TestO, OU=TestOU');
    expect(certChainVerification).toContain('Subject: CN=TestCN, O=TestO, OU=TestOU');
    expect(certChainVerification).toContain('Public-Key: (2048 bit)');
    expect(keyVerification).toContain('RSA key ok');
    expect(keyVerification).toContain('Private-Key: (2048 bit)');
});
test('convert to PKCS #12', async () => {
    const caName = new distinguished_name_1.DistinguishedName({
        CN: 'TestCN',
        O: 'TestO',
        OU: 'TestOU',
    });
    const certName = new distinguished_name_1.DistinguishedName({
        CN: 'CertCN',
        O: 'CertO',
        OU: 'CertOU',
    });
    const ca = await certificate_1.Certificate.fromGenerated(caName, 'signing_passphrase');
    const passphrase = 'test_passphrase';
    const certificate = await certificate_1.Certificate.fromGenerated(certName, passphrase, ca);
    const pkcs12Passphrase = 'test_passphrase';
    // WHEN
    const pkcs12Data = await certificate.toPkcs12(pkcs12Passphrase);
    const fileName = path.join(tmpDir, 'cert.p12');
    await filesystem_1.writeBinaryFile(fileName, pkcs12Data);
    const pkcs12Validation = await exec(`openssl pkcs12 -in ${fileName} -info -nodes -passin env:PW`, { env: { PATH: process.env.PATH, PW: pkcs12Passphrase } });
    const validationOut = pkcs12Validation.stdout;
    // THEN
    expect(pkcs12Validation.stderr).toContain('MAC verified OK');
    // Must have the certificate's cert
    expect(validationOut).toContain('subject=/CN=CertCN/O=CertO/OU=CertOU\nissuer=/CN=TestCN/O=TestO/OU=TestOU\n-----BEGIN CERTIFICATE-----');
    // Must have the CA cert
    expect(validationOut).toContain('subject=/CN=TestCN/O=TestO/OU=TestOU\nissuer=/CN=TestCN/O=TestO/OU=TestOU\n-----BEGIN CERTIFICATE-----');
    // Must have the decrypted private key
    expect(validationOut).toContain('-----BEGIN PRIVATE KEY-----');
});
test('decrypt private key', async () => {
    // GIVEN
    const name = new distinguished_name_1.DistinguishedName({
        CN: 'TestCN',
        O: 'TestO',
        OU: 'TestOU',
    });
    const passphrase = 'test_passphrase';
    const certificate = await certificate_1.Certificate.fromGenerated(name, passphrase);
    // WHEN
    const decryptedKey = await certificate_1.Certificate.decryptKey(certificate.key, passphrase);
    const crtFileName = path.join(tmpDir, 'ca.crt');
    const keyFileName = path.join(tmpDir, 'ca.key');
    await filesystem_1.writeAsciiFile(crtFileName, certificate.cert);
    await filesystem_1.writeAsciiFile(keyFileName, certificate.key);
    const expectedDecryptedKeyOut = await exec(`openssl rsa -in ${keyFileName} -passin env:PW`, { env: { PATH: process.env.PATH, PW: passphrase } });
    const expectedDecryptedKey = expectedDecryptedKeyOut.stdout;
    // THEN
    expect(decryptedKey).toEqual(expectedDecryptedKey);
    // Must have the decrypted private key
    expect(decryptedKey).toContain('-----BEGIN RSA PRIVATE KEY-----');
});
//# sourceMappingURL=data:application/json;base64,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