"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.VersionProvider = exports.Product = exports.Platform = void 0;
/* eslint-disable no-console */
const fs = require("fs");
const https = require("https");
const url = require("url");
var Platform;
(function (Platform) {
    Platform["linux"] = "linux";
    Platform["mac"] = "mac";
    Platform["windows"] = "windows";
})(Platform = exports.Platform || (exports.Platform = {}));
var Product;
(function (Product) {
    Product["deadline"] = "Deadline";
    Product["deadlineDocker"] = "DeadlineDocker";
})(Product = exports.Product || (exports.Product = {}));
/**
 * The version provider parse index JSON which can be downloaded or loaded from local file
 * and returns URIs for specific product.
 * By default returns the last version of URIs or specified full or partial version.
 * If platform is not defined returns URIs for each platform.
 */
class VersionProvider {
    constructor(indexFilePath) {
        this.VALID_VERSION_REGEX = /^(0|[1-9]\d*)(?:\.(0|[1-9]\d*))?(?:\.(0|[1-9]\d*))?(?:\.(0|[1-9]\d*))?$/;
        this.indexFilePath = indexFilePath;
    }
    /**
     * Returns URIs for specified product
     *
     * @param resourceProperties
     */
    async getVersionUris(resourceProperties) {
        /* istanbul ignore next */
        const indexJson = this.indexFilePath ? this.readInstallersIndex() : await this.downloadInstallerIndex();
        const productSection = indexJson[resourceProperties.product];
        if (!productSection) {
            throw new Error(`Information about product ${resourceProperties.product} can't be found`);
        }
        let installers = new Map();
        if (resourceProperties.platform) {
            const versionedUris = await this.getUrisForPlatform(resourceProperties.product, productSection, resourceProperties.platform, resourceProperties.versionString);
            if (versionedUris) {
                installers.set(resourceProperties.platform, versionedUris);
            }
        }
        else {
            Object.values(Platform).forEach(async (p) => {
                const versionedUris = await this.getUrisForPlatform(resourceProperties.product, productSection, p, resourceProperties.versionString);
                if (versionedUris) {
                    installers.set(p, versionedUris);
                }
            });
        }
        return installers;
    }
    implementsIVersionProviderProperties(value) {
        if (!value || typeof (value) !== 'object') {
            return false;
        }
        if (!value.product || !Object.values(Product).includes(value.product)) {
            return false;
        }
        if (value.versionString) {
            if (null === this.parseVersionString(value.versionString)) {
                return false;
            }
        }
        if (value.platform) {
            if (!Object.values(Platform).includes(value.platform.toLowerCase())) {
                return false;
            }
        }
        return true;
    }
    /* istanbul ignore next */ // @ts-ignore
    async downloadInstallerIndex() {
        const productionInfoURL = 'https://downloads.thinkboxsoftware.com/version_info.json';
        const parsedUrl = url.parse(productionInfoURL);
        const options = {
            host: parsedUrl.hostname,
            path: parsedUrl.path,
        };
        return new Promise((resolve, reject) => {
            https.get(options, (res) => {
                let json = '';
                res.on('data', (chunk) => {
                    // keep appending the response chunks until we get 'end' event.
                    json += chunk;
                });
                res.on('end', () => {
                    // complete response is available here:
                    if (res.statusCode === 200) {
                        try {
                            // convert the response to a json object and return.
                            const data = JSON.parse(json);
                            resolve(data);
                        }
                        catch (e) {
                            reject(e);
                        }
                    }
                    else {
                        reject(new Error(`Expected status code 200, but got ${res.statusCode}`));
                    }
                });
            }).on('error', (err) => {
                reject(err);
            });
        });
    }
    /**
     * This method reads index file and return parsed JSON object from this file content.
     */
    readInstallersIndex() {
        if (!this.indexFilePath) {
            throw new Error('File path should be defined.');
        }
        if (!fs.existsSync(this.indexFilePath)) {
            throw new Error(`File ${this.indexFilePath} was not found`);
        }
        const data = fs.readFileSync(this.indexFilePath, 'utf8');
        // convert the response to a json object and return.
        const json = JSON.parse(data);
        return json;
    }
    parseVersionString(versionString) {
        return this.VALID_VERSION_REGEX.exec(versionString);
    }
    /**
     * This method returns IVersionedUris for specific platform
     *
     * @param product
     * @param productSection
     * @param platform
     * @param version
     */
    async getUrisForPlatform(product, productSection, platform, version) {
        const versionString = version ? version : this.getLatestVersion(platform, productSection);
        const requestedVersion = this.parseVersionString(versionString);
        // Based on the requested version, fetches the latest patch and its installer file paths.
        return this.getRequestedUriVersion(requestedVersion, productSection.versions, platform, product);
    }
    /**
     * This method returns the latest version for specified platform.
     *
     * @param platform
     * @param indexedVersionInfo
     */
    getLatestVersion(platform, indexedVersionInfo) {
        const latestSection = indexedVersionInfo.latest;
        if (!latestSection) {
            throw new Error('Information about latest version can not be found');
        }
        const latestVersion = latestSection[platform];
        if (!latestVersion) {
            throw new Error(`Information about latest version for platform ${platform} can not be found`);
        }
        return latestVersion;
    }
    /**
     * This method looks for the requested version (partial or complete) in the
     * indexed version information. Based on the input, it iterates through all
     * four numbers in the version string and compares the requested version
     * with the indexed info.
     * If any of the requested version number is missing, it fetches the latest
     * (highest) available version for it.
     *
     * @param requestedVersion
     * @param indexedVersionInfo
     */
    getRequestedUriVersion(requestedVersion, indexedVersionInfo, platform, product) {
        let versionMap = indexedVersionInfo;
        const versionArray = [];
        // iterate through all 4 major, minor, release and patch numbers,
        // and get the matching version from the indexed version map.
        for (let versionIndex = 0; versionIndex < 4; versionIndex++) {
            let version;
            if ((requestedVersion === null || requestedVersion === void 0 ? void 0 : requestedVersion[versionIndex + 1]) == null) {
                // version is not provided, get the max version.
                const numberValues = (Object.keys(versionMap)).map((val) => {
                    return parseInt(val, 10);
                });
                version = (Math.max(...numberValues)).toString();
            }
            else {
                version = requestedVersion[versionIndex + 1];
            }
            versionArray[versionIndex] = version;
            versionMap = versionMap[version];
        }
        let uriIndex;
        if ((platform in versionMap)) {
            if (product == Product.deadline) {
                const platformVersion = versionMap[platform];
                uriIndex = {
                    bundle: platformVersion.bundle,
                    clientInstaller: versionMap[platform].clientInstaller,
                    repositoryInstaller: versionMap[platform].repositoryInstaller,
                    certificateInstaller: versionMap[platform].certificateInstaller,
                };
            }
            else { // Product.deadlineDocker
                uriIndex = {
                    bundle: versionMap[platform],
                };
            }
        }
        if (uriIndex) {
            return {
                MajorVersion: versionArray[0],
                MinorVersion: versionArray[1],
                ReleaseVersion: versionArray[2],
                PatchVersion: versionArray[3],
                Uris: uriIndex,
            };
        }
        else {
            return undefined;
        }
    }
}
exports.VersionProvider = VersionProvider;
//# sourceMappingURL=data:application/json;base64,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