"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable no-console */
/* eslint-disable dot-notation */
const version_provider_1 = require("../version-provider");
const versionProvider = new version_provider_1.VersionProvider('bin/index-test.json');
const indexTest = versionProvider['readInstallersIndex']();
const productSection = indexTest[version_provider_1.Product.deadline];
test('version parsing', () => {
    const result = versionProvider['parseVersionString']('10.1.10.6');
    expect(result).not.toBeNull();
    if (result === null) {
        return;
    }
    expect(result[0]).toEqual('10.1.10.6');
    expect(result[1]).toEqual('10');
    expect(result[2]).toEqual('1');
    expect(result[3]).toEqual('10');
    expect(result[4]).toEqual('6');
});
test('partial version parsing', () => {
    const result = versionProvider['parseVersionString']('10.1');
    expect(result).not.toBeNull();
    if (result === null) {
        return;
    }
    expect(result[0]).toEqual('10.1');
    expect(result[1]).toEqual('10');
    expect(result[2]).toEqual('1');
    expect(result[3]).toBeUndefined();
    expect(result[4]).toBeUndefined();
});
test.each(['10.1.9.2.1', '10.', '10.1.', '10.-1', 'a.b.c'])('incorrect version %s parsing', (versionString) => {
    const result = versionProvider['parseVersionString'](versionString);
    expect(result).toBeNull();
});
test.each([[version_provider_1.Platform.linux, '10.1.9.2'],
    [version_provider_1.Platform.mac, '10.1.9.2'],
    [version_provider_1.Platform.windows, '10.1.8.5'],
])('latest version ', (platform, versionString) => {
    const result = versionProvider['getLatestVersion'](platform, productSection);
    expect(result).toEqual(versionString);
});
test.each([
    [version_provider_1.Platform.linux, {
            bundle: 's3://thinkbox-installers/Deadline/10.1.9.2/Linux/Deadline-10.1.9.2-linux-installers.tar',
            clientInstaller: 's3://thinkbox-installers/Deadline/10.1.9.2/Linux/DeadlineClient-10.1.9.2-linux-x64-installer.run',
            repositoryInstaller: 's3://thinkbox-installers/Deadline/10.1.9.2/Linux/DeadlineRepository-10.1.9.2-linux-x64-installer.run',
        }],
    [version_provider_1.Platform.windows, {
            bundle: 's3://thinkbox-installers/Deadline/10.1.9.2/Windows/Deadline-10.1.9.2-windows-installers.zip',
            clientInstaller: 's3://thinkbox-installers/Deadline/10.1.9.2/Windows/DeadlineClient-10.1.9.2-windows-installer.exe',
            repositoryInstaller: 's3://thinkbox-installers/Deadline/10.1.9.2/Windows/DeadlineRepository-10.1.9.2-windows-installer.exe',
        }],
    [version_provider_1.Platform.mac, {
            bundle: 's3://thinkbox-installers/Deadline/10.1.9.2/Mac/Deadline-10.1.9.2-osx-installers.dmg',
        }],
])('get Uri for platform', (platform, versionedUris) => {
    versionProvider['getUrisForPlatform'](version_provider_1.Product.deadline, productSection, platform, '10.1.9.2').then(result => {
        expect(result).not.toBeNull();
        expect(result === null || result === void 0 ? void 0 : result.Uris).toEqual(versionedUris);
    }).catch(error => {
        process.stderr.write(`${error.toString()}\n`);
        process.exit(1);
    });
});
test('get deadline version', async () => {
    const result = await versionProvider.getVersionUris({
        product: version_provider_1.Product.deadline,
        platform: version_provider_1.Platform.linux,
        versionString: '10.1',
    });
    expect(result).not.toBeNull();
    const installerVersion = result.get(version_provider_1.Platform.linux);
    expect(installerVersion).not.toBeNull();
    if (result === null) {
        return;
    }
    expect(installerVersion === null || installerVersion === void 0 ? void 0 : installerVersion.Uris).toEqual({
        bundle: 's3://thinkbox-installers/Deadline/10.1.9.2/Linux/Deadline-10.1.9.2-linux-installers.tar',
        clientInstaller: 's3://thinkbox-installers/Deadline/10.1.9.2/Linux/DeadlineClient-10.1.9.2-linux-x64-installer.run',
        repositoryInstaller: 's3://thinkbox-installers/Deadline/10.1.9.2/Linux/DeadlineRepository-10.1.9.2-linux-x64-installer.run',
    });
    expect(installerVersion === null || installerVersion === void 0 ? void 0 : installerVersion.MajorVersion).toEqual('10');
    expect(installerVersion === null || installerVersion === void 0 ? void 0 : installerVersion.MinorVersion).toEqual('1');
    expect(installerVersion === null || installerVersion === void 0 ? void 0 : installerVersion.ReleaseVersion).toEqual('9');
    expect(installerVersion === null || installerVersion === void 0 ? void 0 : installerVersion.PatchVersion).toEqual('2');
});
test('product is not in file', async () => {
    await expect(versionProvider.getVersionUris({
        product: version_provider_1.Product.deadlineDocker,
    })).rejects.toThrowError(/Information about product DeadlineDocker can't be found/);
});
test('get deadline version for all platforms', async () => {
    const result = await versionProvider.getVersionUris({
        product: version_provider_1.Product.deadline,
    });
    expect(result).not.toBeNull();
    const linuxInstallerVersion = result.get(version_provider_1.Platform.linux);
    expect(linuxInstallerVersion).not.toBeNull();
    if (result === null) {
        return;
    }
    expect(linuxInstallerVersion === null || linuxInstallerVersion === void 0 ? void 0 : linuxInstallerVersion.Uris).toEqual({
        bundle: 's3://thinkbox-installers/Deadline/10.1.9.2/Linux/Deadline-10.1.9.2-linux-installers.tar',
        clientInstaller: 's3://thinkbox-installers/Deadline/10.1.9.2/Linux/DeadlineClient-10.1.9.2-linux-x64-installer.run',
        repositoryInstaller: 's3://thinkbox-installers/Deadline/10.1.9.2/Linux/DeadlineRepository-10.1.9.2-linux-x64-installer.run',
    });
    expect(linuxInstallerVersion === null || linuxInstallerVersion === void 0 ? void 0 : linuxInstallerVersion.MajorVersion).toEqual('10');
    expect(linuxInstallerVersion === null || linuxInstallerVersion === void 0 ? void 0 : linuxInstallerVersion.MinorVersion).toEqual('1');
    expect(linuxInstallerVersion === null || linuxInstallerVersion === void 0 ? void 0 : linuxInstallerVersion.ReleaseVersion).toEqual('9');
    expect(linuxInstallerVersion === null || linuxInstallerVersion === void 0 ? void 0 : linuxInstallerVersion.PatchVersion).toEqual('2');
    const macInstallerVersion = result.get(version_provider_1.Platform.mac);
    expect(macInstallerVersion).not.toBeNull();
    if (result === null) {
        return;
    }
    expect(macInstallerVersion === null || macInstallerVersion === void 0 ? void 0 : macInstallerVersion.Uris).toEqual({
        bundle: 's3://thinkbox-installers/Deadline/10.1.9.2/Mac/Deadline-10.1.9.2-osx-installers.dmg',
    });
    expect(macInstallerVersion === null || macInstallerVersion === void 0 ? void 0 : macInstallerVersion.MajorVersion).toEqual('10');
    expect(macInstallerVersion === null || macInstallerVersion === void 0 ? void 0 : macInstallerVersion.MinorVersion).toEqual('1');
    expect(macInstallerVersion === null || macInstallerVersion === void 0 ? void 0 : macInstallerVersion.ReleaseVersion).toEqual('9');
    expect(macInstallerVersion === null || macInstallerVersion === void 0 ? void 0 : macInstallerVersion.PatchVersion).toEqual('2');
    const windowsInstallerVersion = result.get(version_provider_1.Platform.windows);
    expect(windowsInstallerVersion).not.toBeNull();
    if (result === null) {
        return;
    }
    expect(windowsInstallerVersion === null || windowsInstallerVersion === void 0 ? void 0 : windowsInstallerVersion.Uris).toEqual({
        bundle: 's3://thinkbox-installers/Deadline/10.1.8.5/Windows/Deadline-10.1.8.5-windows-installers.zip',
        clientInstaller: 's3://thinkbox-installers/Deadline/10.1.8.5/Windows/DeadlineClient-10.1.8.5-windows-installer.exe',
        repositoryInstaller: 's3://thinkbox-installers/Deadline/10.1.8.5/Windows/DeadlineRepository-10.1.8.5-windows-installer.exe',
    });
    expect(windowsInstallerVersion === null || windowsInstallerVersion === void 0 ? void 0 : windowsInstallerVersion.MajorVersion).toEqual('10');
    expect(windowsInstallerVersion === null || windowsInstallerVersion === void 0 ? void 0 : windowsInstallerVersion.MinorVersion).toEqual('1');
    expect(windowsInstallerVersion === null || windowsInstallerVersion === void 0 ? void 0 : windowsInstallerVersion.ReleaseVersion).toEqual('8');
    expect(windowsInstallerVersion === null || windowsInstallerVersion === void 0 ? void 0 : windowsInstallerVersion.PatchVersion).toEqual('5');
});
test('validate correct input', async () => {
    expect(versionProvider.implementsIVersionProviderProperties({
        product: version_provider_1.Product.deadline,
        versionString: '10.1.9.2',
        platform: 'linux',
    })).toBeTruthy();
});
test('validate non-object input', async () => {
    expect(versionProvider['implementsIVersionProviderProperties']('test')).toEqual(false);
});
test('validate input without product', async () => {
    expect(versionProvider.implementsIVersionProviderProperties({
        versionString: 'version',
    })).toEqual(false);
});
test('validate input with invalid versionString', async () => {
    expect(versionProvider.implementsIVersionProviderProperties({
        product: version_provider_1.Product.deadline,
        versionString: 'version',
    })).toEqual(false);
});
test('validate input with invalid platform', async () => {
    expect(versionProvider['implementsIVersionProviderProperties']({
        product: version_provider_1.Product.deadline,
        platform: 'test',
    })).toEqual(false);
});
test('not defined file path', () => {
    expect(() => (new version_provider_1.VersionProvider())['readInstallersIndex']()).toThrowError(/File path should be defined./);
});
test('invalid file path', () => {
    expect(() => (new version_provider_1.VersionProvider('test.txt'))['readInstallersIndex']()).toThrowError(/File test.txt was not found/);
});
test('get latest version without latest section', () => {
    expect(() => versionProvider['getLatestVersion']('linux', {})).toThrowError(/Information about latest version can not be found/);
});
test('get latest version without informtion for platform', () => {
    expect(() => versionProvider['getLatestVersion']('linux', { latest: {} })).toThrowError(/Information about latest version for platform linux can not be found/);
});
test('get requested Uri version for existing product.', () => {
    const requestedVersion = versionProvider['parseVersionString']('10.1.9.2');
    expect(versionProvider['getRequestedUriVersion'](requestedVersion, {
        10: {
            1: {
                9: {
                    2: {
                        linux: 's3://thinkbox-installers/DeadlineDocker/10.1.9.2/DeadlineDocker-10.1.9.2.tar.gz',
                    },
                },
            },
        }
    }, version_provider_1.Platform.linux, version_provider_1.Product.deadlineDocker)).toEqual({
        MajorVersion: '10',
        MinorVersion: '1',
        ReleaseVersion: '9',
        PatchVersion: '2',
        Uris: { bundle: 's3://thinkbox-installers/DeadlineDocker/10.1.9.2/DeadlineDocker-10.1.9.2.tar.gz' },
    });
});
test('get requested Uri version for not existing product.', () => {
    const requestedVersion = versionProvider['parseVersionString']('10.1.9.2');
    expect(versionProvider['getRequestedUriVersion'](requestedVersion, {
        10: {
            1: {
                9: {
                    2: {
                        linux: 's3://thinkbox-installers/DeadlineDocker/10.1.9.2/DeadlineDocker-10.1.9.2.tar.gz',
                    },
                },
            },
        }
    }, version_provider_1.Platform.windows, version_provider_1.Product.deadlineDocker)).toEqual(undefined);
});
//# sourceMappingURL=data:application/json;base64,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