"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable no-console */
// eslint-disable-next-line import/no-extraneous-dependencies
const crypto_1 = require("crypto");
const AWS = require("aws-sdk");
const aws_sdk_mock_1 = require("aws-sdk-mock");
const dynalite = require("dynalite");
const sinon_1 = require("sinon");
const composite_table_1 = require("../composite-table");
// Enable/disable debugging statements.
const DEBUG = false;
if (!DEBUG) {
    console.debug = () => { };
}
// Test class to expose constructor and tableName
class TestTable extends composite_table_1.CompositeStringIndexTable {
    constructor(client, name, primaryKey, sortKey) {
        super(client, name, primaryKey, sortKey);
    }
}
/**
 * This class is being refactored to use mocks over an actualy connection to DynamoDB. All the DynamoDB tests were
 * converted to use dynalite, a package that sets up a server to replicate DynamoDB locally. These tests can be found
 * in the first describe() block. Newer tests have been written using aws-sdk-mock and found in the describe() block
 * that follows. Any updates to these tests should translate dynalite tests to aws-sdk-mocks and new tests should use
 * aws-sdk-mock.
 */
describe('Tests using dynalite', () => {
    const SUFFIX = crypto_1.randomBytes(8).toString('hex');
    const GOOD_TABLE_NAME = `RFDKTestGood-${SUFFIX}`;
    const BAD_TABLE1_NAME = `RFDKTestBad1-${SUFFIX}`; // Missing the sort key
    const BAD_TABLE2_NAME = `RFDKTestBad2-${SUFFIX}`; // Incorrect type on primary key
    const BAD_TABLE3_NAME = `RFDKTestBad3-${SUFFIX}`; // Incorrect type on sort key
    const dynaliteServer = dynalite({
        createTableMs: 5,
        deleteTableMs: 5,
        updateTableMs: 5,
    });
    let dynamoClient;
    beforeAll(async () => {
        const dynaPort = 43266;
        await dynaliteServer.listen(dynaPort, (err) => {
            if (err) {
                throw err;
            }
        });
        dynamoClient = new AWS.DynamoDB({
            credentials: new AWS.Credentials({
                accessKeyId: '',
                secretAccessKey: '',
            }),
            endpoint: `http://localhost:${dynaPort}`,
            region: 'us-west-2',
        });
        function createTableRequest(tableName, primaryKeyType, sortKey) {
            const request = {
                TableName: tableName,
                AttributeDefinitions: [
                    {
                        AttributeName: 'PrimKey',
                        AttributeType: primaryKeyType,
                    },
                ],
                KeySchema: [
                    {
                        AttributeName: 'PrimKey',
                        KeyType: 'HASH',
                    },
                ],
                BillingMode: 'PAY_PER_REQUEST',
                Tags: [
                    {
                        Key: 'RFDKTesting',
                        Value: 'RFDKTesting',
                    },
                ],
            };
            if (sortKey) {
                request.AttributeDefinitions.push({
                    AttributeName: 'SortKey',
                    AttributeType: sortKey.KeyType,
                });
                request.KeySchema.push({
                    AttributeName: 'SortKey',
                    KeyType: 'RANGE',
                });
            }
            return request;
        }
        let request = createTableRequest(GOOD_TABLE_NAME, 'S', { KeyType: 'S' });
        let response = await dynamoClient.createTable(request).promise();
        let table = response.TableDescription;
        if (!table) {
            throw Error(`Could not create ${GOOD_TABLE_NAME}`);
        }
        console.debug(`Created DynamoDB table: '${table.TableName}'`);
        request = createTableRequest(BAD_TABLE1_NAME, 'S');
        response = await dynamoClient.createTable(request).promise();
        table = response.TableDescription;
        if (!table) {
            throw Error(`Could not create ${BAD_TABLE1_NAME}`);
        }
        console.debug(`Created DynamoDB table: '${table.TableName}'`);
        request = createTableRequest(BAD_TABLE2_NAME, 'N', { KeyType: 'S' });
        response = await dynamoClient.createTable(request).promise();
        table = response.TableDescription;
        if (!table) {
            throw Error(`Could not create ${BAD_TABLE2_NAME}`);
        }
        console.debug(`Created DynamoDB table: '${table.TableName}'`);
        request = createTableRequest(BAD_TABLE3_NAME, 'S', { KeyType: 'N' });
        response = await dynamoClient.createTable(request).promise();
        table = response.TableDescription;
        if (!table) {
            throw Error(`Could not create ${BAD_TABLE3_NAME}`);
        }
        console.debug(`Created DynamoDB table: '${table.TableName}'`);
        let waiting = true;
        do {
            const promises = [];
            for (const name of [GOOD_TABLE_NAME, BAD_TABLE1_NAME, BAD_TABLE2_NAME, BAD_TABLE3_NAME]) {
                promises.push(dynamoClient.describeTable({
                    TableName: name,
                }).promise());
            }
            const responses = await Promise.all(promises);
            waiting = !responses.every(item => { var _a; return ((_a = item.Table) === null || _a === void 0 ? void 0 : _a.TableStatus) === 'ACTIVE'; });
            await new Promise(resolve => setTimeout(resolve, 2000));
        } while (waiting);
    }, 60000);
    afterAll(async () => {
        await dynaliteServer.close();
    });
    test('delete table ResourceNotFoundException', async () => {
        const tableName = 'NonExistantTable';
        const pk = 'PrimKey';
        const sk = 'SortKey';
        const table = new TestTable(dynamoClient, tableName, pk, sk);
        expect(table.tableName).toBeDefined();
        await expect(table.deleteTable()).resolves.not.toThrowError();
        expect(table.tableName).toBeUndefined();
    });
    test.each([
        [GOOD_TABLE_NAME, false],
        [BAD_TABLE1_NAME, true],
        [BAD_TABLE2_NAME, true],
        [BAD_TABLE3_NAME, true],
    ])('fromExisting table %p', async (tableName, expectedToThrow) => {
        if (expectedToThrow) {
            await expect(composite_table_1.CompositeStringIndexTable.fromExisting(dynamoClient, tableName)).rejects.toThrow();
        }
        else {
            await expect(composite_table_1.CompositeStringIndexTable.fromExisting(dynamoClient, tableName)).resolves.not.toThrow();
            const table = await composite_table_1.CompositeStringIndexTable.fromExisting(dynamoClient, tableName);
            expect(table.primaryKey).toBe('PrimKey');
            expect(table.sortKey).toBe('SortKey');
        }
    });
    test('putItem/getItem success', async () => {
        // GIVEN
        const table = await composite_table_1.CompositeStringIndexTable.fromExisting(dynamoClient, GOOD_TABLE_NAME);
        // WHEN
        const attributes = {
            TestAttribute: 'test value',
        };
        const putRequest = {
            primaryKeyValue: crypto_1.randomBytes(24).toString('hex'),
            sortKeyValue: crypto_1.randomBytes(24).toString('hex'),
            attributes,
        };
        const getRequest = {
            primaryKeyValue: putRequest.primaryKeyValue,
            sortKeyValue: putRequest.sortKeyValue,
        };
        // THEN
        await expect(table.putItem(putRequest)).resolves.toBe(true);
        await expect(table.getItem(getRequest)).resolves.toMatchObject(attributes);
    });
    test.each([
        [true], [false],
    ])('putItem overwrite %p', async (allowOverwrite) => {
        // GIVEN
        const table = await composite_table_1.CompositeStringIndexTable.fromExisting(dynamoClient, GOOD_TABLE_NAME);
        // WHEN
        const putRequest = {
            primaryKeyValue: crypto_1.randomBytes(24).toString('hex'),
            sortKeyValue: crypto_1.randomBytes(24).toString('hex'),
            attributes: {
                TestAttribute: 'test value',
            },
            allow_overwrite: true,
        };
        await table.putItem(putRequest);
        // THEN
        putRequest.allow_overwrite = allowOverwrite;
        await expect(table.putItem(putRequest)).resolves.toBe(allowOverwrite);
    });
    test('getItem no match', async () => {
        // GIVEN
        const table = await composite_table_1.CompositeStringIndexTable.fromExisting(dynamoClient, GOOD_TABLE_NAME);
        // WHEN
        const getRequest = {
            primaryKeyValue: crypto_1.randomBytes(24).toString('hex'),
            sortKeyValue: crypto_1.randomBytes(24).toString('hex'),
        };
        // THEN
        await expect(table.getItem(getRequest)).resolves.toBeUndefined();
    });
    test('deleteItem success', async () => {
        // GIVEN
        const table = await composite_table_1.CompositeStringIndexTable.fromExisting(dynamoClient, GOOD_TABLE_NAME);
        // WHEN
        const attributes = {
            TestAttribute: 'test value',
        };
        const putRequest = {
            primaryKeyValue: crypto_1.randomBytes(24).toString('hex'),
            sortKeyValue: crypto_1.randomBytes(24).toString('hex'),
            attributes,
        };
        const deleteRequest = {
            primaryKeyValue: putRequest.primaryKeyValue,
            sortKeyValue: putRequest.sortKeyValue,
        };
        await table.putItem(putRequest);
        // THEN
        await expect(table.deleteItem(deleteRequest)).resolves.toBe(true);
    });
    test('deleteItem no match', async () => {
        // GIVEN
        const table = await composite_table_1.CompositeStringIndexTable.fromExisting(dynamoClient, GOOD_TABLE_NAME);
        // WHEN
        const deleteRequest = {
            primaryKeyValue: crypto_1.randomBytes(24).toString('hex'),
            sortKeyValue: crypto_1.randomBytes(24).toString('hex'),
        };
        // THEN
        await expect(table.deleteItem(deleteRequest)).resolves.toBe(false);
    });
    test('query success', async () => {
        // GIVEN
        const table = await composite_table_1.CompositeStringIndexTable.fromExisting(dynamoClient, GOOD_TABLE_NAME);
        // WHEN
        const primaryKeyValue = crypto_1.randomBytes(24).toString('hex');
        const expected = {};
        for (let i = 0; i < 20; i++) {
            const sortKeyValue = crypto_1.randomBytes(24).toString('hex');
            expected[sortKeyValue] = {
                TestAttribute: `value${i}`,
            };
        }
        for (const [key, value] of Object.entries(expected)) {
            const putRequest = {
                primaryKeyValue,
                sortKeyValue: key,
                attributes: value,
            };
            await table.putItem(putRequest);
        }
        // THEN
        await expect(table.query(primaryKeyValue, 5)).resolves.toMatchObject(expected);
    });
});
describe('Tests using aws-sdk-mock', () => {
    beforeEach(() => {
        aws_sdk_mock_1.setSDKInstance(AWS);
    });
    afterEach(() => {
        aws_sdk_mock_1.restore('DynamoDB');
    });
    describe('fromExisting tests', () => {
        test('Table not found', async () => {
            const callback = sinon_1.fake.resolves({ Table: undefined });
            aws_sdk_mock_1.mock('DynamoDB', 'describeTable', callback);
            const client = new AWS.DynamoDB();
            const tableName = 'Nonexistant';
            await expect(composite_table_1.CompositeStringIndexTable.fromExisting(client, tableName))
                .rejects
                .toThrowError(`Could not describeTable for Table '${tableName}'`);
        });
        test('KeySchema not found', async () => {
            aws_sdk_mock_1.mock('DynamoDB', 'describeTable', sinon_1.fake.resolves({ Table: { KeySchema: undefined } }));
            const client = new AWS.DynamoDB();
            const tableName = 'TestTable';
            await expect(composite_table_1.CompositeStringIndexTable.fromExisting(client, tableName))
                .rejects
                .toThrowError(`Could not get KeySchema for Table '${tableName}'`);
        });
        test('AttributeDefinitions not found', async () => {
            aws_sdk_mock_1.mock('DynamoDB', 'describeTable', sinon_1.fake.resolves({
                Table: {
                    KeySchema: [],
                    AttributeDefinitions: undefined,
                },
            }));
            const client = new AWS.DynamoDB();
            const tableName = 'TestTable';
            await expect(composite_table_1.CompositeStringIndexTable.fromExisting(client, tableName))
                .rejects
                .toThrowError(`Could not get Attributes for Table '${tableName}'`);
        });
        test('PrimaryKey not found', async () => {
            aws_sdk_mock_1.mock('DynamoDB', 'describeTable', sinon_1.fake.resolves({
                Table: {
                    KeySchema: [
                        {
                            AttributeName: 'SortKey',
                            KeyType: 'RANGE',
                        },
                    ],
                    AttributeDefinitions: {},
                },
            }));
            const client = new AWS.DynamoDB();
            const tableName = 'TestTable';
            await expect(composite_table_1.CompositeStringIndexTable.fromExisting(client, tableName))
                .rejects
                .toThrowError(`Could not find PrimaryKey of Table '${tableName}'`);
        });
        test('SortKey not found', async () => {
            aws_sdk_mock_1.mock('DynamoDB', 'describeTable', sinon_1.fake.resolves({
                Table: {
                    KeySchema: [
                        {
                            AttributeName: 'PrimaryKey',
                            KeyType: 'HASH',
                        },
                    ],
                    AttributeDefinitions: {},
                },
            }));
            const client = new AWS.DynamoDB();
            const tableName = 'TestTable';
            await expect(composite_table_1.CompositeStringIndexTable.fromExisting(client, tableName))
                .rejects
                .toThrowError(`Could not find SortKey of Table '${tableName}'`);
        });
        test('PrimaryKey AttributeDefinition not found', async () => {
            aws_sdk_mock_1.mock('DynamoDB', 'describeTable', sinon_1.fake.resolves({
                Table: {
                    KeySchema: [
                        {
                            AttributeName: 'PrimaryKey',
                            KeyType: 'HASH',
                        },
                        {
                            AttributeName: 'SortKey',
                            KeyType: 'RANGE',
                        },
                    ],
                    AttributeDefinitions: [
                        {
                            AttributeName: 'SortKey',
                            AttributeType: 'S',
                        },
                    ],
                },
            }));
            const client = new AWS.DynamoDB();
            const tableName = 'TestTable';
            await expect(composite_table_1.CompositeStringIndexTable.fromExisting(client, tableName))
                .rejects
                .toThrowError("Primary key 'PrimaryKey' must be string type");
        });
        test('SortKey AttributeDefinition not found', async () => {
            aws_sdk_mock_1.mock('DynamoDB', 'describeTable', sinon_1.fake.resolves({
                Table: {
                    KeySchema: [
                        {
                            AttributeName: 'PrimaryKey',
                            KeyType: 'HASH',
                        },
                        {
                            AttributeName: 'SortKey',
                            KeyType: 'RANGE',
                        },
                    ],
                    AttributeDefinitions: [
                        {
                            AttributeName: 'PrimaryKey',
                            AttributeType: 'S',
                        },
                    ],
                },
            }));
            const client = new AWS.DynamoDB();
            const tableName = 'TestTable';
            await expect(composite_table_1.CompositeStringIndexTable.fromExisting(client, tableName))
                .rejects
                .toThrowError("Sort key 'SortKey' must be string type");
        });
    });
    describe('createNew tests', () => {
        test('DynamoDB.createTable() failure throws Error', async () => {
            aws_sdk_mock_1.mock('DynamoDB', 'createTable', sinon_1.fake.rejects({}));
            const client = new AWS.DynamoDB();
            const name = 'TestTable';
            const primaryKeyName = 'PrimaryKey';
            const sortKeyName = 'SortKey';
            await expect(composite_table_1.CompositeStringIndexTable.createNew({
                client,
                name,
                primaryKeyName,
                sortKeyName,
            }))
                .rejects
                .toThrow();
        });
        test('success', async () => {
            const tableName = 'TestTable';
            const pk = 'PrimKey';
            const sk = 'SortKey';
            const tags = [
                {
                    Key: 'RFDKTesting',
                    Value: 'RFDKTesting',
                },
            ];
            aws_sdk_mock_1.mock('DynamoDB', 'createTable', sinon_1.fake.resolves({}));
            const client = new AWS.DynamoDB();
            const table = await composite_table_1.CompositeStringIndexTable.createNew({
                client,
                name: tableName,
                primaryKeyName: pk,
                sortKeyName: sk,
                tags,
            });
            expect(table.primaryKey).toBe(pk);
            expect(table.sortKey).toBe(sk);
        });
    });
    describe('deleteTable tests', () => {
        test('success', async () => {
            const tableName = 'TestTable';
            const pk = 'PrimKey';
            const sk = 'SortKey';
            aws_sdk_mock_1.mock('DynamoDB', 'deleteTable', sinon_1.fake.resolves({}));
            const client = new AWS.DynamoDB();
            const table = new TestTable(client, tableName, pk, sk);
            await expect(table.deleteTable()).resolves.not.toThrow();
        });
        test('Table already deleted', async () => {
            const deleteFake = sinon_1.fake.resolves({});
            aws_sdk_mock_1.mock('DynamoDB', 'deleteTable', deleteFake);
            const client = new AWS.DynamoDB();
            const name = 'TestTable';
            const primaryKeyName = 'PrimaryKey';
            const sortKeyName = 'SortKey';
            const subject = new TestTable(client, name, primaryKeyName, sortKeyName);
            await subject.deleteTable();
            await expect(subject.deleteTable()).resolves.toBe(undefined);
            expect(deleteFake.callCount).toEqual(1);
        });
        test('DynamoDB.deleteTable() failure', async () => {
            aws_sdk_mock_1.mock('DynamoDB', 'deleteTable', sinon_1.fake.rejects(new Error()));
            const client = new AWS.DynamoDB();
            const name = 'TestTable';
            const primaryKeyName = 'PrimaryKey';
            const sortKeyName = 'SortKey';
            const subject = new TestTable(client, name, primaryKeyName, sortKeyName);
            await expect(subject.deleteTable()).rejects.toThrowError();
        });
    });
    describe('putItem tests', () => {
        test('Table already deleted', async () => {
            const deleteFake = sinon_1.fake.resolves({});
            aws_sdk_mock_1.mock('DynamoDB', 'deleteTable', deleteFake);
            const putFake = sinon_1.fake.resolves({});
            aws_sdk_mock_1.mock('DynamoDB', 'putItem', putFake);
            const client = new AWS.DynamoDB();
            const name = 'TestTable';
            const primaryKeyName = 'PrimaryKey';
            const sortKeyName = 'SortKey';
            const subject = new TestTable(client, name, primaryKeyName, sortKeyName);
            await subject.deleteTable();
            await expect(subject.putItem({ primaryKeyValue: 'TestPrimVal', sortKeyValue: 'TestSortVal' }))
                .rejects
                .toThrowError('Attempt to PutItem in deleted table');
            expect(deleteFake.callCount).toEqual(1);
            expect(putFake.notCalled).toBeTruthy();
        });
        test('DynamoDB.putItem() failure', async () => {
            aws_sdk_mock_1.mock('DynamoDB', 'putItem', sinon_1.fake.rejects(new Error()));
            const client = new AWS.DynamoDB();
            const name = 'TestTable';
            const primaryKeyName = 'PrimaryKey';
            const sortKeyName = 'SortKey';
            const subject = new TestTable(client, name, primaryKeyName, sortKeyName);
            await expect(subject.putItem({ primaryKeyValue: 'TestPrimVal', sortKeyValue: 'TestSortVal' })).rejects.toThrowError();
        });
    });
    describe('getItem tests', () => {
        test('Table already deleted', async () => {
            const deleteFake = sinon_1.fake.resolves({});
            aws_sdk_mock_1.mock('DynamoDB', 'deleteTable', deleteFake);
            const getFake = sinon_1.fake.resolves({});
            aws_sdk_mock_1.mock('DynamoDB', 'getItem', getFake);
            const client = new AWS.DynamoDB();
            const name = 'TestTable';
            const primaryKeyName = 'PrimaryKey';
            const sortKeyName = 'SortKey';
            const subject = new TestTable(client, name, primaryKeyName, sortKeyName);
            await subject.deleteTable();
            await expect(subject.getItem({ primaryKeyValue: 'TestPrimVal', sortKeyValue: 'TestSortVal' }))
                .rejects
                .toThrowError('Attempt to GetItem from deleted table');
            expect(deleteFake.callCount).toEqual(1);
            expect(getFake.notCalled).toBeTruthy();
        });
        test('DynamoDB.getItem() failure', async () => {
            aws_sdk_mock_1.mock('DynamoDB', 'getItem', sinon_1.fake.rejects(new Error()));
            const client = new AWS.DynamoDB();
            const name = 'TestTable';
            const primaryKeyName = 'PrimaryKey';
            const sortKeyName = 'SortKey';
            const subject = new TestTable(client, name, primaryKeyName, sortKeyName);
            await expect(subject.getItem({ primaryKeyValue: 'TestPrimVal', sortKeyValue: 'TestSortVal' })).rejects.toThrowError();
        });
    });
    describe('deleteItem tests', () => {
        test('Table already deleted', async () => {
            const deleteTableFake = sinon_1.fake.resolves({});
            aws_sdk_mock_1.mock('DynamoDB', 'deleteTable', deleteTableFake);
            const deleteItemFake = sinon_1.fake.resolves({});
            aws_sdk_mock_1.mock('DynamoDB', 'deleteItem', deleteItemFake);
            const client = new AWS.DynamoDB();
            const name = 'TestTable';
            const primaryKeyName = 'PrimaryKey';
            const sortKeyName = 'SortKey';
            const subject = new TestTable(client, name, primaryKeyName, sortKeyName);
            await subject.deleteTable();
            await expect(subject.deleteItem({ primaryKeyValue: 'TestPrimVal', sortKeyValue: 'TestSortVal' }))
                .rejects
                .toThrowError('Attempt to DeleteItem from deleted table');
            expect(deleteTableFake.callCount).toEqual(1);
            expect(deleteItemFake.notCalled).toBeTruthy();
        });
        test('DynamoDB.deleteItem() failure', async () => {
            aws_sdk_mock_1.mock('DynamoDB', 'deleteItem', sinon_1.fake.rejects(new Error()));
            const client = new AWS.DynamoDB();
            const name = 'TestTable';
            const primaryKeyName = 'PrimaryKey';
            const sortKeyName = 'SortKey';
            const subject = new TestTable(client, name, primaryKeyName, sortKeyName);
            await expect(subject.deleteItem({ primaryKeyValue: 'TestPrimVal', sortKeyValue: 'TestSortVal' })).rejects.toThrowError();
        });
    });
    describe('query tests', () => {
        test('Returns empty', async () => {
            const queryFake = sinon_1.fake.resolves({});
            aws_sdk_mock_1.mock('DynamoDB', 'query', queryFake);
            const client = new AWS.DynamoDB();
            const name = 'TestTable';
            const primaryKeyName = 'PrimaryKey';
            const sortKeyName = 'SortKey';
            const subject = new TestTable(client, name, primaryKeyName, sortKeyName);
            await expect(subject.query('TestPrimVal'))
                .resolves
                .toEqual({});
            expect(queryFake.callCount).toEqual(1);
        });
        test('Table already deleted', async () => {
            const deleteTableFake = sinon_1.fake.resolves({});
            aws_sdk_mock_1.mock('DynamoDB', 'deleteTable', deleteTableFake);
            const queryFake = sinon_1.fake.resolves({});
            aws_sdk_mock_1.mock('DynamoDB', 'query', queryFake);
            const client = new AWS.DynamoDB();
            const name = 'TestTable';
            const primaryKeyName = 'PrimaryKey';
            const sortKeyName = 'SortKey';
            const subject = new TestTable(client, name, primaryKeyName, sortKeyName);
            await subject.deleteTable();
            await expect(subject.query('TestPrimVal'))
                .rejects
                .toThrowError('Attempt to Query a deleted table');
            expect(deleteTableFake.callCount).toEqual(1);
            expect(queryFake.notCalled).toBeTruthy();
        });
        test('DynamoDB.query() failure', async () => {
            aws_sdk_mock_1.mock('DynamoDB', 'query', sinon_1.fake.rejects(new Error()));
            const client = new AWS.DynamoDB();
            const name = 'TestTable';
            const primaryKeyName = 'PrimaryKey';
            const sortKeyName = 'SortKey';
            const subject = new TestTable(client, name, primaryKeyName, sortKeyName);
            await expect(subject.query('TestPrimVal')).rejects.toThrowError();
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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