"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CompositeStringIndexTable = void 0;
/* eslint-disable no-console */
// eslint-disable-next-line import/no-extraneous-dependencies
const aws_sdk_1 = require("aws-sdk");
class CompositeStringIndexTable {
    constructor(client, name, primaryKey, sortKey) {
        this.client = client;
        this.tableName = name;
        this.primaryKey = primaryKey;
        this.sortKey = sortKey;
    }
    static async fromExisting(client, tableName) {
        // Determine the key schema of the table
        // We let this throw if the table does not exist.
        var _a, _b;
        // See: https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/DynamoDB.html#describeTable-property
        const describeResponse = await client.describeTable({ TableName: tableName }).promise();
        if (!describeResponse.Table) {
            throw Error(`Could not describeTable for Table '${tableName}'`);
        }
        const keySchema = describeResponse.Table.KeySchema;
        if (!keySchema) {
            throw Error(`Could not get KeySchema for Table '${tableName}'`);
        }
        const attributes = describeResponse.Table.AttributeDefinitions;
        if (!attributes) {
            throw Error(`Could not get Attributes for Table '${tableName}'`);
        }
        // Find the names of the Primary & Sort keys.
        const hashIndex = keySchema.findIndex(item => item.KeyType === 'HASH');
        const sortIndex = keySchema.findIndex(item => item.KeyType === 'RANGE');
        if (hashIndex < 0 || sortIndex < 0) {
            console.debug(`Error initializing DynamoDatabase. KeySchema: ${JSON.stringify(keySchema)}`);
            if (hashIndex < 0) {
                throw Error(`Could not find PrimaryKey of Table '${tableName}'`);
            }
            else {
                throw Error(`Could not find SortKey of Table '${tableName}'`);
            }
        }
        const primaryKey = keySchema[hashIndex].AttributeName;
        const sortKey = keySchema[sortIndex].AttributeName;
        // Make sure that the primary & sort key types are both string types
        // (( We didn't make this flexible enough for other attribute types for the key ))
        if ('S' !== ((_a = attributes.find(item => item.AttributeName === primaryKey)) === null || _a === void 0 ? void 0 : _a.AttributeType)) {
            throw Error(`Primary key '${primaryKey}' must be string type`);
        }
        if ('S' !== ((_b = attributes.find(item => item.AttributeName === sortKey)) === null || _b === void 0 ? void 0 : _b.AttributeType)) {
            throw Error(`Sort key '${sortKey}' must be string type`);
        }
        return new CompositeStringIndexTable(client, tableName, primaryKey, sortKey);
    }
    /**
     * A simplified interface to create a new DynamoDB Table with a composite index
     * consisting of a pair of string attributes.
     * @param args
     */
    static async createNew(args) {
        // See: https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/DynamoDB.html#createTable-property
        const request = {
            TableName: args.name,
            AttributeDefinitions: [
                {
                    AttributeName: args.primaryKeyName,
                    AttributeType: 'S',
                },
                {
                    AttributeName: args.sortKeyName,
                    AttributeType: 'S',
                },
            ],
            KeySchema: [
                {
                    AttributeName: args.primaryKeyName,
                    KeyType: 'HASH',
                },
                {
                    AttributeName: args.sortKeyName,
                    KeyType: 'RANGE',
                },
            ],
            BillingMode: 'PAY_PER_REQUEST',
            Tags: args.tags,
        };
        try {
            await args.client.createTable(request).promise();
            const table = new CompositeStringIndexTable(args.client, args.name, args.primaryKeyName, args.sortKeyName);
            return table;
        }
        catch (e) {
            throw new Error(`CreateTable '${args.name}': ${e.code} -- ${e.message}`);
        }
    }
    /**
     * Delete this table from DynamoDB.
     */
    async deleteTable() {
        if (!this.tableName) {
            return; // Already gone.
        }
        // See: https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/DynamoDB.html#deleteTable-property
        const request = {
            TableName: this.tableName,
        };
        try {
            await this.client.deleteTable(request).promise();
            this.tableName = undefined;
        }
        catch (e) {
            if (e.code === 'ResourceNotFoundException') {
                // Already gone. We're good.
                this.tableName = undefined;
            }
            else {
                throw new Error(`DeleteTable '${this.tableName}': ${e.code} -- ${e.message}`);
            }
        }
    }
    /**
     * Puts an item into the Table. The item it put into the table index with the given
     * primary and sort key attribute values. If any attributes are given, then they are
     * stored in the item.
     *
     * @param props
     * @throws Error if the request fails.
     * @returns True if the item was stored in the table; false otherwise
     */
    async putItem(props) {
        var _a;
        if (!this.tableName) {
            throw Error('Attempt to PutItem in deleted table');
        }
        // See: https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/DynamoDB.html#putItem-property
        //      https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/DynamoDB/Converter.html
        const item = aws_sdk_1.DynamoDB.Converter.marshall((_a = props.attributes) !== null && _a !== void 0 ? _a : {});
        item[this.primaryKey] = aws_sdk_1.DynamoDB.Converter.input(props.primaryKeyValue);
        item[this.sortKey] = aws_sdk_1.DynamoDB.Converter.input(props.sortKeyValue);
        const request = {
            TableName: this.tableName,
            Item: item,
            ReturnConsumedCapacity: 'NONE',
            ReturnItemCollectionMetrics: 'NONE',
            ReturnValues: 'NONE',
        };
        if (!props.allow_overwrite) {
            request.ConditionExpression = `attribute_not_exists(${this.primaryKey}) AND attribute_not_exists(${this.sortKey})`;
        }
        try {
            console.debug(`Dynamo.PutItem request: ${JSON.stringify(request)}`);
            const response = await this.client.putItem(request).promise();
            console.debug(`PutItem response: ${JSON.stringify(response)}`);
        }
        catch (e) {
            if (e.code === 'ConditionalCheckFailedException' && !props.allow_overwrite) {
                return false;
            }
            throw new Error(`PutItem '${props.primaryKeyValue}' '${props.sortKeyValue}:" ` +
                `${e.code} -- ${e.message}`);
        }
        return true;
    }
    /**
     * Does a consistent read to get the item from the Table with the given primary and sort key, if one exists.
     *
     * @param props
     * @throws Error if the request fails.
     * @returns The attributes of the item **excluding** the primary and sort key, or undefined if there was no item
     *         found.
     */
    async getItem(props) {
        if (!this.tableName) {
            throw Error('Attempt to GetItem from deleted table');
        }
        // See: https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/DynamoDB.html#getItem-property
        const key = {};
        key[this.primaryKey] = props.primaryKeyValue;
        key[this.sortKey] = props.sortKeyValue;
        const request = {
            TableName: this.tableName,
            Key: aws_sdk_1.DynamoDB.Converter.marshall(key),
            ConsistentRead: true,
            ReturnConsumedCapacity: 'NONE',
        };
        try {
            console.debug(`Dynamo.GetItem request: ${JSON.stringify(request)}`);
            const response = await this.client.getItem(request).promise();
            console.debug(`GetItem response: ${JSON.stringify(response)}`);
            if (!response.Item) {
                // The item was not present in the DB
                return undefined;
            }
            const item = aws_sdk_1.DynamoDB.Converter.unmarshall(response.Item);
            delete item[this.primaryKey];
            delete item[this.sortKey];
            return item;
        }
        catch (e) {
            throw new Error(`GetItem '${props.primaryKeyValue}' '${props.sortKeyValue}:" ` +
                `${e.code} -- ${e.message}`);
        }
    }
    /**
     * Deletes the item from the table that is indexed by the given primary and sort key value
     * @param props
     * @throws Error if the request fails
     * @returns True if the item was deleted; false if there was no matching item to delete.
     */
    async deleteItem(props) {
        if (!this.tableName) {
            throw Error('Attempt to DeleteItem from deleted table');
        }
        // See: https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/DynamoDB.html#deleteItem-property
        const key = {};
        key[this.primaryKey] = props.primaryKeyValue;
        key[this.sortKey] = props.sortKeyValue;
        const request = {
            TableName: this.tableName,
            Key: aws_sdk_1.DynamoDB.Converter.marshall(key),
            ReturnValues: 'ALL_OLD',
            ReturnConsumedCapacity: 'NONE',
            ReturnItemCollectionMetrics: 'NONE',
        };
        try {
            console.debug(`Dynamo.DeleteItem request: ${JSON.stringify(request)}`);
            const response = await this.client.deleteItem(request).promise();
            console.debug(`DeleteItem response: ${JSON.stringify(response)}`);
            if (response.Attributes) {
                // There was a match in the DB, and we deleted it
                return true;
            }
            return false;
        }
        catch (e) {
            throw new Error(`DeleteItem '${props.primaryKeyValue}' '${props.sortKeyValue}:" ` +
                `${e.code} -- ${e.message}`);
        }
    }
    /**
     * Query the table for all items with the given primary key value.
     * @param primaryKeyValue
     * @param pageLimit Maximum number of table items to evaluate per request.
     * @throws Error if the request fails.
     * @returns All of the found items, keyed by their unique sort key values. i.e.
     *         {
     *             'sort key value 1': {
     *                 # attributes other than sort & primary key for this item
     *             },
     *             'sort key value 2': {
     *                 # attributes other than sort & primary key for this item
     *             },
     *             ... etc
     *         }
     */
    async query(primaryKeyValue, pageLimit) {
        if (!this.tableName) {
            throw Error('Attempt to Query a deleted table');
        }
        // See: https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/DynamoDB.html#query-property
        const request = {
            TableName: this.tableName,
            Select: 'ALL_ATTRIBUTES',
            ConsistentRead: true,
            ReturnConsumedCapacity: 'NONE',
            ExpressionAttributeNames: {
                '#PK': this.primaryKey,
            },
            ExpressionAttributeValues: {
                ':PKV': aws_sdk_1.DynamoDB.Converter.input(primaryKeyValue),
            },
            KeyConditionExpression: '#PK = :PKV',
            Limit: pageLimit,
        };
        console.debug(`DynamoDB.Query: ${JSON.stringify(request)}`);
        const items = {};
        try {
            do {
                const response = await this.client.query(request).promise();
                request.ExclusiveStartKey = response.LastEvaluatedKey;
                if (response.Items) {
                    for (const item of response.Items) {
                        const unmarshalled = aws_sdk_1.DynamoDB.Converter.unmarshall(item);
                        const sortValue = unmarshalled[this.sortKey];
                        delete unmarshalled[this.primaryKey];
                        delete unmarshalled[this.sortKey];
                        items[sortValue] = unmarshalled;
                    }
                }
            } while (request.ExclusiveStartKey);
            return items;
        }
        catch (e) {
            throw new Error(`Query '${primaryKeyValue}':" ${e.code} -- ${e.message}`);
        }
    }
}
exports.CompositeStringIndexTable = CompositeStringIndexTable;
CompositeStringIndexTable.API_VERSION = '2012-08-10';
//# sourceMappingURL=data:application/json;base64,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