"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.SimpleCustomResource = void 0;
const hash_1 = require("./hash");
const reply_1 = require("./reply");
/* istanbul ignore file */
class SimpleCustomResource {
    constructor() {
        var _a;
        // Optionally suppress debugging statements.
        this.debugMode = ((_a = process.env.DEBUG) !== null && _a !== void 0 ? _a : 'false') === 'true';
        if (!this.debugMode) {
            console.debug = () => { };
        }
    }
    /**
     * Handler/engine for the CustomResource state machine. Users of this class should
     * instantiate the class, and then immediately call this function.
     * @param event The event passed to the lambda handler.
     * @param context The lambda context passed to the lambda handler.
     */
    async handler(event, context) {
        var _a, _b;
        let status = reply_1.CfnResponseStatus.SUCCESS;
        let failReason;
        let cfnData;
        console.log(`Handling event: ${JSON.stringify(event)}`);
        const requestType = event.RequestType;
        const resourceProperties = (_a = event.ResourceProperties) !== null && _a !== void 0 ? _a : {};
        const physicalId = hash_1.calculateSha256Hash(resourceProperties);
        try {
            if (requestType === 'Create') {
                if (!this.validateInput(resourceProperties)) {
                    throw Error('Input did not pass validation check');
                }
                cfnData = await this.doCreate(physicalId, resourceProperties);
                console.debug(`Create data: ${JSON.stringify(cfnData)}`);
            }
            else if (requestType === 'Update') {
                if (!this.validateInput(resourceProperties)) {
                    throw Error('Input did not pass validation check');
                }
                const oldResourceProperties = (_b = event.OldResourceProperties) !== null && _b !== void 0 ? _b : {};
                const oldPhysicalId = hash_1.calculateSha256Hash(oldResourceProperties);
                if (oldPhysicalId !== physicalId) {
                    console.log('Doing Create -- ResoureceProperties differ.');
                    cfnData = await this.doCreate(physicalId, resourceProperties);
                    console.debug(`Update data: ${JSON.stringify(cfnData)}`);
                }
            }
            else {
                await this.doDelete(physicalId, resourceProperties);
            }
        }
        catch (e) {
            // We want to always catch the exception for a CfnCustomResource CloudFormation
            // must be notified about the success/failure of the lambda at all times;
            // failure to notify results in a stuck stack that takes at least an hour to
            // timeout.
            status = reply_1.CfnResponseStatus.FAILED;
            failReason = `${e.message}\n${e.stack}`;
        }
        finally {
            // Always send a response to CloudFormation, signal success or
            // failure based on whether or not we had an exception.
            await reply_1.sendCfnResponse({
                event,
                context,
                status,
                reason: failReason,
                physicalId,
                data: cfnData,
            });
        }
        const response = `${status}` + (failReason !== null && failReason !== void 0 ? failReason : '');
        console.log(`Result: ${response}`);
        return response;
    }
}
exports.SimpleCustomResource = SimpleCustomResource;
//# sourceMappingURL=data:application/json;base64,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