# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['switchbot_client']

package_data = \
{'': ['*']}

install_requires = \
['requests>=2.26,<3.0']

setup_kwargs = {
    'name': 'switchbot-client',
    'version': '0.1.0',
    'description': 'A Python client library for SwitchBot API.',
    'long_description': '# switchbot-client\n\n[![PyPI - Python Version](https://img.shields.io/pypi/pyversions/switchbot-client.svg)](https://pypi.org/project/switchbot-client/)\n[![PyPI - Downloads](https://img.shields.io/pypi/dm/switchbot-client)](https://pypi.org/project/switchbot-client)\n[![License](https://img.shields.io/badge/license-MIT%2FApache--2.0-informational?style=flat-square)](README.md#License)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n\n\n## Table of Contents\n\n- [Authentication](#authentication)\n- [Usage](#usage)\n- [License](#license)\n\n\n## Authentication\n\nBefore you start using this client, you need to get an open token.\nPlease follow the instructions in the official documentation below.\n\nhttps://github.com/OpenWonderLabs/SwitchBotAPI#authentication\n\nOnce you have the token, use one of the following methods to pass the information to the client.\n\n### Environment variables\n\nIf the environment variable `SWITCHBOT_OPEN_TOKEN` is present, \nthis client will automatically use this value.\n\n```shell\nexport SWITCHBOT_OPEN_TOKEN=yourswitchbotopentoken\npython3 your_script_using_switchbot_client.py\n```\n\n```python\n# your_script_using_switchbot_client.py\nclient = SwitchBotAPIClient()\nresult = client.devices()\n```\n\n### Constructor Arguments\n\nIt is also possible to initialize the client by passing a token directly as an argument.\n\n```python\nfrom switchbot_client import SwitchBotAPIClient\n\nyour_token = "yourswitchbotopentoken"\nclient = SwitchBotAPIClient(token=your_token)\nresult = client.devices()\n```\n\n\n## Usage\n\n### Get Device List\n\n```python\nfrom switchbot_client import SwitchBotAPIClient\n\nclient = SwitchBotAPIClient()\nresult = client.devices()\nprint(result.body)\n```\n\n```\n{\'deviceList\': [{\'deviceId\': \'ABCDEFG\', \'deviceName\': \'Meter 0A\', \'deviceType\': \'Meter\', \'enableCloudService\': True, \'hubDeviceId\': \'ABCDE\'}, {\'deviceId\': \'ABCDE\', \'deviceName\': \'Hub Mini 0\', \'deviceType\': \'Hub Mini\', \'hubDeviceId\': \'ABCDE\'}], \'infraredRemoteList\': [{\'deviceId\': \'12345\', \'deviceName\': \'My Light\', \'remoteType\': \'Light\', \'hubDeviceId\': \'ABCDE\'}, {\'deviceId\': \'12345, \'deviceName\': \'My Air Conditioner\', \'remoteType\': \'Air Conditioner\', \'hubDeviceId\': \'ABCDE\'}]}\n```\n\nIf you run the above code, you will get a list of all the devices associated with your SwitchBot account. \nYou can perform operations on the acquired `deviceId`, such as manipulating it or getting its status.\n\n### Get Device Status\n\n```python\nfrom switchbot_client import SwitchBotAPIClient\n\nclient = SwitchBotAPIClient()\ndevice_id = "YOURDEVICEID"\nprint(client.devices_status(device_id))\n```\n\n```\nSwitchBotAPIResponse(status_code=100, message=\'success\', body={\'deviceId\': \'ABCDE\', \'deviceType\': \'Meter\', \'hubDeviceId\': \'ABCDE\', \'humidity\': 50, \'temperature\': 25.0})\n```\n\nThis function allows you to get the status of a device.\nNote that only physical devices can be targeted, not virtual infrared remote devices.\n\nPlease refer to the following official document to know what kind of information can be obtained from each device.\n\nhttps://github.com/OpenWonderLabs/SwitchBotAPI#get-device-status\n\n### Control Device\n\n```python\nfrom switchbot_client import SwitchBotAPIClient, ControlCommand\n\nclient = SwitchBotAPIClient()\ndevice_id = "12345" # My Light(virtual infrared remote devices)\nprint(client.devices_control(device_id, ControlCommand.VirtualInfrared.TURN_ON))\n```\n\n```\nSwitchBotAPIResponse(status_code=100, message=\'success\', body={})\n```\n\nIt allows you to control the specified device.\nThe `ControlCommand` class and the following documents define the commands that can be executed.\n\nhttps://github.com/OpenWonderLabs/SwitchBotAPI#send-device-control-commands\n\n### Get Scene List\n\n```python\nfrom switchbot_client import SwitchBotAPIClient\n\nclient = SwitchBotAPIClient()\nprint(client.scenes())\n```\n\n```\nSwitchBotAPIResponse(status_code=100, message=\'success\', body=[{\'sceneId\': \'12345\', \'sceneName\': \'My Scene\'}])\n```\n\nYou can get a list of all the scenes associated with your SwitchBot account.\n\n### Execute Scene\n```python\nfrom switchbot_client import SwitchBotAPIClient\n\nclient = SwitchBotAPIClient()\nprint(client.scenes_execute("12345"))\n```\n\n```\nSwitchBotAPIResponse(status_code=100, message=\'success\', body={})\n```\nThe specified scene can be executed immediately.\n\n\n### Examples\n\n```python\nfrom switchbot_client.enums import ControlCommand\nfrom switchbot_client import SwitchBotAPIClient\n\n\ndef control_all_infrared_remotes_by_type(type: str, command: str):\n    client = SwitchBotAPIClient()\n    devices = client.devices()\n    infrared_remotes = devices.body["infraredRemoteList"]\n    devices = filter(lambda d: d["remoteType"] == type, infrared_remotes)\n\n    for d in devices:\n        client.devices_control(d["deviceId"], command)\n\n\ndef call_this_function_when_i_go_out():\n    print("turn off all lights and air conditioners...")\n    control_all_infrared_remotes_by_type(\n        "Light", ControlCommand.VirtualInfrared.TURN_OFF\n    )\n    control_all_infrared_remotes_by_type(\n        "Air Conditioner", ControlCommand.VirtualInfrared.TURN_OFF\n    )\n    print("done")\n```\n\n## License\n\nLicensed under either of\n\n * Apache License, Version 2.0\n   ([LICENSE-APACHE](LICENSE-APACHE) or http://www.apache.org/licenses/LICENSE-2.0)\n * MIT license\n   ([LICENSE-MIT](LICENSE-MIT) or http://opensource.org/licenses/MIT)\n\nat your option.\n\n## Contribution\n\nUnless you explicitly state otherwise, any contribution intentionally submitted\nfor inclusion in the work by you, as defined in the Apache-2.0 license, shall be\ndual licensed as above, without any additional terms or conditions.\n',
    'author': 'Kazuhito Osabe',
    'author_email': 'kzosabe@wip.place',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/kzosabe/switchbot-client',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
