"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MicroAppsCF = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cf = require("aws-cdk-lib/aws-cloudfront");
const cforigins = require("aws-cdk-lib/aws-cloudfront-origins");
const r53 = require("aws-cdk-lib/aws-route53");
const r53targets = require("aws-cdk-lib/aws-route53-targets");
const constructs_1 = require("constructs");
const ReverseDomain_1 = require("./utils/ReverseDomain");
/**
 * Create a new MicroApps CloudFront Distribution.
 */
class MicroAppsCF extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (props === undefined) {
            throw new Error('props must be set');
        }
        if ((props.r53Zone === undefined && props.domainNameEdge !== undefined) ||
            (props.r53Zone !== undefined && props.domainNameEdge === undefined)) {
            throw new Error('If either of r53Zone or domainNameEdge are set then the other must be set');
        }
        const { domainNameEdge, domainNameOrigin, httpApi, removalPolicy, certEdge, assetNameRoot, assetNameSuffix, r53Zone, bucketLogs, bucketAppsOrigin, rootPathPrefix, createAPIPathRoute = true, createNextDataPathRoute = true, edgeToOriginLambdas, } = props;
        const apigwyOriginRequestPolicy = MicroAppsCF.createAPIOriginPolicy(this, {
            assetNameRoot,
            assetNameSuffix,
            domainNameEdge,
        });
        //
        // Determine URL of the origin FQDN
        //
        let httpOriginFQDN = 'invalid.pwrdrvr.com';
        if (domainNameOrigin !== undefined) {
            httpOriginFQDN = domainNameOrigin;
        }
        else {
            httpOriginFQDN = `${httpApi.apiId}.execute-api.${aws_cdk_lib_1.Aws.REGION}.amazonaws.com`;
        }
        //
        // Get the Edge to Origin Lambdas
        //
        //
        // CloudFront Distro
        //
        const apiGwyOrigin = new cforigins.HttpOrigin(httpOriginFQDN, {
            protocolPolicy: cf.OriginProtocolPolicy.HTTPS_ONLY,
            originSslProtocols: [cf.OriginSslPolicy.TLS_V1_2],
        });
        this._cloudFrontDistro = new cf.Distribution(this, 'cft', {
            comment: assetNameRoot ? `${assetNameRoot}${assetNameSuffix}` : domainNameEdge,
            domainNames: domainNameEdge !== undefined ? [domainNameEdge] : undefined,
            certificate: certEdge,
            httpVersion: cf.HttpVersion.HTTP2,
            defaultBehavior: {
                allowedMethods: cf.AllowedMethods.ALLOW_ALL,
                cachePolicy: cf.CachePolicy.CACHING_DISABLED,
                compress: true,
                originRequestPolicy: apigwyOriginRequestPolicy,
                origin: apiGwyOrigin,
                viewerProtocolPolicy: cf.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                edgeLambdas: edgeToOriginLambdas,
            },
            enableIpv6: true,
            priceClass: cf.PriceClass.PRICE_CLASS_100,
            logBucket: bucketLogs,
            logFilePrefix: props.domainNameEdge
                ? `${ReverseDomain_1.reverseDomain(props.domainNameEdge)}/cloudfront-raw/`
                : undefined,
        });
        if (removalPolicy !== undefined) {
            this._cloudFrontDistro.applyRemovalPolicy(removalPolicy);
        }
        // Add routes to the CloudFront Distribution
        MicroAppsCF.addRoutes(scope, {
            apiGwyOrigin,
            bucketAppsOrigin,
            distro: this._cloudFrontDistro,
            apigwyOriginRequestPolicy: apigwyOriginRequestPolicy,
            rootPathPrefix,
            createAPIPathRoute,
            createNextDataPathRoute,
            apigwyEdgeFunctions: edgeToOriginLambdas,
        });
        //
        // Create the edge name for the CloudFront distro
        //
        if (r53Zone !== undefined) {
            const rrAppsEdge = new r53.RecordSet(this, 'edge-arecord', {
                recordName: domainNameEdge,
                recordType: r53.RecordType.A,
                target: r53.RecordTarget.fromAlias(new r53targets.CloudFrontTarget(this._cloudFrontDistro)),
                zone: r53Zone,
            });
            if (removalPolicy !== undefined) {
                rrAppsEdge.applyRemovalPolicy(removalPolicy);
            }
        }
    }
    /**
     * Create or get the origin request policy
     *
     * If a custom domain name is NOT used for the origin then a policy
     * will be created.
     *
     * If a custom domain name IS used for the origin then the ALL_VIEWER
     * policy will be returned.  This policy passes the Host header to the
     * origin, which is fine when using a custom domain name on the origin.
     *
     * @param _scope
     * @param _props
     */
    static createAPIOriginPolicy(_scope, _props) {
        // const { assetNameRoot, assetNameSuffix, domainNameEdge } = props;
        // let apigwyOriginRequestPolicy: cf.IOriginRequestPolicy = cf.OriginRequestPolicy.ALL_VIEWER;
        // if (domainNameEdge === undefined) {
        //   // When not using a custom domain name we must limit down the origin policy to
        //   // prevent it from passing the Host header (distribution_id.cloudfront.net) to
        //   // apigwy which will then reject it with a 403 because it does not match the
        //   // execute-api name that apigwy is expecting.
        //   //
        //   // 2021-12-28 - There is a bug in the name generation that causes the same asset
        //   // in different stacks to have the same generated name.  We have to make the id
        //   // in all cases to ensure the generated name is unique.
        //   apigwyOriginRequestPolicy = new cf.OriginRequestPolicy(
        //     scope,
        //     `apigwy-origin-policy-${Stack.of(scope).stackName}`,
        //     {
        //       comment: assetNameRoot ? `${assetNameRoot}-apigwy${assetNameSuffix}` : undefined,
        //       originRequestPolicyName: assetNameRoot
        //         ? `${assetNameRoot}-apigwy${assetNameSuffix}`
        //         : undefined,
        //       cookieBehavior: cf.OriginRequestCookieBehavior.all(),
        //       queryStringBehavior: cf.OriginRequestQueryStringBehavior.all(),
        //       // TODO: If signing is enabled this should forward all signature headers
        //       // TODO: If set to "cfront.OriginRequestHeaderBehavior.all()" then
        //       // `replaceHostHeader` must be set to true to prevent API Gateway from rejecting
        //       // the request
        //       // headerBehavior: cf.OriginRequestHeaderBehavior.allowList('user-agent', 'referer'),
        //       headerBehavior: cf.OriginRequestHeaderBehavior.all(),
        //     },
        //   );
        // }
        return cf.OriginRequestPolicy.ALL_VIEWER;
    }
    /**
     * Add API Gateway and S3 routes to an existing CloudFront Distribution
     * @param _scope
     * @param props
     */
    static addRoutes(_scope, props) {
        const { apiGwyOrigin, bucketAppsOrigin, distro, apigwyOriginRequestPolicy, rootPathPrefix = '', createAPIPathRoute = true, createNextDataPathRoute = true, } = props;
        //
        // Add Behaviors
        //
        const s3BehaviorOptions = {
            allowedMethods: cf.AllowedMethods.ALLOW_GET_HEAD_OPTIONS,
            cachePolicy: cf.CachePolicy.CACHING_OPTIMIZED,
            compress: true,
            originRequestPolicy: cf.OriginRequestPolicy.CORS_S3_ORIGIN,
            viewerProtocolPolicy: cf.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        };
        const apiGwyBehaviorOptions = {
            allowedMethods: cf.AllowedMethods.ALLOW_ALL,
            // TODO: Caching needs to be set by the app response
            cachePolicy: cf.CachePolicy.CACHING_DISABLED,
            compress: true,
            originRequestPolicy: apigwyOriginRequestPolicy,
            viewerProtocolPolicy: cf.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            edgeLambdas: props.apigwyEdgeFunctions,
        };
        //
        // If a route specifically has `/api/` in it, send it to API Gateway
        // This is needed to catch routes that have periods in the API path data,
        // such as: /release/0.0.0/api/update/default/release/0.0.0
        //
        if (createAPIPathRoute) {
            distro.addBehavior(path_1.posix.join(rootPathPrefix, '/*/*/api/*'), apiGwyOrigin, apiGwyBehaviorOptions);
        }
        //
        // If a route specifically has `/_next/data/` in it, send it to API Gateway
        // This is needed to catch routes that have periods in the API path data,
        // such as: /release/0.0.0/_next/data/app.json
        //
        if (createNextDataPathRoute) {
            distro.addBehavior(path_1.posix.join(rootPathPrefix, '/*/*/_next/data/*'), apiGwyOrigin, apiGwyBehaviorOptions);
        }
        //
        // All static assets are assumed to have a dot in them
        //
        distro.addBehavior(path_1.posix.join(rootPathPrefix, '/*/*/*.*'), bucketAppsOrigin, s3BehaviorOptions);
        //
        // Everything that isn't a static asset is going to API Gateway
        // There is no trailing slash because Serverless Next.js wants
        // go load pages at /release/0.0.3 (with no trailing slash).
        //
        distro.addBehavior(path_1.posix.join(rootPathPrefix, '/*'), apiGwyOrigin, apiGwyBehaviorOptions);
    }
    get cloudFrontDistro() {
        return this._cloudFrontDistro;
    }
}
exports.MicroAppsCF = MicroAppsCF;
_a = JSII_RTTI_SYMBOL_1;
MicroAppsCF[_a] = { fqn: "@pwrdrvr/microapps-cdk.MicroAppsCF", version: "0.3.0-alpha.1" };
//# sourceMappingURL=data:application/json;base64,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