"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFormationStackNotificationCheck = exports.CloudFormationStackDriftDetectionCheck = exports.AccessKeysRotated = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const rule_1 = require("./rule");
/**
 * Checks whether the active access keys are rotated within the number of days
 * specified in `maxAge`.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/access-keys-rotated.html
 *
 * @resource AWS::Config::ConfigRule
 */
class AccessKeysRotated extends rule_1.ManagedRule {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            ...props,
            identifier: 'ACCESS_KEYS_ROTATED',
            inputParameters: {
                ...props.maxAge
                    ? {
                        maxAccessKeyAge: props.maxAge.toDays(),
                    }
                    : {},
            },
        });
    }
}
exports.AccessKeysRotated = AccessKeysRotated;
/**
 * Checks whether your CloudFormation stacks' actual configuration differs, or
 * has drifted, from its expected configuration.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudformation-stack-drift-detection-check.html
 *
 * @resource AWS::Config::ConfigRule
 */
class CloudFormationStackDriftDetectionCheck extends rule_1.ManagedRule {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            ...props,
            identifier: 'CLOUDFORMATION_STACK_DRIFT_DETECTION_CHECK',
            inputParameters: {
                cloudformationRoleArn: core_1.Lazy.stringValue({ produce: () => this.role.roleArn }),
            },
        });
        this.scopeToResource('AWS::CloudFormation::Stack', props.ownStackOnly ? core_1.Stack.of(this).stackId : undefined);
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('config.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('ReadOnlyAccess'),
            ],
        });
    }
}
exports.CloudFormationStackDriftDetectionCheck = CloudFormationStackDriftDetectionCheck;
/**
 * Checks whether your CloudFormation stacks are sending event notifications to
 * a SNS topic. Optionally checks whether specified SNS topics are used.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudformation-stack-notification-check.html
 *
 * @resource AWS::Config::ConfigRule
 */
class CloudFormationStackNotificationCheck extends rule_1.ManagedRule {
    constructor(scope, id, props = {}) {
        if (props.topics && props.topics.length > 5) {
            throw new Error('At most 5 topics can be specified.');
        }
        super(scope, id, {
            ...props,
            identifier: 'CLOUDFORMATION_STACK_NOTIFICATION_CHECK',
            inputParameters: props.topics && props.topics.reduce((params, topic, idx) => ({ ...params, [`snsTopic${idx + 1}`]: topic.topicArn }), {}),
        });
        this.scopeToResource('AWS::CloudFormation::Stack');
    }
}
exports.CloudFormationStackNotificationCheck = CloudFormationStackNotificationCheck;
//# sourceMappingURL=data:application/json;base64,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