import { IConstruct, IDependable, IResource, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { ClientVpnEndpoint, ClientVpnEndpointOptions } from './client-vpn-endpoint';
import { CfnNatGateway } from './ec2.generated';
import { NatProvider } from './nat';
import { INetworkAcl } from './network-acl';
import { SubnetFilter } from './subnet';
import { GatewayVpcEndpoint, GatewayVpcEndpointOptions, InterfaceVpcEndpoint, InterfaceVpcEndpointOptions } from './vpc-endpoint';
import { FlowLog, FlowLogOptions } from './vpc-flow-logs';
import { VpcLookupOptions } from './vpc-lookup';
import { EnableVpnGatewayOptions, VpnConnection, VpnConnectionOptions } from './vpn';
/**
 * @stability stable
 */
export interface ISubnet extends IResource {
    /**
     * The Availability Zone the subnet is located in.
     *
     * @stability stable
     */
    readonly availabilityZone: string;
    /**
     * The subnetId for this particular subnet.
     *
     * @stability stable
     * @attribute true
     */
    readonly subnetId: string;
    /**
     * Dependable that can be depended upon to force internet connectivity established on the VPC.
     *
     * @stability stable
     */
    readonly internetConnectivityEstablished: IDependable;
    /**
     * The IPv4 CIDR block for this subnet.
     *
     * @stability stable
     */
    readonly ipv4CidrBlock: string;
    /**
     * The route table for this subnet.
     *
     * @stability stable
     */
    readonly routeTable: IRouteTable;
    /**
     * Associate a Network ACL with this subnet.
     *
     * @param acl The Network ACL to associate.
     * @stability stable
     */
    associateNetworkAcl(id: string, acl: INetworkAcl): void;
}
/**
 * An abstract route table.
 *
 * @stability stable
 */
export interface IRouteTable {
    /**
     * Route table ID.
     *
     * @stability stable
     */
    readonly routeTableId: string;
}
/**
 * @stability stable
 */
export interface IVpc extends IResource {
    /**
     * Identifier for this VPC.
     *
     * @stability stable
     * @attribute true
     */
    readonly vpcId: string;
    /**
     * CIDR range for this VPC.
     *
     * @stability stable
     * @attribute true
     */
    readonly vpcCidrBlock: string;
    /**
     * List of public subnets in this VPC.
     *
     * @stability stable
     */
    readonly publicSubnets: ISubnet[];
    /**
     * List of private subnets in this VPC.
     *
     * @stability stable
     */
    readonly privateSubnets: ISubnet[];
    /**
     * List of isolated subnets in this VPC.
     *
     * @stability stable
     */
    readonly isolatedSubnets: ISubnet[];
    /**
     * AZs for this VPC.
     *
     * @stability stable
     */
    readonly availabilityZones: string[];
    /**
     * Identifier for the VPN gateway.
     *
     * @stability stable
     */
    readonly vpnGatewayId?: string;
    /**
     * Dependable that can be depended upon to force internet connectivity established on the VPC.
     *
     * @stability stable
     */
    readonly internetConnectivityEstablished: IDependable;
    /**
     * Return information on the subnets appropriate for the given selection strategy.
     *
     * Requires that at least one subnet is matched, throws a descriptive
     * error message otherwise.
     *
     * @stability stable
     */
    selectSubnets(selection?: SubnetSelection): SelectedSubnets;
    /**
     * Adds a VPN Gateway to this VPC.
     *
     * @stability stable
     */
    enableVpnGateway(options: EnableVpnGatewayOptions): void;
    /**
     * Adds a new VPN connection to this VPC.
     *
     * @stability stable
     */
    addVpnConnection(id: string, options: VpnConnectionOptions): VpnConnection;
    /**
     * Adds a new client VPN endpoint to this VPC.
     *
     * @stability stable
     */
    addClientVpnEndpoint(id: string, options: ClientVpnEndpointOptions): ClientVpnEndpoint;
    /**
     * Adds a new gateway endpoint to this VPC.
     *
     * @stability stable
     */
    addGatewayEndpoint(id: string, options: GatewayVpcEndpointOptions): GatewayVpcEndpoint;
    /**
     * Adds a new interface endpoint to this VPC.
     *
     * @stability stable
     */
    addInterfaceEndpoint(id: string, options: InterfaceVpcEndpointOptions): InterfaceVpcEndpoint;
    /**
     * Adds a new Flow Log to this VPC.
     *
     * @stability stable
     */
    addFlowLog(id: string, options?: FlowLogOptions): FlowLog;
}
/**
 * The type of Subnet.
 *
 * @stability stable
 */
export declare enum SubnetType {
    /**
     * (deprecated) Isolated Subnets do not route traffic to the Internet (in this VPC), and as such, do not require NAT gateways.
     *
     * Isolated subnets can only connect to or be connected to from other
     * instances in the same VPC. A default VPC configuration will not include
     * isolated subnets.
     *
     * This can be good for subnets with RDS or Elasticache instances,
     * or which route Internet traffic through a peer VPC.
     *
     * @deprecated use `SubnetType.PRIVATE_ISOLATED`
     */
    ISOLATED = "Isolated",
    /**
     * Isolated Subnets do not route traffic to the Internet (in this VPC),
     * and as such, do not require NAT gateways.
     *
     * Isolated subnets can only connect to or be connected to from other
     * instances in the same VPC. A default VPC configuration will not include
     * isolated subnets.
     *
     * This can be good for subnets with RDS or Elasticache instances,
     * or which route Internet traffic through a peer VPC.
     */
    PRIVATE_ISOLATED = "Isolated",
    /**
     * (deprecated) Subnet that routes to the internet, but not vice versa.
     *
     * Instances in a private subnet can connect to the Internet, but will not
     * allow connections to be initiated from the Internet. NAT Gateway(s) are
     * required with this subnet type to route the Internet traffic through.
     * If a NAT Gateway is not required or desired, use `SubnetType.PRIVATE_ISOLATED` instead.
     *
     * By default, a NAT gateway is created in every public subnet for maximum availability.
     * Be aware that you will be charged for NAT gateways.
     *
     * Normally a Private subnet will use a NAT gateway in the same AZ, but
     * if `natGateways` is used to reduce the number of NAT gateways, a NAT
     * gateway from another AZ will be used instead.
     *
     * @deprecated use `PRIVATE_WITH_NAT`
     */
    PRIVATE = "Private",
    /**
     * Subnet that routes to the internet (via a NAT gateway), but not vice versa.
     *
     * Instances in a private subnet can connect to the Internet, but will not
     * allow connections to be initiated from the Internet. NAT Gateway(s) are
     * required with this subnet type to route the Internet traffic through.
     * If a NAT Gateway is not required or desired, use `SubnetType.PRIVATE_ISOLATED` instead.
     *
     * By default, a NAT gateway is created in every public subnet for maximum availability.
     * Be aware that you will be charged for NAT gateways.
     *
     * Normally a Private subnet will use a NAT gateway in the same AZ, but
     * if `natGateways` is used to reduce the number of NAT gateways, a NAT
     * gateway from another AZ will be used instead.
     */
    PRIVATE_WITH_NAT = "Private",
    /**
     * Subnet connected to the Internet.
     *
     * Instances in a Public subnet can connect to the Internet and can be
     * connected to from the Internet as long as they are launched with public
     * IPs (controlled on the AutoScalingGroup or other constructs that launch
     * instances).
     *
     * Public subnets route outbound traffic via an Internet Gateway.
     *
     * @stability stable
     */
    PUBLIC = "Public"
}
/**
 * Customize subnets that are selected for placement of ENIs.
 *
 * Constructs that allow customization of VPC placement use parameters of this
 * type to provide placement settings.
 *
 * By default, the instances are placed in the private subnets.
 *
 * @stability stable
 */
export interface SubnetSelection {
    /**
     * Select all subnets of the given type.
     *
     * At most one of `subnetType` and `subnetGroupName` can be supplied.
     *
     * @default SubnetType.PRIVATE_WITH_NAT (or ISOLATED or PUBLIC if there are no PRIVATE_WITH_NAT subnets)
     * @stability stable
     */
    readonly subnetType?: SubnetType;
    /**
     * Select subnets only in the given AZs.
     *
     * @default no filtering on AZs is done
     * @stability stable
     */
    readonly availabilityZones?: string[];
    /**
     * Select the subnet group with the given name.
     *
     * Select the subnet group with the given name. This only needs
     * to be used if you have multiple subnet groups of the same type
     * and you need to distinguish between them. Otherwise, prefer
     * `subnetType`.
     *
     * This field does not select individual subnets, it selects all subnets that
     * share the given subnet group name. This is the name supplied in
     * `subnetConfiguration`.
     *
     * At most one of `subnetType` and `subnetGroupName` can be supplied.
     *
     * @default - Selection by type instead of by name
     * @stability stable
     */
    readonly subnetGroupName?: string;
    /**
     * (deprecated) Alias for `subnetGroupName`.
     *
     * Select the subnet group with the given name. This only needs
     * to be used if you have multiple subnet groups of the same type
     * and you need to distinguish between them.
     *
     * @deprecated Use `subnetGroupName` instead
     */
    readonly subnetName?: string;
    /**
     * If true, return at most one subnet per AZ.
     *
     * @default false
     * @stability stable
     */
    readonly onePerAz?: boolean;
    /**
     * List of provided subnet filters.
     *
     * @default - none
     * @stability stable
     */
    readonly subnetFilters?: SubnetFilter[];
    /**
     * Explicitly select individual subnets.
     *
     * Use this if you don't want to automatically use all subnets in
     * a group, but have a need to control selection down to
     * individual subnets.
     *
     * Cannot be specified together with `subnetType` or `subnetGroupName`.
     *
     * @default - Use all subnets in a selected group (all private subnets by default)
     * @stability stable
     */
    readonly subnets?: ISubnet[];
}
/**
 * Result of selecting a subset of subnets from a VPC.
 *
 * @stability stable
 */
export interface SelectedSubnets {
    /**
     * The subnet IDs.
     *
     * @stability stable
     */
    readonly subnetIds: string[];
    /**
     * The respective AZs of each subnet.
     *
     * @stability stable
     */
    readonly availabilityZones: string[];
    /**
     * Dependency representing internet connectivity for these subnets.
     *
     * @stability stable
     */
    readonly internetConnectivityEstablished: IDependable;
    /**
     * Selected subnet objects.
     *
     * @stability stable
     */
    readonly subnets: ISubnet[];
    /**
     * Whether any of the given subnets are from the VPC's public subnets.
     *
     * @stability stable
     */
    readonly hasPublic: boolean;
}
/**
 * A new or imported VPC
 */
declare abstract class VpcBase extends Resource implements IVpc {
    /**
     * Identifier for this VPC
     */
    abstract readonly vpcId: string;
    /**
     * CIDR range for this VPC
     */
    abstract readonly vpcCidrBlock: string;
    /**
     * List of public subnets in this VPC
     */
    abstract readonly publicSubnets: ISubnet[];
    /**
     * List of private subnets in this VPC
     */
    abstract readonly privateSubnets: ISubnet[];
    /**
     * List of isolated subnets in this VPC
     */
    abstract readonly isolatedSubnets: ISubnet[];
    /**
     * AZs for this VPC
     */
    abstract readonly availabilityZones: string[];
    /**
     * Dependencies for internet connectivity
     */
    abstract readonly internetConnectivityEstablished: IDependable;
    /**
     * (deprecated) Dependencies for NAT connectivity.
     *
     * @deprecated - This value is no longer used.
     */
    protected readonly natDependencies: IConstruct[];
    /**
     * If this is set to true, don't error out on trying to select subnets.
     *
     * @stability stable
     */
    protected incompleteSubnetDefinition: boolean;
    /**
     * Mutable private field for the vpnGatewayId
     *
     * @internal
     */
    protected _vpnGatewayId?: string;
    /**
     * Returns IDs of selected subnets.
     *
     * @stability stable
     */
    selectSubnets(selection?: SubnetSelection): SelectedSubnets;
    /**
     * Adds a VPN Gateway to this VPC.
     *
     * @stability stable
     */
    enableVpnGateway(options: EnableVpnGatewayOptions): void;
    /**
     * Adds a new VPN connection to this VPC.
     *
     * @stability stable
     */
    addVpnConnection(id: string, options: VpnConnectionOptions): VpnConnection;
    /**
     * Adds a new client VPN endpoint to this VPC.
     *
     * @stability stable
     */
    addClientVpnEndpoint(id: string, options: ClientVpnEndpointOptions): ClientVpnEndpoint;
    /**
     * Adds a new interface endpoint to this VPC.
     *
     * @stability stable
     */
    addInterfaceEndpoint(id: string, options: InterfaceVpcEndpointOptions): InterfaceVpcEndpoint;
    /**
     * Adds a new gateway endpoint to this VPC.
     *
     * @stability stable
     */
    addGatewayEndpoint(id: string, options: GatewayVpcEndpointOptions): GatewayVpcEndpoint;
    /**
     * Adds a new flow log to this VPC.
     *
     * @stability stable
     */
    addFlowLog(id: string, options?: FlowLogOptions): FlowLog;
    /**
     * Returns the id of the VPN Gateway (if enabled).
     *
     * @stability stable
     */
    get vpnGatewayId(): string | undefined;
    /**
     * Return the subnets appropriate for the placement strategy.
     *
     * @stability stable
     */
    protected selectSubnetObjects(selection?: SubnetSelection): ISubnet[];
    private applySubnetFilters;
    private selectSubnetObjectsByName;
    private selectSubnetObjectsByType;
    /**
     * Validate the fields in a SubnetSelection object, and reify defaults if necessary
     *
     * In case of default selection, select the first type of PRIVATE, ISOLATED,
     * PUBLIC (in that order) that has any subnets.
     */
    private reifySelectionDefaults;
}
/**
 * Properties that reference an external Vpc.
 *
 * @stability stable
 */
export interface VpcAttributes {
    /**
     * VPC's identifier.
     *
     * @stability stable
     */
    readonly vpcId: string;
    /**
     * VPC's CIDR range.
     *
     * @default - Retrieving the CIDR from the VPC will fail
     * @stability stable
     */
    readonly vpcCidrBlock?: string;
    /**
     * List of availability zones for the subnets in this VPC.
     *
     * @stability stable
     */
    readonly availabilityZones: string[];
    /**
     * List of public subnet IDs.
     *
     * Must be undefined or match the availability zones in length and order.
     *
     * @stability stable
     */
    readonly publicSubnetIds?: string[];
    /**
     * List of names for the public subnets.
     *
     * Must be undefined or have a name for every public subnet group.
     *
     * @stability stable
     */
    readonly publicSubnetNames?: string[];
    /**
     * List of IDs of routing tables for the public subnets.
     *
     * Must be undefined or have a name for every public subnet group.
     *
     * @stability stable
     */
    readonly publicSubnetRouteTableIds?: string[];
    /**
     * List of private subnet IDs.
     *
     * Must be undefined or match the availability zones in length and order.
     *
     * @stability stable
     */
    readonly privateSubnetIds?: string[];
    /**
     * List of names for the private subnets.
     *
     * Must be undefined or have a name for every private subnet group.
     *
     * @stability stable
     */
    readonly privateSubnetNames?: string[];
    /**
     * List of IDs of routing tables for the private subnets.
     *
     * Must be undefined or have a name for every private subnet group.
     *
     * @stability stable
     */
    readonly privateSubnetRouteTableIds?: string[];
    /**
     * List of isolated subnet IDs.
     *
     * Must be undefined or match the availability zones in length and order.
     *
     * @stability stable
     */
    readonly isolatedSubnetIds?: string[];
    /**
     * List of names for the isolated subnets.
     *
     * Must be undefined or have a name for every isolated subnet group.
     *
     * @stability stable
     */
    readonly isolatedSubnetNames?: string[];
    /**
     * List of IDs of routing tables for the isolated subnets.
     *
     * Must be undefined or have a name for every isolated subnet group.
     *
     * @stability stable
     */
    readonly isolatedSubnetRouteTableIds?: string[];
    /**
     * VPN gateway's identifier.
     *
     * @stability stable
     */
    readonly vpnGatewayId?: string;
}
/**
 * @stability stable
 */
export interface SubnetAttributes {
    /**
     * The Availability Zone the subnet is located in.
     *
     * @default - No AZ information, cannot use AZ selection features
     * @stability stable
     */
    readonly availabilityZone?: string;
    /**
     * The IPv4 CIDR block associated with the subnet.
     *
     * @default - No CIDR information, cannot use CIDR filter features
     * @stability stable
     */
    readonly ipv4CidrBlock?: string;
    /**
     * The ID of the route table for this particular subnet.
     *
     * @default - No route table information, cannot create VPC endpoints
     * @stability stable
     */
    readonly routeTableId?: string;
    /**
     * The subnetId for this particular subnet.
     *
     * @stability stable
     */
    readonly subnetId: string;
}
/**
 * Configuration for Vpc.
 *
 * @stability stable
 */
export interface VpcProps {
    /**
     * The CIDR range to use for the VPC, e.g. '10.0.0.0/16'.
     *
     * Should be a minimum of /28 and maximum size of /16. The range will be
     * split across all subnets per Availability Zone.
     *
     * @default Vpc.DEFAULT_CIDR_RANGE
     * @stability stable
     */
    readonly cidr?: string;
    /**
     * Indicates whether the instances launched in the VPC get public DNS hostnames.
     *
     * If this attribute is true, instances in the VPC get public DNS hostnames,
     * but only if the enableDnsSupport attribute is also set to true.
     *
     * @default true
     * @stability stable
     */
    readonly enableDnsHostnames?: boolean;
    /**
     * Indicates whether the DNS resolution is supported for the VPC.
     *
     * If this attribute is false, the Amazon-provided DNS server in the VPC that
     * resolves public DNS hostnames to IP addresses is not enabled. If this
     * attribute is true, queries to the Amazon provided DNS server at the
     * 169.254.169.253 IP address, or the reserved IP address at the base of the
     * VPC IPv4 network range plus two will succeed.
     *
     * @default true
     * @stability stable
     */
    readonly enableDnsSupport?: boolean;
    /**
     * The default tenancy of instances launched into the VPC.
     *
     * By setting this to dedicated tenancy, instances will be launched on
     * hardware dedicated to a single AWS customer, unless specifically specified
     * at instance launch time. Please note, not all instance types are usable
     * with Dedicated tenancy.
     *
     * @default DefaultInstanceTenancy.Default (shared) tenancy
     * @stability stable
     */
    readonly defaultInstanceTenancy?: DefaultInstanceTenancy;
    /**
     * Define the maximum number of AZs to use in this region.
     *
     * If the region has more AZs than you want to use (for example, because of
     * EIP limits), pick a lower number here. The AZs will be sorted and picked
     * from the start of the list.
     *
     * If you pick a higher number than the number of AZs in the region, all AZs
     * in the region will be selected. To use "all AZs" available to your
     * account, use a high number (such as 99).
     *
     * Be aware that environment-agnostic stacks will be created with access to
     * only 2 AZs, so to use more than 2 AZs, be sure to specify the account and
     * region on your stack.
     *
     * @default 3
     * @stability stable
     */
    readonly maxAzs?: number;
    /**
     * The number of NAT Gateways/Instances to create.
     *
     * The type of NAT gateway or instance will be determined by the
     * `natGatewayProvider` parameter.
     *
     * You can set this number lower than the number of Availability Zones in your
     * VPC in order to save on NAT cost. Be aware you may be charged for
     * cross-AZ data traffic instead.
     *
     * @default - One NAT gateway/instance per Availability Zone
     * @stability stable
     */
    readonly natGateways?: number;
    /**
     * Configures the subnets which will have NAT Gateways/Instances.
     *
     * You can pick a specific group of subnets by specifying the group name;
     * the picked subnets must be public subnets.
     *
     * Only necessary if you have more than one public subnet group.
     *
     * @default - All public subnets.
     * @stability stable
     */
    readonly natGatewaySubnets?: SubnetSelection;
    /**
     * What type of NAT provider to use.
     *
     * Select between NAT gateways or NAT instances. NAT gateways
     * may not be available in all AWS regions.
     *
     * @default NatProvider.gateway()
     * @stability stable
     */
    readonly natGatewayProvider?: NatProvider;
    /**
     * Configure the subnets to build for each AZ.
     *
     * Each entry in this list configures a Subnet Group; each group will contain a
     * subnet for each Availability Zone.
     *
     * For example, if you want 1 public subnet, 1 private subnet, and 1 isolated
     * subnet in each AZ provide the following:
     *
     * ```ts
     * new ec2.Vpc(this, 'VPC', {
     *    subnetConfiguration: [
     *       {
     *         cidrMask: 24,
     *         name: 'ingress',
     *         subnetType: ec2.SubnetType.PUBLIC,
     *       },
     *       {
     *         cidrMask: 24,
     *         name: 'application',
     *         subnetType: ec2.SubnetType.PRIVATE_WITH_NAT,
     *       },
     *       {
     *         cidrMask: 28,
     *         name: 'rds',
     *         subnetType: ec2.SubnetType.PRIVATE_ISOLATED,
     *       }
     *    ]
     * });
     * ```
     *
     * @default - The VPC CIDR will be evenly divided between 1 public and 1
     * private subnet per AZ.
     * @stability stable
     */
    readonly subnetConfiguration?: SubnetConfiguration[];
    /**
     * Indicates whether a VPN gateway should be created and attached to this VPC.
     *
     * @default - true when vpnGatewayAsn or vpnConnections is specified
     * @stability stable
     */
    readonly vpnGateway?: boolean;
    /**
     * The private Autonomous System Number (ASN) for the VPN gateway.
     *
     * @default - Amazon default ASN.
     * @stability stable
     */
    readonly vpnGatewayAsn?: number;
    /**
     * VPN connections to this VPC.
     *
     * @default - No connections.
     * @stability stable
     */
    readonly vpnConnections?: {
        [id: string]: VpnConnectionOptions;
    };
    /**
     * Where to propagate VPN routes.
     *
     * @default - On the route tables associated with private subnets. If no
     * private subnets exists, isolated subnets are used. If no isolated subnets
     * exists, public subnets are used.
     * @stability stable
     */
    readonly vpnRoutePropagation?: SubnetSelection[];
    /**
     * Gateway endpoints to add to this VPC.
     *
     * @default - None.
     * @stability stable
     */
    readonly gatewayEndpoints?: {
        [id: string]: GatewayVpcEndpointOptions;
    };
    /**
     * Flow logs to add to this VPC.
     *
     * @default - No flow logs.
     * @stability stable
     */
    readonly flowLogs?: {
        [id: string]: FlowLogOptions;
    };
}
/**
 * The default tenancy of instances launched into the VPC.
 *
 * @stability stable
 */
export declare enum DefaultInstanceTenancy {
    /**
     * Instances can be launched with any tenancy.
     *
     * @stability stable
     */
    DEFAULT = "default",
    /**
     * Any instance launched into the VPC automatically has dedicated tenancy, unless you launch it with the default tenancy.
     *
     * @stability stable
     */
    DEDICATED = "dedicated"
}
/**
 * Specify configuration parameters for a single subnet group in a VPC.
 *
 * @stability stable
 */
export interface SubnetConfiguration {
    /**
     * The number of leading 1 bits in the routing mask.
     *
     * The number of available IP addresses in each subnet of this group
     * will be equal to `2^(32 - cidrMask) - 2`.
     *
     * Valid values are `16--28`.
     *
     * @default - Available IP space is evenly divided across subnets.
     * @stability stable
     */
    readonly cidrMask?: number;
    /**
     * The type of Subnet to configure.
     *
     * The Subnet type will control the ability to route and connect to the
     * Internet.
     *
     * @stability stable
     */
    readonly subnetType: SubnetType;
    /**
     * Logical name for the subnet group.
     *
     * This name can be used when selecting VPC subnets to distinguish
     * between different subnet groups of the same type.
     *
     * @stability stable
     */
    readonly name: string;
    /**
     * Controls if subnet IP space needs to be reserved.
     *
     * When true, the IP space for the subnet is reserved but no actual
     * resources are provisioned. This space is only dependent on the
     * number of availability zones and on `cidrMask` - all other subnet
     * properties are ignored.
     *
     * @default false
     * @stability stable
     */
    readonly reserved?: boolean;
}
/**
 * Define an AWS Virtual Private Cloud.
 *
 * See the package-level documentation of this package for an overview
 * of the various dimensions in which you can configure your VPC.
 *
 * For example:
 *
 * ```ts
 * const vpc = new ec2.Vpc(this, 'TheVPC', {
 *    cidr: "10.0.0.0/16"
 * })
 *
 * // Iterate the private subnets
 * const selection = vpc.selectSubnets({
 *    subnetType: ec2.SubnetType.PRIVATE_WITH_NAT
 * });
 *
 * for (const subnet of selection.subnets) {
 *    // ...
 * }
 * ```
 *
 * @stability stable
 * @resource AWS::EC2::VPC
 */
export declare class Vpc extends VpcBase {
    /**
     * The default CIDR range used when creating VPCs.
     *
     * This can be overridden using VpcProps when creating a VPCNetwork resource.
     * e.g. new VpcResource(this, { cidr: '192.168.0.0./16' })
     *
     * @stability stable
     */
    static readonly DEFAULT_CIDR_RANGE: string;
    /**
     * The default subnet configuration.
     *
     * 1 Public and 1 Private subnet per AZ evenly split
     *
     * @stability stable
     */
    static readonly DEFAULT_SUBNETS: SubnetConfiguration[];
    /**
     * The default subnet configuration if natGateways specified to be 0.
     *
     * 1 Public and 1 Isolated Subnet per AZ evenly split
     *
     * @stability stable
     */
    static readonly DEFAULT_SUBNETS_NO_NAT: SubnetConfiguration[];
    /**
     * Import a VPC by supplying all attributes directly.
     *
     * NOTE: using `fromVpcAttributes()` with deploy-time parameters (like a `Fn.importValue()` or
     * `CfnParameter` to represent a list of subnet IDs) sometimes accidentally works. It happens
     * to work for constructs that need a list of subnets (like `AutoScalingGroup` and `eks.Cluster`)
     * but it does not work for constructs that need individual subnets (like
     * `Instance`). See https://github.com/aws/aws-cdk/issues/4118 for more
     * information.
     *
     * Prefer to use `Vpc.fromLookup()` instead.
     *
     * @stability stable
     */
    static fromVpcAttributes(scope: Construct, id: string, attrs: VpcAttributes): IVpc;
    /**
     * Import an existing VPC from by querying the AWS environment this stack is deployed to.
     *
     * This function only needs to be used to use VPCs not defined in your CDK
     * application. If you are looking to share a VPC between stacks, you can
     * pass the `Vpc` object between stacks and use it as normal.
     *
     * Calling this method will lead to a lookup when the CDK CLI is executed.
     * You can therefore not use any values that will only be available at
     * CloudFormation execution time (i.e., Tokens).
     *
     * The VPC information will be cached in `cdk.context.json` and the same VPC
     * will be used on future runs. To refresh the lookup, you will have to
     * evict the value from the cache using the `cdk context` command. See
     * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
     *
     * @stability stable
     */
    static fromLookup(scope: Construct, id: string, options: VpcLookupOptions): IVpc;
    /**
     * Identifier for this VPC.
     *
     * @stability stable
     */
    readonly vpcId: string;
    /**
     * CIDR range for this VPC.
     *
     * @stability stable
     * @attribute true
     */
    readonly vpcCidrBlock: string;
    /**
     * @stability stable
     * @attribute true
     */
    readonly vpcDefaultNetworkAcl: string;
    /**
     * @stability stable
     * @attribute true
     */
    readonly vpcCidrBlockAssociations: string[];
    /**
     * @stability stable
     * @attribute true
     */
    readonly vpcDefaultSecurityGroup: string;
    /**
     * @stability stable
     * @attribute true
     */
    readonly vpcIpv6CidrBlocks: string[];
    /**
     * List of public subnets in this VPC.
     *
     * @stability stable
     */
    readonly publicSubnets: ISubnet[];
    /**
     * List of private subnets in this VPC.
     *
     * @stability stable
     */
    readonly privateSubnets: ISubnet[];
    /**
     * List of isolated subnets in this VPC.
     *
     * @stability stable
     */
    readonly isolatedSubnets: ISubnet[];
    /**
     * AZs for this VPC.
     *
     * @stability stable
     */
    readonly availabilityZones: string[];
    /**
     * Internet Gateway for the VPC.
     *
     * Note that in case the VPC is configured only
     * with ISOLATED subnets, this attribute will be `undefined`.
     *
     * @stability stable
     */
    readonly internetGatewayId?: string;
    /**
     * Dependencies for internet connectivity.
     *
     * @stability stable
     */
    readonly internetConnectivityEstablished: IDependable;
    /**
     * Indicates if instances launched in this VPC will have public DNS hostnames.
     *
     * @stability stable
     */
    readonly dnsHostnamesEnabled: boolean;
    /**
     * Indicates if DNS support is enabled for this VPC.
     *
     * @stability stable
     */
    readonly dnsSupportEnabled: boolean;
    /**
     * The VPC resource
     */
    private readonly resource;
    /**
     * The NetworkBuilder
     */
    private networkBuilder;
    /**
     * Subnet configurations for this VPC
     */
    private subnetConfiguration;
    private readonly _internetConnectivityEstablished;
    /**
     * Vpc creates a VPC that spans a whole region.
     *
     * It will automatically divide the provided VPC CIDR range, and create public and private subnets per Availability Zone.
     * Network routing for the public subnets will be configured to allow outbound access directly via an Internet Gateway.
     * Network routing for the private subnets will be configured to allow outbound access via a set of resilient NAT Gateways (one per AZ).
     *
     * @stability stable
     */
    constructor(scope: Construct, id: string, props?: VpcProps);
    /**
     * (deprecated) Adds a new S3 gateway endpoint to this VPC.
     *
     * @deprecated use `addGatewayEndpoint()` instead
     */
    addS3Endpoint(id: string, subnets?: SubnetSelection[]): GatewayVpcEndpoint;
    /**
     * (deprecated) Adds a new DynamoDB gateway endpoint to this VPC.
     *
     * @deprecated use `addGatewayEndpoint()` instead
     */
    addDynamoDbEndpoint(id: string, subnets?: SubnetSelection[]): GatewayVpcEndpoint;
    private createNatGateways;
    /**
     * createSubnets creates the subnets specified by the subnet configuration
     * array or creates the `DEFAULT_SUBNETS` configuration
     */
    private createSubnets;
    private createSubnetResources;
}
/**
 * Specify configuration parameters for a VPC subnet.
 *
 * @stability stable
 */
export interface SubnetProps {
    /**
     * The availability zone for the subnet.
     *
     * @stability stable
     */
    readonly availabilityZone: string;
    /**
     * The VPC which this subnet is part of.
     *
     * @stability stable
     */
    readonly vpcId: string;
    /**
     * The CIDR notation for this subnet.
     *
     * @stability stable
     */
    readonly cidrBlock: string;
    /**
     * Controls if a public IP is associated to an instance at launch.
     *
     * @default true in Subnet.Public, false in Subnet.Private or Subnet.Isolated.
     * @stability stable
     */
    readonly mapPublicIpOnLaunch?: boolean;
}
/**
 * Represents a new VPC subnet resource.
 *
 * @stability stable
 * @resource AWS::EC2::Subnet
 */
export declare class Subnet extends Resource implements ISubnet {
    /**
     * @stability stable
     */
    static isVpcSubnet(x: any): x is Subnet;
    /**
     * @stability stable
     */
    static fromSubnetAttributes(scope: Construct, id: string, attrs: SubnetAttributes): ISubnet;
    /**
     * Import existing subnet from id.
     *
     * @stability stable
     */
    static fromSubnetId(scope: Construct, id: string, subnetId: string): ISubnet;
    /**
     * The Availability Zone the subnet is located in.
     *
     * @stability stable
     */
    readonly availabilityZone: string;
    /**
     * The IPv4 CIDR block for this subnet.
     *
     * @stability stable
     * @attribute true
     */
    readonly ipv4CidrBlock: string;
    /**
     * The subnetId for this particular subnet.
     *
     * @stability stable
     */
    readonly subnetId: string;
    /**
     * @stability stable
     * @attribute true
     */
    readonly subnetVpcId: string;
    /**
     * @stability stable
     * @attribute true
     */
    readonly subnetAvailabilityZone: string;
    /**
     * @stability stable
     * @attribute true
     */
    readonly subnetIpv6CidrBlocks: string[];
    /**
     * The Amazon Resource Name (ARN) of the Outpost for this subnet (if one exists).
     *
     * @stability stable
     * @attribute true
     */
    readonly subnetOutpostArn: string;
    /**
     * @stability stable
     * @attribute true
     */
    readonly subnetNetworkAclAssociationId: string;
    /**
     * Parts of this VPC subnet.
     *
     * @stability stable
     */
    readonly dependencyElements: IDependable[];
    /**
     * The routeTableId attached to this subnet.
     *
     * @stability stable
     */
    readonly routeTable: IRouteTable;
    /**
     * Dependable that can be depended upon to force internet connectivity established on the VPC.
     *
     * @stability stable
     */
    readonly internetConnectivityEstablished: IDependable;
    private readonly _internetConnectivityEstablished;
    private _networkAcl;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: SubnetProps);
    /**
     * Create a default route that points to a passed IGW, with a dependency on the IGW's attachment to the VPC.
     *
     * @param gatewayId the logical ID (ref) of the gateway attached to your VPC.
     * @param gatewayAttachment the gateway attachment construct to be added as a dependency.
     * @stability stable
     */
    addDefaultInternetRoute(gatewayId: string, gatewayAttachment: IDependable): void;
    /**
     * Network ACL associated with this Subnet.
     *
     * Upon creation, this is the default ACL which allows all traffic, except
     * explicit DENY entries that you add.
     *
     * You can replace it with a custom ACL which denies all traffic except
     * the explicit ALLOW entries that you add by creating a `NetworkAcl`
     * object and calling `associateNetworkAcl()`.
     *
     * @stability stable
     */
    get networkAcl(): INetworkAcl;
    /**
     * Adds an entry to this subnets route table that points to the passed NATGatewayId.
     *
     * @param natGatewayId The ID of the NAT gateway.
     * @stability stable
     */
    addDefaultNatRoute(natGatewayId: string): void;
    /**
     * Adds an entry to this subnets route table.
     *
     * @stability stable
     */
    addRoute(id: string, options: AddRouteOptions): void;
    /**
     * Associate a Network ACL with this subnet.
     *
     * @stability stable
     */
    associateNetworkAcl(id: string, networkAcl: INetworkAcl): void;
}
/**
 * Options for adding a new route to a subnet.
 *
 * @stability stable
 */
export interface AddRouteOptions {
    /**
     * IPv4 range this route applies to.
     *
     * @default '0.0.0.0/0'
     * @stability stable
     */
    readonly destinationCidrBlock?: string;
    /**
     * IPv6 range this route applies to.
     *
     * @default - Uses IPv6
     * @stability stable
     */
    readonly destinationIpv6CidrBlock?: string;
    /**
     * What type of router to route this traffic to.
     *
     * @stability stable
     */
    readonly routerType: RouterType;
    /**
     * The ID of the router.
     *
     * Can be an instance ID, gateway ID, etc, depending on the router type.
     *
     * @stability stable
     */
    readonly routerId: string;
    /**
     * Whether this route will enable internet connectivity.
     *
     * If true, this route will be added before any AWS resources that depend
     * on internet connectivity in the VPC will be created.
     *
     * @default false
     * @stability stable
     */
    readonly enablesInternetConnectivity?: boolean;
}
/**
 * Type of router used in route.
 *
 * @stability stable
 */
export declare enum RouterType {
    /**
     * Egress-only Internet Gateway.
     *
     * @stability stable
     */
    EGRESS_ONLY_INTERNET_GATEWAY = "EgressOnlyInternetGateway",
    /**
     * Internet Gateway.
     *
     * @stability stable
     */
    GATEWAY = "Gateway",
    /**
     * Instance.
     *
     * @stability stable
     */
    INSTANCE = "Instance",
    /**
     * NAT Gateway.
     *
     * @stability stable
     */
    NAT_GATEWAY = "NatGateway",
    /**
     * Network Interface.
     *
     * @stability stable
     */
    NETWORK_INTERFACE = "NetworkInterface",
    /**
     * VPC peering connection.
     *
     * @stability stable
     */
    VPC_PEERING_CONNECTION = "VpcPeeringConnection"
}
/**
 * @stability stable
 */
export interface PublicSubnetProps extends SubnetProps {
}
/**
 * @stability stable
 */
export interface IPublicSubnet extends ISubnet {
}
/**
 * @stability stable
 */
export interface PublicSubnetAttributes extends SubnetAttributes {
}
/**
 * Represents a public VPC subnet resource.
 *
 * @stability stable
 */
export declare class PublicSubnet extends Subnet implements IPublicSubnet {
    /**
     * @stability stable
     */
    static fromPublicSubnetAttributes(scope: Construct, id: string, attrs: PublicSubnetAttributes): IPublicSubnet;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: PublicSubnetProps);
    /**
     * Creates a new managed NAT gateway attached to this public subnet.
     *
     * Also adds the EIP for the managed NAT.
     *
     * @returns A ref to the the NAT Gateway ID
     * @stability stable
     */
    addNatGateway(): CfnNatGateway;
}
/**
 * @stability stable
 */
export interface PrivateSubnetProps extends SubnetProps {
}
/**
 * @stability stable
 */
export interface IPrivateSubnet extends ISubnet {
}
/**
 * @stability stable
 */
export interface PrivateSubnetAttributes extends SubnetAttributes {
}
/**
 * Represents a private VPC subnet resource.
 *
 * @stability stable
 */
export declare class PrivateSubnet extends Subnet implements IPrivateSubnet {
    /**
     * @stability stable
     */
    static fromPrivateSubnetAttributes(scope: Construct, id: string, attrs: PrivateSubnetAttributes): IPrivateSubnet;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: PrivateSubnetProps);
}
export {};
