"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MultipartUserData = exports.MultipartBody = exports.UserData = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const machine_image_1 = require("./machine-image");
/**
 * Instance User Data.
 *
 * @stability stable
 */
class UserData {
    /**
     * Create a userdata object for Linux hosts.
     *
     * @stability stable
     */
    static forLinux(options = {}) {
        return new LinuxUserData(options);
    }
    /**
     * Create a userdata object for Windows hosts.
     *
     * @stability stable
     */
    static forWindows() {
        return new WindowsUserData();
    }
    /**
     * Create a userdata object with custom content.
     *
     * @stability stable
     */
    static custom(content) {
        const userData = new CustomUserData();
        userData.addCommands(content);
        return userData;
    }
    /**
     * @stability stable
     */
    static forOperatingSystem(os) {
        switch (os) {
            case machine_image_1.OperatingSystemType.LINUX: return UserData.forLinux();
            case machine_image_1.OperatingSystemType.WINDOWS: return UserData.forWindows();
            case machine_image_1.OperatingSystemType.UNKNOWN: throw new Error('Cannot determine UserData for unknown operating system type');
        }
    }
}
exports.UserData = UserData;
_a = JSII_RTTI_SYMBOL_1;
UserData[_a] = { fqn: "@aws-cdk/aws-ec2.UserData", version: "1.123.0" };
/**
 * Linux Instance User Data
 */
class LinuxUserData extends UserData {
    constructor(props = {}) {
        super();
        this.props = props;
        this.lines = [];
        this.onExitLines = [];
    }
    addCommands(...commands) {
        this.lines.push(...commands);
    }
    addOnExitCommands(...commands) {
        this.onExitLines.push(...commands);
    }
    render() {
        var _d;
        const shebang = (_d = this.props.shebang) !== null && _d !== void 0 ? _d : '#!/bin/bash';
        return [shebang, ...(this.renderOnExitLines()), ...this.lines].join('\n');
    }
    addS3DownloadCommand(params) {
        const s3Path = `s3://${params.bucket.bucketName}/${params.bucketKey}`;
        const localPath = (params.localFile && params.localFile.length !== 0) ? params.localFile : `/tmp/${params.bucketKey}`;
        this.addCommands(`mkdir -p $(dirname '${localPath}')`, `aws s3 cp '${s3Path}' '${localPath}'`);
        return localPath;
    }
    addExecuteFileCommand(params) {
        var _d;
        this.addCommands('set -e', `chmod +x '${params.filePath}'`, `'${params.filePath}' ${(_d = params.arguments) !== null && _d !== void 0 ? _d : ''}`.trim());
    }
    addSignalOnExitCommand(resource) {
        const stack = core_1.Stack.of(resource);
        const resourceID = stack.getLogicalId(resource.node.defaultChild);
        this.addOnExitCommands(`/opt/aws/bin/cfn-signal --stack ${stack.stackName} --resource ${resourceID} --region ${stack.region} -e $exitCode || echo 'Failed to send Cloudformation Signal'`);
    }
    renderOnExitLines() {
        if (this.onExitLines.length > 0) {
            return ['function exitTrap(){', 'exitCode=$?', ...this.onExitLines, '}', 'trap exitTrap EXIT'];
        }
        return [];
    }
}
/**
 * Windows Instance User Data
 */
class WindowsUserData extends UserData {
    constructor() {
        super();
        this.lines = [];
        this.onExitLines = [];
    }
    addCommands(...commands) {
        this.lines.push(...commands);
    }
    addOnExitCommands(...commands) {
        this.onExitLines.push(...commands);
    }
    render() {
        return `<powershell>${[...(this.renderOnExitLines()),
            ...this.lines,
            ...(this.onExitLines.length > 0 ? ['throw "Success"'] : [])].join('\n')}</powershell>`;
    }
    addS3DownloadCommand(params) {
        const localPath = (params.localFile && params.localFile.length !== 0) ? params.localFile : `C:/temp/${params.bucketKey}`;
        this.addCommands(`mkdir (Split-Path -Path '${localPath}' ) -ea 0`, `Read-S3Object -BucketName '${params.bucket.bucketName}' -key '${params.bucketKey}' -file '${localPath}' -ErrorAction Stop`);
        return localPath;
    }
    addExecuteFileCommand(params) {
        var _d;
        this.addCommands(`&'${params.filePath}' ${(_d = params.arguments) !== null && _d !== void 0 ? _d : ''}`.trim(), `if (!$?) { Write-Error 'Failed to execute the file "${params.filePath}"' -ErrorAction Stop }`);
    }
    addSignalOnExitCommand(resource) {
        const stack = core_1.Stack.of(resource);
        const resourceID = stack.getLogicalId(resource.node.defaultChild);
        this.addOnExitCommands(`cfn-signal --stack ${stack.stackName} --resource ${resourceID} --region ${stack.region} --success ($success.ToString().ToLower())`);
    }
    renderOnExitLines() {
        if (this.onExitLines.length > 0) {
            return ['trap {', '$success=($PSItem.Exception.Message -eq "Success")', ...this.onExitLines, 'break', '}'];
        }
        return [];
    }
}
/**
 * Custom Instance User Data
 */
class CustomUserData extends UserData {
    constructor() {
        super();
        this.lines = [];
    }
    addCommands(...commands) {
        this.lines.push(...commands);
    }
    addOnExitCommands() {
        throw new Error('CustomUserData does not support addOnExitCommands, use UserData.forLinux() or UserData.forWindows() instead.');
    }
    render() {
        return this.lines.join('\n');
    }
    addS3DownloadCommand() {
        throw new Error('CustomUserData does not support addS3DownloadCommand, use UserData.forLinux() or UserData.forWindows() instead.');
    }
    addExecuteFileCommand() {
        throw new Error('CustomUserData does not support addExecuteFileCommand, use UserData.forLinux() or UserData.forWindows() instead.');
    }
    addSignalOnExitCommand() {
        throw new Error('CustomUserData does not support addSignalOnExitCommand, use UserData.forLinux() or UserData.forWindows() instead.');
    }
}
/**
 * The base class for all classes which can be used as {@link MultipartUserData}.
 *
 * @stability stable
 */
class MultipartBody {
    /**
     * @stability stable
     */
    constructor() {
    }
    /**
     * Constructs the new `MultipartBody` wrapping existing `UserData`. Modification to `UserData` are reflected in subsequent renders of the part.
     *
     * For more information about content types see {@link MultipartBodyOptions.contentType}.
     *
     * @param userData user data to wrap into body part.
     * @param contentType optional content type, if default one should not be used.
     * @stability stable
     */
    static fromUserData(userData, contentType) {
        return new MultipartBodyUserDataWrapper(userData, contentType);
    }
    /**
     * Constructs the raw `MultipartBody` using specified body, content type and transfer encoding.
     *
     * When transfer encoding is specified (typically as Base64), it's caller responsibility to convert body to
     * Base64 either by wrapping with `Fn.base64` or by converting it by other converters.
     *
     * @stability stable
     */
    static fromRawBody(opts) {
        return new MultipartBodyRaw(opts);
    }
}
exports.MultipartBody = MultipartBody;
_b = JSII_RTTI_SYMBOL_1;
MultipartBody[_b] = { fqn: "@aws-cdk/aws-ec2.MultipartBody", version: "1.123.0" };
/**
 * Content type for shell scripts.
 *
 * @stability stable
 */
MultipartBody.SHELL_SCRIPT = 'text/x-shellscript; charset="utf-8"';
/**
 * Content type for boot hooks.
 *
 * @stability stable
 */
MultipartBody.CLOUD_BOOTHOOK = 'text/cloud-boothook; charset="utf-8"';
/**
 * The raw part of multi-part user data, which can be added to {@link MultipartUserData}.
 */
class MultipartBodyRaw extends MultipartBody {
    constructor(props) {
        super();
        this.props = props;
    }
    /**
     * Render body part as the string.
     */
    renderBodyPart() {
        const result = [];
        result.push(`Content-Type: ${this.props.contentType}`);
        if (this.props.transferEncoding != null) {
            result.push(`Content-Transfer-Encoding: ${this.props.transferEncoding}`);
        }
        // One line free after separator
        result.push('');
        if (this.props.body != null) {
            result.push(this.props.body);
            // The new line added after join will be consumed by encapsulating or closing boundary
        }
        return result;
    }
}
/**
 * Wrapper for `UserData`.
 */
class MultipartBodyUserDataWrapper extends MultipartBody {
    constructor(userData, contentType) {
        super();
        this.userData = userData;
        this.contentType = contentType || MultipartBody.SHELL_SCRIPT;
    }
    /**
     * Render body part as the string.
     */
    renderBodyPart() {
        const result = [];
        result.push(`Content-Type: ${this.contentType}`);
        result.push('Content-Transfer-Encoding: base64');
        result.push('');
        result.push(core_1.Fn.base64(this.userData.render()));
        return result;
    }
}
/**
 * Mime multipart user data.
 *
 * This class represents MIME multipart user data, as described in.
 * [Specifying Multiple User Data Blocks Using a MIME Multi Part Archive](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/bootstrap_container_instance.html#multi-part_user_data)
 *
 * @stability stable
 */
class MultipartUserData extends UserData {
    /**
     * @stability stable
     */
    constructor(opts) {
        super();
        this.parts = [];
        let partsSeparator;
        // Validate separator
        if ((opts === null || opts === void 0 ? void 0 : opts.partsSeparator) != null) {
            if (new RegExp(MultipartUserData.BOUNDRY_PATTERN).test(opts.partsSeparator)) {
                throw new Error(`Invalid characters in separator. Separator has to match pattern ${MultipartUserData.BOUNDRY_PATTERN}`);
            }
            else {
                partsSeparator = opts.partsSeparator;
            }
        }
        else {
            partsSeparator = '+AWS+CDK+User+Data+Separator==';
        }
        this.opts = {
            partsSeparator: partsSeparator,
        };
    }
    /**
     * Adds a part to the list of parts.
     *
     * @stability stable
     */
    addPart(part) {
        this.parts.push(part);
    }
    /**
     * Adds a multipart part based on a UserData object.
     *
     * If `makeDefault` is true, then the UserData added by this method
     * will also be the target of calls to the `add*Command` methods on
     * this MultipartUserData object.
     *
     * If `makeDefault` is false, then this is the same as calling:
     *
     * ```ts
     * multiPart.addPart(MultipartBody.fromUserData(userData, contentType));
     * ```
     *
     * An undefined `makeDefault` defaults to either:
     * - `true` if no default UserData has been set yet; or
     * - `false` if there is no default UserData set.
     *
     * @stability stable
     */
    addUserDataPart(userData, contentType, makeDefault) {
        this.addPart(MultipartBody.fromUserData(userData, contentType));
        makeDefault = makeDefault !== null && makeDefault !== void 0 ? makeDefault : (this.defaultUserData === undefined ? true : false);
        if (makeDefault) {
            this.defaultUserData = userData;
        }
    }
    /**
     * Render the UserData for use in a construct.
     *
     * @stability stable
     */
    render() {
        const boundary = this.opts.partsSeparator;
        // Now build final MIME archive - there are few changes from MIME message which are accepted by cloud-init:
        // - MIME RFC uses CRLF to separate lines - cloud-init is fine with LF \n only
        // Note: new lines matters, matters a lot.
        var resultArchive = new Array();
        resultArchive.push(`Content-Type: multipart/mixed; boundary="${boundary}"`);
        resultArchive.push('MIME-Version: 1.0');
        // Add new line, the next one will be boundary (encapsulating or closing)
        // so this line will count into it.
        resultArchive.push('');
        // Add parts - each part starts with boundary
        this.parts.forEach(part => {
            resultArchive.push(`--${boundary}`);
            resultArchive.push(...part.renderBodyPart());
        });
        // Add closing boundary
        resultArchive.push(`--${boundary}--`);
        resultArchive.push(''); // Force new line at the end
        return resultArchive.join('\n');
    }
    /**
     * Adds commands to download a file from S3.
     *
     * @stability stable
     */
    addS3DownloadCommand(params) {
        if (this.defaultUserData) {
            return this.defaultUserData.addS3DownloadCommand(params);
        }
        else {
            throw new Error(MultipartUserData.USE_PART_ERROR);
        }
    }
    /**
     * Adds commands to execute a file.
     *
     * @stability stable
     */
    addExecuteFileCommand(params) {
        if (this.defaultUserData) {
            this.defaultUserData.addExecuteFileCommand(params);
        }
        else {
            throw new Error(MultipartUserData.USE_PART_ERROR);
        }
    }
    /**
     * Adds a command which will send a cfn-signal when the user data script ends.
     *
     * @stability stable
     */
    addSignalOnExitCommand(resource) {
        if (this.defaultUserData) {
            this.defaultUserData.addSignalOnExitCommand(resource);
        }
        else {
            throw new Error(MultipartUserData.USE_PART_ERROR);
        }
    }
    /**
     * Add one or more commands to the user data.
     *
     * @stability stable
     */
    addCommands(...commands) {
        if (this.defaultUserData) {
            this.defaultUserData.addCommands(...commands);
        }
        else {
            throw new Error(MultipartUserData.USE_PART_ERROR);
        }
    }
    /**
     * Add one or more commands to the user data that will run when the script exits.
     *
     * @stability stable
     */
    addOnExitCommands(...commands) {
        if (this.defaultUserData) {
            this.defaultUserData.addOnExitCommands(...commands);
        }
        else {
            throw new Error(MultipartUserData.USE_PART_ERROR);
        }
    }
}
exports.MultipartUserData = MultipartUserData;
_c = JSII_RTTI_SYMBOL_1;
MultipartUserData[_c] = { fqn: "@aws-cdk/aws-ec2.MultipartUserData", version: "1.123.0" };
MultipartUserData.USE_PART_ERROR = 'MultipartUserData only supports this operation if it has a default UserData. Call addUserDataPart with makeDefault=true.';
MultipartUserData.BOUNDRY_PATTERN = '[^a-zA-Z0-9()+,-./:=?]';
//# sourceMappingURL=data:application/json;base64,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