"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.KinesisStreamsToKinesisFirehoseToS3 = void 0;
const kdfToS3 = require("@aws-solutions-constructs/aws-kinesisfirehose-s3");
const core_1 = require("@aws-cdk/core");
const defaults = require("@aws-solutions-constructs/core");
const iam = require("@aws-cdk/aws-iam");
const core_2 = require("@aws-solutions-constructs/core");
class KinesisStreamsToKinesisFirehoseToS3 extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the KinesisStreamsToKinesisFirehoseToS3 class.
     * @since 1.68.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Setup the Kinesis Stream
        this.kinesisStream = defaults.buildKinesisStream(this, {
            existingStreamObj: props.existingStreamObj,
            kinesisStreamProps: props.kinesisStreamProps
        });
        const kinesisStreamsRole = new iam.Role(scope, 'KinesisStreamsRole', {
            assumedBy: new iam.ServicePrincipal('firehose.amazonaws.com'),
            inlinePolicies: {
                KinesisStreamsRoleRolePolicy: new iam.PolicyDocument({
                    statements: [new iam.PolicyStatement({
                            actions: [
                                "kinesis:DescribeStream",
                                "kinesis:GetShardIterator",
                                "kinesis:GetRecords",
                                "kinesis:ListShards"
                            ],
                            resources: [this.kinesisStream.streamArn]
                        })]
                })
            }
        });
        // This Construct requires that the deliveryStreamType be overriden regardless of what is specified in the user props
        if (props.kinesisFirehoseProps) {
            if (props.kinesisFirehoseProps.deliveryStreamType !== undefined) {
                defaults.printWarning('Overriding deliveryStreamType type to be KinesisStreamAsSource');
            }
            if (props.kinesisFirehoseProps.kinesisStreamSourceConfiguration !== undefined) {
                defaults.printWarning('Overriding kinesisStreamSourceConfiguration');
            }
        }
        const _kinesisFirehoseProps = {
            deliveryStreamType: 'KinesisStreamAsSource',
            kinesisStreamSourceConfiguration: {
                kinesisStreamArn: this.kinesisStream.streamArn,
                roleArn: kinesisStreamsRole.roleArn
            }
        };
        const kdfToS3Construct = new kdfToS3.KinesisFirehoseToS3(this, 'KinesisFirehoseToS3', {
            kinesisFirehoseProps: core_2.overrideProps(props.kinesisFirehoseProps, _kinesisFirehoseProps),
            existingBucketObj: props.existingBucketObj,
            bucketProps: props.bucketProps,
            logGroupProps: props.logGroupProps
        });
        this.kinesisFirehose = kdfToS3Construct.kinesisFirehose;
        this.kinesisFirehoseRole = kdfToS3Construct.kinesisFirehoseRole;
        this.kinesisFirehoseLogGroup = kdfToS3Construct.kinesisFirehoseLogGroup;
        if (props.createCloudWatchAlarms === undefined || props.createCloudWatchAlarms) {
            // Deploy best practices CW Alarms for Kinesis Stream
            this.cloudwatchAlarms = defaults.buildKinesisStreamCWAlarms(this);
        }
    }
}
exports.KinesisStreamsToKinesisFirehoseToS3 = KinesisStreamsToKinesisFirehoseToS3;
//# sourceMappingURL=data:application/json;base64,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