"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JobExecutable = exports.JobType = exports.PythonVersion = exports.JobLanguage = exports.GlueVersion = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * AWS Glue version determines the versions of Apache Spark and Python that are available to the job.
 *
 * @see https://docs.aws.amazon.com/glue/latest/dg/add-job.html.
 *
 * If you need to use a GlueVersion that doesn't exist as a static member, you
 * can instantiate a `GlueVersion` object, e.g: `GlueVersion.of('1.5')`.
 */
class GlueVersion {
    constructor(name) {
        this.name = name;
    }
    /**
     * Custom Glue version
     * @param version custom version
     */
    static of(version) {
        return new GlueVersion(version);
    }
}
exports.GlueVersion = GlueVersion;
_a = JSII_RTTI_SYMBOL_1;
GlueVersion[_a] = { fqn: "@aws-cdk/aws-glue.GlueVersion", version: "1.197.0" };
/**
 * Glue version using Spark 2.2.1 and Python 2.7
 */
GlueVersion.V0_9 = new GlueVersion('0.9');
/**
 * Glue version using Spark 2.4.3, Python 2.7 and Python 3.6
 */
GlueVersion.V1_0 = new GlueVersion('1.0');
/**
 * Glue version using Spark 2.4.3 and Python 3.7
 */
GlueVersion.V2_0 = new GlueVersion('2.0');
/**
 * Glue version using Spark 3.1.1 and Python 3.7
 */
GlueVersion.V3_0 = new GlueVersion('3.0');
/**
 * Runtime language of the Glue job
 */
var JobLanguage;
(function (JobLanguage) {
    /**
     * Scala
     */
    JobLanguage["SCALA"] = "scala";
    /**
     * Python
     */
    JobLanguage["PYTHON"] = "python";
})(JobLanguage = exports.JobLanguage || (exports.JobLanguage = {}));
/**
 * Python version
 */
var PythonVersion;
(function (PythonVersion) {
    /**
     * Python 2 (the exact version depends on GlueVersion and JobCommand used)
     */
    PythonVersion["TWO"] = "2";
    /**
     * Python 3 (the exact version depends on GlueVersion and JobCommand used)
     */
    PythonVersion["THREE"] = "3";
})(PythonVersion = exports.PythonVersion || (exports.PythonVersion = {}));
/**
 * The job type.
 *
 * If you need to use a JobType that doesn't exist as a static member, you
 * can instantiate a `JobType` object, e.g: `JobType.of('other name')`.
 */
class JobType {
    constructor(name) {
        this.name = name;
    }
    /**
     * Custom type name
     * @param name type name
     */
    static of(name) {
        return new JobType(name);
    }
}
exports.JobType = JobType;
_b = JSII_RTTI_SYMBOL_1;
JobType[_b] = { fqn: "@aws-cdk/aws-glue.JobType", version: "1.197.0" };
/**
 * Command for running a Glue ETL job.
 */
JobType.ETL = new JobType('glueetl');
/**
 * Command for running a Glue streaming job.
 */
JobType.STREAMING = new JobType('gluestreaming');
/**
 * Command for running a Glue python shell job.
 */
JobType.PYTHON_SHELL = new JobType('pythonshell');
/**
 * The executable properties related to the Glue job's GlueVersion, JobType and code
 */
class JobExecutable {
    constructor(config) {
        if (JobType.PYTHON_SHELL === config.type) {
            if (config.language !== JobLanguage.PYTHON) {
                throw new Error('Python shell requires the language to be set to Python');
            }
            if ([GlueVersion.V0_9, GlueVersion.V2_0, GlueVersion.V3_0].includes(config.glueVersion)) {
                throw new Error(`Specified GlueVersion ${config.glueVersion.name} does not support Python Shell`);
            }
        }
        if (config.extraJarsFirst && [GlueVersion.V0_9, GlueVersion.V1_0].includes(config.glueVersion)) {
            throw new Error(`Specified GlueVersion ${config.glueVersion.name} does not support extraJarsFirst`);
        }
        if (config.pythonVersion === PythonVersion.TWO && ![GlueVersion.V0_9, GlueVersion.V1_0].includes(config.glueVersion)) {
            throw new Error(`Specified GlueVersion ${config.glueVersion.name} does not support PythonVersion ${config.pythonVersion}`);
        }
        if (JobLanguage.PYTHON !== config.language && config.extraPythonFiles) {
            throw new Error('extraPythonFiles is not supported for languages other than JobLanguage.PYTHON');
        }
        this.config = config;
    }
    /**
     * Create Scala executable props for Apache Spark ETL job.
     *
     * @param props Scala Apache Spark Job props
     */
    static scalaEtl(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_ScalaJobExecutableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.scalaEtl);
            }
            throw error;
        }
        return new JobExecutable({
            ...props,
            type: JobType.ETL,
            language: JobLanguage.SCALA,
        });
    }
    /**
     * Create Scala executable props for Apache Spark Streaming job.
     *
     * @param props Scala Apache Spark Job props
     */
    static scalaStreaming(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_ScalaJobExecutableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.scalaStreaming);
            }
            throw error;
        }
        return new JobExecutable({
            ...props,
            type: JobType.STREAMING,
            language: JobLanguage.SCALA,
        });
    }
    /**
     * Create Python executable props for Apache Spark ETL job.
     *
     * @param props Python Apache Spark Job props
     */
    static pythonEtl(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_PythonSparkJobExecutableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.pythonEtl);
            }
            throw error;
        }
        return new JobExecutable({
            ...props,
            type: JobType.ETL,
            language: JobLanguage.PYTHON,
        });
    }
    /**
     * Create Python executable props for Apache Spark Streaming job.
     *
     * @param props Python Apache Spark Job props
     */
    static pythonStreaming(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_PythonSparkJobExecutableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.pythonStreaming);
            }
            throw error;
        }
        return new JobExecutable({
            ...props,
            type: JobType.STREAMING,
            language: JobLanguage.PYTHON,
        });
    }
    /**
     * Create Python executable props for python shell jobs.
     *
     * @param props Python Shell Job props.
     */
    static pythonShell(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_PythonShellExecutableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.pythonShell);
            }
            throw error;
        }
        return new JobExecutable({
            ...props,
            type: JobType.PYTHON_SHELL,
            language: JobLanguage.PYTHON,
        });
    }
    /**
     * Create a custom JobExecutable.
     *
     * @param config custom job executable configuration.
     */
    static of(config) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_JobExecutableConfig(config);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.of);
            }
            throw error;
        }
        return new JobExecutable(config);
    }
    /**
     * Called during Job initialization to get JobExecutableConfig.
     */
    bind() {
        return this.config;
    }
}
exports.JobExecutable = JobExecutable;
_c = JSII_RTTI_SYMBOL_1;
JobExecutable[_c] = { fqn: "@aws-cdk/aws-glue.JobExecutable", version: "1.197.0" };
//# sourceMappingURL=data:application/json;base64,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