"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ec2Deployer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const codedeploy = require("@aws-cdk/aws-codedeploy");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const customresources = require("@aws-cdk/custom-resources");
;
/**
 * Represents a Deployer resource for deploying an artifact to EC2 using CodeDeploy.
 *
 * @stability stable
 */
class Ec2Deployer extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        // Validate that props.deploymentTimeout is less than 2 hours, per maximum value accepted by downstream customresources.Provider.totalTimeout
        if (props.deploymentTimeout && props.deploymentTimeout.toMilliseconds() > Ec2Deployer.MAX_DEPLOYMENT_TIMEOUT.toMilliseconds()) { // have to convert to milliseconds in case the cdk.Duration is passed in milliseconds
            throw new Error(`Invalid prop: deploymentTimeout must be less than ${Ec2Deployer.MAX_DEPLOYMENT_TIMEOUT.toHumanString()}.`);
        }
        // Validate that at least one instanceRole is supplied if we cannot get them from deploymentGroup.autoScalingGroups
        if (!props.deploymentGroup.autoScalingGroups && (!props.instanceRoles || props.instanceRoles.length === 0)) {
            throw new Error('If deploymentGroup is of type IServerDeploymentGroup, you must supply at least one role in instanceRoles.');
        }
        super(scope, id);
        // Set defaults for any missing props
        this.code = props.code.bind(this);
        this.deploymentGroup = props.deploymentGroup;
        this.waitToComplete = props.waitToComplete !== undefined ? props.waitToComplete : true;
        this.deploymentTimeout = this.waitToComplete ? props.deploymentTimeout || cdk.Duration.minutes(5) : undefined; // can only be defined if waitToComplete=true because of downstream customresources.Provider.totalTimeout
        // Create OnEventHandler Lambda function for custom resource
        // Can't use SingletonFunction because permissions are dependent on props passed into each Ec2Deployer instance
        const onEvent = new lambda.Function(this, 'OnEventHandler', {
            // const onEvent = new lambda.SingletonFunction(this, 'OnEventHandler', {
            //   uuid: '3a9c56a9-1dd5-42dc-af2f-10b76edde830',
            code: lambda.Code.fromAsset(path.join(__dirname, '../custom-resource-runtime/ec2-deployer')),
            runtime: lambda.Runtime.PYTHON_3_8,
            handler: 'index.on_event',
            initialPolicy: [
                new iam.PolicyStatement({
                    actions: ['codedeploy:GetDeploymentConfig'],
                    resources: [codedeploy.ServerDeploymentConfig.ONE_AT_A_TIME.deploymentConfigArn],
                }),
                new iam.PolicyStatement({
                    actions: ['codedeploy:CreateDeployment'],
                    resources: [this.deploymentGroup.deploymentGroupArn],
                }),
                new iam.PolicyStatement({
                    actions: ['codedeploy:GetApplicationRevision', 'codedeploy:RegisterApplicationRevision'],
                    resources: [this.deploymentGroup.application.applicationArn],
                }),
            ],
        });
        // Create IsCompleteHandler Lambda function for custom resource, only if waitToComplete=true
        // Can't use SingletonFunction because permissions are dependent on props passed into each Ec2Deployer instance
        let isComplete = undefined;
        if (this.waitToComplete) {
            // isComplete = new lambda.SingletonFunction(this, 'IsCompleteHandler', {
            //   uuid: 'f58e4e2e-8b7e-4bd0-b33b-c5c9f19f5546',
            isComplete = new lambda.Function(this, 'IsCompleteHandler', {
                code: lambda.Code.fromAsset(path.join(__dirname, '../custom-resource-runtime/ec2-deployer')),
                runtime: lambda.Runtime.PYTHON_3_8,
                handler: 'index.is_complete',
                initialPolicy: [
                    new iam.PolicyStatement({
                        resources: [this.deploymentGroup.deploymentGroupArn],
                        actions: ['codedeploy:GetDeployment'],
                    }),
                ],
            });
        }
        // Create provider for custom resource
        const deployerProvider = new customresources.Provider(this, 'Provider', {
            onEventHandler: onEvent,
            totalTimeout: this.deploymentTimeout,
            isCompleteHandler: isComplete,
        });
        // Ensure ASGs have read access to code S3 object for deployment
        const policyStatement = new iam.PolicyStatement({
            actions: ['s3:GetObject*'],
            resources: [`arn:${cdk.Stack.of(this).partition}:s3:::${this.code.s3Location.bucketName}/${this.code.s3Location.objectKey}`],
        });
        if (props.instanceRoles) {
            for (let role of props.instanceRoles) {
                role.addToPrincipalPolicy(policyStatement);
            }
        }
        else {
            for (let asg of this.deploymentGroup.autoScalingGroups) {
                asg.role.addToPrincipalPolicy(policyStatement);
            }
        }
        // Create custom resource that triggers a deployment
        new cdk.CustomResource(this, 'CustomResource', {
            serviceToken: deployerProvider.serviceToken,
            properties: {
                applicationName: this.deploymentGroup.application.applicationName,
                deploymentGroupName: this.deploymentGroup.deploymentGroupName,
                codeS3BucketName: this.code.s3Location.bucketName,
                codeS3ObjectKey: this.code.s3Location.objectKey,
                codeS3ObjectVersion: this.code.s3Location.objectVersion,
            },
        });
    }
}
exports.Ec2Deployer = Ec2Deployer;
_a = JSII_RTTI_SYMBOL_1;
Ec2Deployer[_a] = { fqn: "cdk-deployer.Ec2Deployer", version: "1.0.11" };
/**
 * Maximum allowed value for deploymentTimeout prop.
 *
 * @stability stable
 */
Ec2Deployer.MAX_DEPLOYMENT_TIMEOUT = cdk.Duration.hours(2);
//# sourceMappingURL=data:application/json;base64,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