'''
Client extensions for the authorization code flow, granting private user access to the API.
'''

import os
import ssl
import webbrowser
from datetime import datetime, timedelta
from http.server import BaseHTTPRequestHandler, ThreadingHTTPServer

from cryptography import x509
from cryptography.hazmat.primitives import hashes, serialization
from cryptography.hazmat.primitives.asymmetric import rsa
from cryptography.x509.oid import NameOID


class UserModeAuthMixin:
    '''
    Mixins that enable the client to authenticate against the API using the authorization code flow,
    granting user-level access to private parts of the API.

    Note that for this to work, at least one of the client's redirect URL must be set to
    https://localhost:4443 in FF Logs' client management (or whatever port is used by the client).
    This is because the client will attempt to redirect the user to a web server hosted locally
    by the client in order to capture the authorization response.
    '''

    OAUTH_USER_AUTH_URI = 'https://www.fflogs.com/oauth/authorize'
    OAUTH_CAPTURE_PORT = 4443
    OAUTH_REDIRECT_URI = f'https://localhost:{OAUTH_CAPTURE_PORT}/'

    # How long the self-signed certificate generated for the OAuth redirect last
    CERT_EXPIRY_MINUTES = 5  # minutes
    CERT_PATH = './fflogsapi/fflogs_auth_redirect_cert.pem'
    KEY_PATH = './fflogsapi/fflogs_auth_redirect_key.pem'

    MANUAL_AUTH_RESPONSE = ''

    def set_auth_response(self, response: str) -> None:
        '''
        Manually set the authorization response after user login.

        You can use this if you want to handle the user auth flow yourself,
        but this must be called before the first use of the client.

        When using this, you must also handle token refresh logic yourself.
        '''
        self.MANUAL_AUTH_RESPONSE = response

    def user_auth(self) -> None:
        '''
        Prompt the user to login through their browser and fetch an authorization token.
        '''
        response = ''
        if not self.MANUAL_AUTH_RESPONSE:
            auth_url, _ = self.oauth_session.authorization_url(
                self.OAUTH_USER_AUTH_URI,
            )
            # notifying them that their browser will open for login is not really an option
            input('Press any key to open your browser to authenticate with FF Logs.')

            self._generate_ss_x509_cert()
            print(f'You now have {self.CERT_EXPIRY_MINUTES} minutes to complete sign-in.')
            print('Your browser may warn you about a self-signed certificate. This is normal.')
            webbrowser.open(auth_url, new=1)
            response = self.capture_auth_response()
            self._remove_x509_cert()
        else:
            response = self.MANUAL_AUTH_RESPONSE

        self.token = self.oauth_session.fetch_token(
            self.OAUTH_TOKEN_URL,
            authorization_response=response,
            auth=self.auth,
        )

    def _generate_ss_x509_cert(self) -> None:
        '''
        Generate and save a self-signed x509 certificate to host a HTTPS session with.
        '''
        key = rsa.generate_private_key(
            public_exponent=65537,
            key_size=2048,
        )

        subject = issuer = x509.Name([
            x509.NameAttribute(NameOID.COMMON_NAME, u'localhost')
        ])

        now = datetime.utcnow()
        cert = x509.CertificateBuilder().subject_name(subject) \
            .issuer_name(issuer) \
            .public_key(key.public_key()) \
            .serial_number(x509.random_serial_number()) \
            .not_valid_before(now) \
            .not_valid_after(now + timedelta(minutes=self.CERT_EXPIRY_MINUTES)) \
            .add_extension(
                x509.SubjectAlternativeName([x509.DNSName(u'localhost')]),
                critical=False,
        ).sign(key, hashes.SHA256())

        if not os.path.exists(os.path.dirname(self.KEY_PATH)):
            os.makedirs(self.KEY_PATH)
        with open(self.KEY_PATH, 'wb') as f:
            f.write(key.private_bytes(
                encoding=serialization.Encoding.PEM,
                format=serialization.PrivateFormat.TraditionalOpenSSL,
                encryption_algorithm=serialization.NoEncryption(),
            ))

        if not os.path.exists(os.path.dirname(self.CERT_PATH)):
            os.makedirs(self.CERT_PATH)
        with open(self.CERT_PATH, 'wb') as f:
            f.write(cert.public_bytes(serialization.Encoding.PEM))

    def _remove_x509_cert(self) -> None:
        '''
        Removes the x509 certificate generated by the client.
        '''
        if os.path.exists(self.CERT_PATH):
            os.remove(self.CERT_PATH)
        if os.path.exists(self.KEY_PATH):
            os.remove(self.KEY_PATH)

        # remove the entire dir if it is empty
        # dir = os.path.dirname(self.CERT_PATH)
        # if not os.listdir(dir):
        #     os.remove(dir)

    def capture_auth_response(self) -> str:
        '''
        Capture the authorization response code from the user login flow.
        '''
        auth_response = ''

        class AuthResponseHandler(BaseHTTPRequestHandler):
            def do_GET(self):
                # i'm not a huge fan of this but it works
                nonlocal auth_response
                auth_response = UserModeAuthMixin.OAUTH_REDIRECT_URI + self.path
                self.send_response(200)
                self.send_header('Content-Type', 'text/html')
                self.end_headers()
                self.wfile.write(bytes(
                    '<p>Authorization was successful!</p>',
                    encoding='utf8',
                ))
                self.wfile.write(bytes(
                    '<p>You may now close the browser window and continue.</p>',
                    encoding='utf8',
                ))
                self.wfile.flush()

                self.server.shutdown()

        httpd = ThreadingHTTPServer(('localhost', self.OAUTH_CAPTURE_PORT), AuthResponseHandler)
        httpd.socket = ssl.wrap_socket(
            httpd.socket,
            keyfile=self.KEY_PATH,
            certfile=self.CERT_PATH,
            server_side=True,
        )
        httpd.serve_forever()

        return auth_response
