# coding: utf-8

from __future__ import absolute_import

from bitmovin_api_sdk.common import BaseApi, BitmovinApiLoggerBase
from bitmovin_api_sdk.common.poscheck import poscheck_except
from bitmovin_api_sdk.models.bif import Bif
from bitmovin_api_sdk.models.bitmovin_response import BitmovinResponse
from bitmovin_api_sdk.models.response_envelope import ResponseEnvelope
from bitmovin_api_sdk.models.response_error import ResponseError
from bitmovin_api_sdk.encoding.encodings.streams.bifs.customdata.customdata_api import CustomdataApi
from bitmovin_api_sdk.encoding.encodings.streams.bifs.bif_list_query_params import BifListQueryParams


class BifsApi(BaseApi):
    @poscheck_except(2)
    def __init__(self, api_key, tenant_org_id=None, base_url=None, logger=None):
        # type: (str, str, str, BitmovinApiLoggerBase) -> None

        super(BifsApi, self).__init__(
            api_key=api_key,
            tenant_org_id=tenant_org_id,
            base_url=base_url,
            logger=logger
        )

        self.customdata = CustomdataApi(
            api_key=api_key,
            tenant_org_id=tenant_org_id,
            base_url=base_url,
            logger=logger
        )

    def create(self, encoding_id, stream_id, bif, **kwargs):
        # type: (string_types, string_types, Bif, dict) -> Bif
        """Add a Roku Bif file

        :param encoding_id: Id of the encoding.
        :type encoding_id: string_types, required
        :param stream_id: Id of the stream.
        :type stream_id: string_types, required
        :param bif: The Roku Bif file to be added
        :type bif: Bif, required
        :return: Bif details
        :rtype: Bif
        """

        return self.api_client.post(
            '/encoding/encodings/{encoding_id}/streams/{stream_id}/bifs',
            bif,
            path_params={'encoding_id': encoding_id, 'stream_id': stream_id},
            type=Bif,
            **kwargs
        )

    def delete(self, encoding_id, stream_id, bif_id, **kwargs):
        # type: (string_types, string_types, string_types, dict) -> BitmovinResponse
        """Delete Bif

        :param encoding_id: Id of the encoding.
        :type encoding_id: string_types, required
        :param stream_id: Id of the stream.
        :type stream_id: string_types, required
        :param bif_id: Id of the Bif.
        :type bif_id: string_types, required
        :return: Id of the Bif
        :rtype: BitmovinResponse
        """

        return self.api_client.delete(
            '/encoding/encodings/{encoding_id}/streams/{stream_id}/bifs/{bif_id}',
            path_params={'encoding_id': encoding_id, 'stream_id': stream_id, 'bif_id': bif_id},
            type=BitmovinResponse,
            **kwargs
        )

    def get(self, encoding_id, stream_id, bif_id, **kwargs):
        # type: (string_types, string_types, string_types, dict) -> Bif
        """Bif Details

        :param encoding_id: Id of the encoding.
        :type encoding_id: string_types, required
        :param stream_id: Id of the stream.
        :type stream_id: string_types, required
        :param bif_id: Id of the Bif configuration.
        :type bif_id: string_types, required
        :return: Bif details
        :rtype: Bif
        """

        return self.api_client.get(
            '/encoding/encodings/{encoding_id}/streams/{stream_id}/bifs/{bif_id}',
            path_params={'encoding_id': encoding_id, 'stream_id': stream_id, 'bif_id': bif_id},
            type=Bif,
            **kwargs
        )

    def list(self, encoding_id, stream_id, query_params=None, **kwargs):
        # type: (string_types, string_types, BifListQueryParams, dict) -> Bif
        """List Bifs

        :param encoding_id: Id of the encoding.
        :type encoding_id: string_types, required
        :param stream_id: Id of the stream.
        :type stream_id: string_types, required
        :param query_params: Query parameters
        :type query_params: BifListQueryParams
        :return: List of Bifs
        :rtype: Bif
        """

        return self.api_client.get(
            '/encoding/encodings/{encoding_id}/streams/{stream_id}/bifs',
            path_params={'encoding_id': encoding_id, 'stream_id': stream_id},
            query_params=query_params,
            pagination_response=True,
            type=Bif,
            **kwargs
        )
