"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KinesisFirehoseTransformer = void 0;
const glue = require("@aws-cdk/aws-glue");
const iam = require("@aws-cdk/aws-iam");
const kdf = require("@aws-cdk/aws-kinesisfirehose");
const kms = require("@aws-cdk/aws-kms");
const logs = require("@aws-cdk/aws-logs");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
class KinesisFirehoseTransformer extends cdk.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        const firehoseLogGroup = new logs.LogGroup(this, 'KinesisFirehoseLogGroup', {
            logGroupName: props.logsConfig ? props.logsConfig.logsGroupName : `/aws/kinesis-firehose/${props.deliveryStreamName}`,
            retention: props.logsConfig ? props.logsConfig.logsRetentionDays : logs.RetentionDays.ONE_WEEK,
            removalPolicy: props.logsConfig ? props.logsConfig.logsRemovalPolicy : cdk.RemovalPolicy.RETAIN
        });
        const firehoseLogStream = new logs.LogStream(this, 'KinesisFirehoseLogStream', {
            logGroup: firehoseLogGroup,
            removalPolicy: props.logsConfig ? props.logsConfig.logsRemovalPolicy : cdk.RemovalPolicy.RETAIN
        });
        const firehoseLogStreamArn = `arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:${firehoseLogGroup.logGroupName}:log-stream:${firehoseLogStream.logStreamName}`;
        const partitionKeys = [
            {
                name: 'year',
                type: glue.Schema.SMALL_INT
            },
            {
                name: 'month',
                type: glue.Schema.SMALL_INT
            },
            {
                name: 'day',
                type: glue.Schema.SMALL_INT
            },
            {
                name: 'hour',
                type: glue.Schema.SMALL_INT
            }
        ];
        const kdfTransformerRole = new iam.Role(this, 'FirehoseRole', {
            assumedBy: new iam.ServicePrincipal('firehose.amazonaws.com'),
            inlinePolicies: {
                'GluePermissions': new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            actions: [
                                "glue:GetTableVersions"
                            ],
                            resources: ["*"]
                        })
                    ]
                }),
                'CloudWatchPermissions': new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            actions: ['cloudwatch:PutMetricData'],
                            resources: ['*']
                        })
                    ]
                }),
                'LogsPermissions': new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            actions: ['logs:DescribeLogStreams', 'logs:DescribeLogGroups'],
                            resources: [
                                firehoseLogGroup.logGroupArn,
                                `arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:*`
                            ]
                        })
                    ]
                }),
                'LogsPutPermissions': new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            actions: ['logs:PutLogEvents'],
                            resources: [firehoseLogStreamArn]
                        })
                    ]
                })
            }
        });
        let encryptionConfig;
        if (props.createEncryptionKey) {
            const encryptionKey = new kms.Key(this, 'TransformerKmsKey', {
                alias: `${props.deliveryStreamName}TransformerKey`,
                description: `Encryption key for all data using ${props.deliveryStreamName}`,
                enabled: true,
                enableKeyRotation: true,
                removalPolicy: cdk.RemovalPolicy.RETAIN // how to handle this?
            });
            encryptionKey.grantEncryptDecrypt(kdfTransformerRole);
            encryptionConfig = {
                kmsEncryptionConfig: {
                    awskmsKeyArn: encryptionKey.keyArn
                }
            };
        }
        let backupConfig;
        if (props.sourceBackupConfig) {
            const sourceDatabase = glue.Database.fromDatabaseArn(this, 'GlueSourceDatabase', props.sourceBackupConfig.glueDatabaseArn);
            let sourceBucket;
            if (props.sourceBackupConfig.targetS3BucketArn) {
                sourceBucket = s3.Bucket.fromBucketArn(this, 'SourceS3Bucket', props.sourceBackupConfig.targetS3BucketArn);
            }
            else {
                sourceBucket = new s3.Bucket(this, 'SourceS3Bucket', {});
            }
            sourceBucket.grantReadWrite(kdfTransformerRole);
            kdfTransformerRole.addToPolicy(new iam.PolicyStatement({
                actions: [
                    's3:AbortMultipartUpload',
                    's3:GetBucketLocation',
                    's3:GetObject',
                    's3:ListBucket',
                    's3:ListBucketMultipartUploads',
                    's3:PutObject',
                ],
                resources: [
                    sourceBucket.bucketArn,
                    sourceBucket.bucketArn + '/*'
                ]
            }));
            new glue.Table(this, 'SourceGlueTable', {
                columns: props.sourceBackupConfig.columns,
                dataFormat: glue.DataFormat.JSON,
                database: sourceDatabase,
                tableName: props.sourceBackupConfig.tableName,
                bucket: sourceBucket,
                compressed: false,
                description: `Backup original data table for ${props.sourceBackupConfig.tableName}`,
                s3Prefix: (_a = props.sourceBackupConfig.targetS3prefix) !== null && _a !== void 0 ? _a : `${props.sourceBackupConfig.tableName}/processed/`,
                storedAsSubDirectories: false,
                partitionKeys: partitionKeys
            });
            backupConfig = {
                bucketArn: sourceBucket.bucketArn,
                roleArn: kdfTransformerRole.roleArn,
                bufferingHints: {
                    intervalInSeconds: 60,
                    sizeInMBs: 128
                },
                cloudWatchLoggingOptions: {
                    logGroupName: firehoseLogGroup.logGroupName,
                    logStreamName: 'raw'
                },
                compressionFormat: 'UNCOMPRESSED',
                encryptionConfiguration: encryptionConfig,
                errorOutputPrefix: 'failed/!{firehose:error-output-type}/year=!{timestamp:yyyy}/month=!{timestamp:MM}/day=!{timestamp:dd}/hour=!{timestamp:HH}/',
                prefix: 'raw/year=!{timestamp:yyyy}/month=!{timestamp:MM}/day=!{timestamp:dd}/hour=!{timestamp:HH}/'
            };
        }
        let targetDatabase = glue.Database.fromDatabaseArn(this, 'GlueTargetDatabase', props.targetTableConfig.glueDatabaseArn);
        let targetBucket;
        if (props.targetTableConfig.targetS3BucketArn) {
            targetBucket = s3.Bucket.fromBucketArn(this, 'TargetS3Bucket', props.targetTableConfig.targetS3BucketArn);
        }
        else {
            targetBucket = new s3.Bucket(this, 'TargetS3Bucket', {});
        }
        targetBucket.grantReadWrite(kdfTransformerRole);
        kdfTransformerRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                's3:AbortMultipartUpload',
                's3:GetBucketLocation',
                's3:GetObject',
                's3:ListBucket',
                's3:ListBucketMultipartUploads',
                's3:PutObject',
            ],
            resources: [
                targetBucket.bucketArn,
                targetBucket.bucketArn + '/*'
            ]
        }));
        // append the YYYY/MM/DD/HH columns to the list of columns
        new glue.Table(this, 'TargetParquetTable', {
            columns: props.targetTableConfig.columns,
            dataFormat: glue.DataFormat.PARQUET,
            database: targetDatabase,
            tableName: props.targetTableConfig.tableName,
            bucket: targetBucket,
            compressed: false,
            description: `Target Parquet Table for ${props.targetTableConfig.tableName}`,
            s3Prefix: (_b = props.targetTableConfig.targetS3prefix) !== null && _b !== void 0 ? _b : `${props.targetTableConfig.tableName}/processed/`,
            storedAsSubDirectories: false,
            partitionKeys: partitionKeys
        });
        const transformerDS = new kdf.CfnDeliveryStream(this, 'TransformerDeliveryStream', {
            deliveryStreamName: props.deliveryStreamName,
            deliveryStreamType: 'DirectPut',
            extendedS3DestinationConfiguration: {
                bucketArn: targetBucket.bucketArn,
                roleArn: kdfTransformerRole.roleArn,
                bufferingHints: {
                    intervalInSeconds: 60,
                    sizeInMBs: 128
                },
                cloudWatchLoggingOptions: {
                    enabled: props.enableCloudwatchLogging,
                    logGroupName: firehoseLogGroup.logGroupName,
                    logStreamName: firehoseLogStream.logStreamName
                },
                compressionFormat: 'UNCOMPRESSED',
                dataFormatConversionConfiguration: {
                    enabled: true,
                    inputFormatConfiguration: {
                        deserializer: {
                            openXJsonSerDe: {}
                        }
                    },
                    outputFormatConfiguration: {
                        serializer: {
                            parquetSerDe: {}
                        }
                    },
                    schemaConfiguration: {
                        roleArn: kdfTransformerRole.roleArn,
                        catalogId: cdk.Aws.ACCOUNT_ID,
                        databaseName: targetDatabase.databaseName,
                        tableName: props.targetTableConfig.tableName,
                        region: cdk.Aws.REGION,
                        versionId: 'LATEST'
                    }
                },
                encryptionConfiguration: encryptionConfig,
                errorOutputPrefix: 'failed/!{firehose:error-output-type}/year=!{timestamp:yyyy}/month=!{timestamp:MM}/day=!{timestamp:dd}/hour=!{timestamp:HH}/',
                prefix: 'processed/year=!{timestamp:yyyy}/month=!{timestamp:MM}/day=!{timestamp:dd}/hour=!{timestamp:HH}/',
                processingConfiguration: props.processingConfig,
                s3BackupConfiguration: backupConfig,
                s3BackupMode: props.sourceBackupConfig ? 'Enabled' : 'Disabled'
            }
        });
        this.kinesisFirehoseArn = transformerDS.attrArn;
    }
}
exports.KinesisFirehoseTransformer = KinesisFirehoseTransformer;
//# sourceMappingURL=data:application/json;base64,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