import csv
import datetime
import functools
import hashlib
import io
import re
import tempfile
import uuid
from typing import Literal, Optional

import falcon
import orjson
import peewee
import requests
from falcon.media.validators import jsonschema
from falcon_cors import CORS
from falcon_multipart.middleware import MultipartMiddleware
from PIL import Image
from sentry_sdk.integrations.falcon import FalconIntegration

from robotoff import settings
from robotoff.app import schema
from robotoff.app.auth import BasicAuthDecodeError, basic_decode
from robotoff.app.core import (
    SkipVotedOn,
    SkipVotedType,
    get_image_predictions,
    get_images,
    get_insights,
    get_logo_annotation,
    get_predictions,
    save_annotation,
    update_logo_annotations,
)
from robotoff.app.middleware import DBConnectionMiddleware
from robotoff.elasticsearch import get_es_client
from robotoff.insights.extraction import (
    DEFAULT_OCR_PREDICTION_TYPES,
    extract_ocr_predictions,
)
from robotoff.insights.question import QuestionFormatter, QuestionFormatterFactory
from robotoff.logos import (
    generate_insights_from_annotated_logos,
    generate_insights_from_annotated_logos_job,
    knn_search,
)
from robotoff.models import (
    ImageModel,
    ImagePrediction,
    LogoAnnotation,
    LogoEmbedding,
    Prediction,
    ProductInsight,
    batch_insert,
    db,
)
from robotoff.off import (
    OFFAuthentication,
    generate_image_path,
    get_barcode_from_url,
    get_product,
)
from robotoff.prediction.category import predict_category
from robotoff.prediction.object_detection import ObjectDetectionModelRegistry
from robotoff.prediction.ocr.dataclass import OCRParsingException
from robotoff.products import get_product_dataset_etag
from robotoff.spellcheck import SPELLCHECKERS, Spellchecker
from robotoff.taxonomy import is_prefixed_value, match_taxonomized_value
from robotoff.types import (
    ElasticSearchIndex,
    JSONType,
    NeuralCategoryClassifierModel,
    PredictionType,
    ProductIdentifier,
    ServerType,
)
from robotoff.utils import get_image_from_url, get_logger, http_session
from robotoff.utils.i18n import TranslationStore
from robotoff.utils.text import get_tag
from robotoff.workers.queues import enqueue_in_job, enqueue_job, high_queue, low_queue
from robotoff.workers.tasks import (
    delete_product_insights_job,
    download_product_dataset_job,
    run_import_image_job,
    update_insights_job,
)

logger = get_logger()

settings.init_sentry(integrations=[FalconIntegration()])

es_client = get_es_client()

TRANSLATION_STORE = TranslationStore()
TRANSLATION_STORE.load()


def get_server_type_from_req(
    req: falcon.Request, default: ServerType = ServerType.off
) -> ServerType:
    """Get `ServerType` value from POST x-www-form-urlencoded or GET requests."""
    server_type_str = req.get_param("server_type")

    if server_type_str is None:
        return default

    try:
        return ServerType[server_type_str]
    except KeyError:
        raise falcon.HTTPBadRequest(f"invalid `server_type`: {server_type_str}")


def _get_skip_voted_on(
    auth: Optional[OFFAuthentication], device_id: str
) -> SkipVotedOn:
    """Helper function for constructing SkipVotedOn objects based on request params."""
    if not auth:
        return SkipVotedOn(SkipVotedType.DEVICE_ID, device_id)

    username: Optional[str] = auth.get_username()
    if not username:
        return SkipVotedOn(SkipVotedType.DEVICE_ID, device_id)

    return SkipVotedOn(SkipVotedType.USERNAME, username)


###########
# IMPORTANT: remember to update documentation at doc/references/api.yml if you change API
###########


class ProductInsightResource:
    def on_get(self, req: falcon.Request, resp: falcon.Response, barcode: str):
        response: JSONType = {}
        server_type = get_server_type_from_req(req)
        insights = [
            insight.to_dict()
            for insight in get_insights(
                barcode=barcode, server_type=server_type, limit=None
            )
        ]

        if not insights:
            response["status"] = "no_insights"
        else:
            response["insights"] = insights
            response["status"] = "found"

        resp.media = response


class ProductInsightDetail:
    def on_get(self, req: falcon.Request, resp: falcon.Response, insight_id: uuid.UUID):
        try:
            insight: ProductInsight = ProductInsight.get_by_id(insight_id)
        except ProductInsight.DoesNotExist:
            raise falcon.HTTPNotFound()

        resp.media = insight.to_dict()


class InsightCollection:
    def on_get(self, req: falcon.Request, resp: falcon.Response):
        response: JSONType = {}
        count: int = req.get_param_as_int("count", min_value=1, default=25)
        page: int = req.get_param_as_int("page", min_value=1, default=1)
        keep_types: Optional[list[str]] = req.get_param_as_list(
            "insight_types", required=False
        )
        barcode: Optional[str] = req.get_param("barcode")
        country: Optional[str] = req.get_param("country")
        annotated: Optional[bool] = req.get_param_as_bool("annotated")
        annotation: Optional[int] = req.get_param_as_int("annotation")
        value_tag: str = req.get_param("value_tag")
        brands = req.get_param_as_list("brands") or None
        predictor = req.get_param("predictor")
        server_type = get_server_type_from_req(req)

        if keep_types:
            # Limit the number of types to prevent slow SQL queries
            keep_types = keep_types[:10]

        if brands is not None:
            # Limit the number of brands to prevent slow SQL queries
            brands = brands[:10]

        get_insights_ = functools.partial(
            get_insights,
            server_type=server_type,
            keep_types=keep_types,
            country=country,
            value_tag=value_tag,
            brands=brands,
            annotated=annotated,
            annotation=annotation,
            barcode=barcode,
            predictor=predictor,
        )

        offset: int = (page - 1) * count
        insights = [i.to_dict() for i in get_insights_(limit=count, offset=offset)]
        response["count"] = get_insights_(count=True)

        if not insights:
            response["insights"] = []
            response["status"] = "no_insights"
        else:
            response["insights"] = insights
            response["status"] = "found"

        resp.media = response


class RandomInsightResource:
    def on_get(self, req: falcon.Request, resp: falcon.Response):
        response: JSONType = {}

        insight_type: Optional[str] = req.get_param("type")
        country: Optional[str] = req.get_param("country")
        value_tag: Optional[str] = req.get_param("value_tag")
        count: int = req.get_param_as_int("count", min_value=1, default=25)
        predictor = req.get_param("predictor")
        server_type = get_server_type_from_req(req)

        keep_types = [insight_type] if insight_type else None
        get_insights_ = functools.partial(
            get_insights,
            server_type=server_type,
            keep_types=keep_types,
            country=country,
            value_tag=value_tag,
            order_by="random",
            predictor=predictor,
        )

        insights = [insight.to_dict() for insight in get_insights_(limit=count)]
        response["count"] = get_insights_(count=True)

        if not insights:
            response["insights"] = []
            response["status"] = "no_insights"
        else:
            response["insights"] = insights
            response["status"] = "found"

        resp.media = response


def parse_auth(req: falcon.Request) -> Optional[OFFAuthentication]:
    session_cookie = req.get_cookie_values("session")

    if session_cookie:
        session_cookie = session_cookie[0]

    authorization = req.get_header("Authorization")

    username = None
    password = None
    if authorization is not None:
        try:
            username, password = basic_decode(authorization)
        except BasicAuthDecodeError:
            raise falcon.HTTPUnauthorized(
                "Invalid authentication, Basic auth expected."
            )

    if not session_cookie and not username and not password:
        return None

    return OFFAuthentication(
        session_cookie=session_cookie, username=username, password=password
    )


def device_id_from_request(req: falcon.Request) -> str:
    """Returns the 'device_id' from the request parameters, or a hash of the
    access route (which should be the IPs of the proxies and the client)."""
    return req.get_param(
        "device_id",
        default=hashlib.sha1(str(req.access_route).encode()).hexdigest(),
    )


class AnnotateInsightResource:
    def on_post(self, req: falcon.Request, resp: falcon.Response):
        insight_id = req.get_param_as_uuid("insight_id", required=True)
        annotation = req.get_param_as_int(
            "annotation", required=True, min_value=-1, max_value=1
        )

        update = req.get_param_as_bool("update", default=True)
        # This field is only needed for nutritional table structure insights.
        data = req.get_param_as_json("data")

        auth = parse_auth(req)
        if auth is not None and auth.get_username() == "null":
            # Smoothie currently sends 'null' as username for anonymous voters
            auth = None

        trusted_annotator = auth is not None

        device_id = device_id_from_request(req)
        logger.info(
            "New annotation received from "
            f"{auth.get_username() if auth else 'unknown annotator'} "
            f"(annotation: {annotation}, insight: {insight_id})"
        )

        annotation_result = save_annotation(
            insight_id,
            annotation,
            update=update,
            data=data,
            auth=auth,
            device_id=device_id,
            trusted_annotator=trusted_annotator,
        )

        resp.media = {
            "status_code": annotation_result.status_code,
            "status": annotation_result.status,
            "description": annotation_result.description,
        }


class IngredientSpellcheckResource:
    def on_get(self, req: falcon.Request, resp: falcon.Response):
        self.spellcheck(req, resp)

    def on_post(self, req: falcon.Request, resp: falcon.Response):
        self.spellcheck(req, resp)

    def spellcheck(self, req: falcon.Request, resp: falcon.Response):
        text = req.get_param("text")
        server_type = get_server_type_from_req(req)
        if text is None:
            barcode = req.get_param("barcode")
            if barcode is None:
                raise falcon.HTTPBadRequest("text or barcode is required.")

            product = get_product(ProductIdentifier(barcode, server_type)) or {}
            text = product.get("ingredients_text_fr")
            if text is None:
                resp.media = {"status": "not_found"}
                return

        index_name = req.get_param("index", default=ElasticSearchIndex.product.name)
        confidence = req.get_param_as_float("confidence", default=0.5)
        pipeline = req.get_param_as_list("pipeline") or None
        safe = req.get_param_as_bool("safe", blank_as_true=False)

        if safe is not None and pipeline:
            raise falcon.HTTPBadRequest(
                "pipeline and safe parameters cannot be used together"
            )

        if pipeline:
            for item in pipeline:
                if item not in SPELLCHECKERS:
                    raise falcon.HTTPBadRequest(f"unknown pipeline item: {item}")
        elif safe:
            pipeline = ["patterns", "percentages", "vocabulary"]

        spellchecker = Spellchecker.load(
            client=es_client,
            pipeline=pipeline,
            index_name=index_name,
            confidence=confidence,
        )
        correction_item = spellchecker.correct(text)

        resp.media = {
            "text": text,
            "corrected": correction_item.latest_correction,
            "corrections": correction_item.corrections,
        }


class NutrientPredictorResource:
    def on_get(self, req: falcon.Request, resp: falcon.Response):
        ocr_url = req.get_param("ocr_url", required=True)
        server_type = get_server_type_from_req(req)

        if not ocr_url.endswith(".json"):
            raise falcon.HTTPBadRequest("a JSON file is expected")

        barcode = get_barcode_from_url(ocr_url)

        if barcode is None:
            raise falcon.HTTPBadRequest(f"invalid OCR URL: {ocr_url}")

        try:
            predictions = extract_ocr_predictions(
                ProductIdentifier(barcode, server_type),
                ocr_url,
                [PredictionType.nutrient],
            )

        except requests.exceptions.RequestException:
            resp.media = {
                "error": "download_error",
                "error_description": "an error occurred during OCR JSON download",
            }
            return

        except OCRParsingException as e:
            logger.error(e)
            resp.media = {
                "error": "invalid_ocr",
                "error_description": "an error occurred during OCR parsing",
            }
            return

        resp.media = {"nutrients": [p.to_dict() for p in predictions]}


class OCRInsightsPredictorResource:
    def on_get(self, req: falcon.Request, resp: falcon.Response):
        ocr_url = req.get_param("ocr_url", required=True)
        server_type = get_server_type_from_req(req)
        barcode = get_barcode_from_url(ocr_url)
        if barcode is None:
            raise falcon.HTTPBadRequest(f"invalid OCR URL: {ocr_url}")

        try:
            insights = extract_ocr_predictions(
                ProductIdentifier(barcode, server_type),
                ocr_url,
                DEFAULT_OCR_PREDICTION_TYPES,
            )

        except requests.exceptions.RequestException:
            resp.media = {
                "error": "download_error",
                "error_description": "an error occurred during OCR JSON download",
            }
            return

        except OCRParsingException as e:
            logger.error(e)
            resp.media = {
                "error": "invalid_ocr",
                "error_description": "an error occurred during OCR parsing",
            }
            return

        resp.media = {
            "insights": insights,
        }


class CategoryPredictorResource:
    @jsonschema.validate(schema.PREDICT_CATEGORY_SCHEMA)
    def on_post(self, req: falcon.Request, resp: falcon.Response):
        """Predict categories using neural categorizer and matching algorithm
        for a specific product."""
        server_type: ServerType = ServerType[req.media.get("server_type", "off")]

        if server_type != ServerType.off:
            raise falcon.HTTPBadRequest(
                f"category predictor is only available for 'off' server type (here: '{server_type.name}')"
            )

        predictors: list[str] = req.media.get("predictors") or ["neural", "matcher"]
        neural_model_name = None
        if (neural_model_name_str := req.media.get("neural_model_name")) is not None:
            neural_model_name = NeuralCategoryClassifierModel[neural_model_name_str]

        if "barcode" in req.media:
            # Fetch product from DB
            barcode: str = req.media["barcode"]
            product = get_product(ProductIdentifier(barcode, server_type)) or {}
            if not product:
                raise falcon.HTTPNotFound(description=f"product {barcode} not found")
            product_id = ProductIdentifier(barcode, server_type)
        else:
            product = req.media["product"]
            product_id = ProductIdentifier("NULL", server_type)
            if "matcher" in predictors:
                if "lang" not in req.media:
                    raise falcon.HTTPBadRequest(
                        description="lang field is required when using matcher predictor"
                    )
                lang = req.media["lang"]
                product[f"product_name_{lang}"] = product["product_name"]
                product["languages_codes"] = [lang]

        resp.media = predict_category(
            product,
            product_id,
            neural_predictor="neural" in predictors,
            matcher_predictor="matcher" in predictors,
            deepest_only=req.media.get("deepest_only", False),
            threshold=req.media.get("threshold"),
            neural_model_name=neural_model_name,
        )


class UpdateDatasetResource:
    def on_post(self, req: falcon.Request, resp: falcon.Response):
        """Re-import the Product Opener product dump."""

        enqueue_job(
            download_product_dataset_job, queue=low_queue, job_kwargs={"timeout": "1h"}
        )

        resp.media = {
            "status": "scheduled",
        }

    def on_get(self, req: falcon.Request, resp: falcon.Response):
        resp.media = {
            "etag": get_product_dataset_etag(),
        }


class ImageImporterResource:
    def on_post(self, req: falcon.Request, resp: falcon.Response):
        barcode = req.get_param("barcode", required=True)
        image_url = req.get_param("image_url", required=True)
        ocr_url = req.get_param("ocr_url", required=True)
        server_domain: str = req.get_param(
            "server_domain", default="api.openfoodfacts.org"
        )
        check_server_domain(server_domain)
        server_type = ServerType.get_from_server_domain(server_domain)

        enqueue_job(
            run_import_image_job,
            high_queue,
            job_kwargs={"result_ttl": 0},
            product_id=ProductIdentifier(barcode, server_type),
            image_url=image_url,
            ocr_url=ocr_url,
        )
        resp.media = {
            "status": "scheduled",
        }


class ImageCropResource:
    def on_get(self, req: falcon.Request, resp: falcon.Response):
        image_url = req.get_param("image_url", required=True)
        y_min = req.get_param_as_float("y_min", required=True)
        x_min = req.get_param_as_float("x_min", required=True)
        y_max = req.get_param_as_float("y_max", required=True)
        x_max = req.get_param_as_float("x_max", required=True)
        image = get_image_from_url(image_url, session=http_session, error_raise=False)

        if image is None:
            raise falcon.HTTPBadRequest(f"Could not fetch image: {image_url}")

        (left, right, top, bottom) = (
            x_min * image.width,
            x_max * image.width,
            y_min * image.height,
            y_max * image.height,
        )
        cropped_image = image.crop((left, top, right, bottom))
        image_response(cropped_image, resp)


class ImagePredictionImporterResource:
    @jsonschema.validate(schema.IMAGE_PREDICTION_IMPORTER_SCHEMA)
    def on_post(self, req: falcon.Request, resp: falcon.Response):
        timestamp = datetime.datetime.utcnow()
        inserts = []

        for prediction in req.media["predictions"]:
            source_image = generate_image_path(
                prediction["barcode"], prediction.pop("image_id")
            )
            inserts.append(
                {
                    "timestamp": timestamp,
                    "source_image": source_image,
                    **prediction,
                }
            )

        inserted = batch_insert(ImagePrediction, inserts)
        logger.info("{} image predictions inserted".format(inserted))


class ImagePredictionResource:
    def on_get(self, req: falcon.Request, resp: falcon.Response):
        count: int = req.get_param_as_int("count", min_value=1, default=25)
        page: int = req.get_param_as_int("page", min_value=1, default=1)
        with_logo: Optional[bool] = req.get_param_as_bool("with_logo", default=False)
        model_name: Optional[str] = req.get_param("model_name")
        type_: Optional[str] = req.get_param("type")
        model_version: Optional[str] = req.get_param("model_version")
        barcode: Optional[str] = req.get_param("barcode")
        min_confidence: Optional[float] = req.get_param_as_float("min_confidence")
        server_type = get_server_type_from_req(req)

        get_image_predictions_ = functools.partial(
            get_image_predictions,
            with_logo=with_logo,
            barcode=barcode,
            type=type_,
            server_type=server_type,
            model_name=model_name,
            model_version=model_version,
            min_confidence=min_confidence,
        )

        offset: int = (page - 1) * count
        image_predictions = [
            i.to_dict() for i in get_image_predictions_(limit=count, offset=offset)
        ]
        response: JSONType = {"count": get_image_predictions_(count=True)}

        if not image_predictions:
            response["image_predictions"] = []
            response["status"] = "no_image_predictions"
        else:
            response["image_predictions"] = image_predictions
            response["status"] = "found"

        resp.media = response


class ImagePredictorResource:
    def on_get(self, req: falcon.Request, resp: falcon.Response):
        image_url = req.get_param("image_url", required=True)
        models: list[str] = req.get_param_as_list("models", required=True)

        available_models = ObjectDetectionModelRegistry.get_available_models()

        for model_name in models:
            if model_name not in available_models:
                raise falcon.HTTPBadRequest(
                    "invalid_model",
                    "unknown model {}, available models: {}"
                    "".format(model_name, ", ".join(available_models)),
                )

        output_image = req.get_param_as_bool("output_image")

        if output_image is None:
            output_image = False

        if output_image and len(models) != 1:
            raise falcon.HTTPBadRequest(
                "invalid_request",
                "a single model must be specified with the `models` parameter "
                "when `output_image` is True",
            )

        image = get_image_from_url(image_url, session=http_session, error_raise=False)

        if image is None:
            raise falcon.HTTPBadRequest(f"Could not fetch image: {image_url}")

        predictions = {}

        for model_name in models:
            model = ObjectDetectionModelRegistry.get(model_name)
            result = model.detect_from_image(image, output_image=output_image)

            if output_image:
                image_response(result.boxed_image, resp)
                return
            else:
                predictions[model_name] = result.to_json()

        resp.media = {"predictions": predictions}


def image_response(image: Image.Image, resp: falcon.Response) -> None:
    resp.content_type = "image/jpeg"
    fp = io.BytesIO()
    image.save(fp, "JPEG")
    resp.stream_len = fp.tell()
    fp.seek(0)
    resp.stream = fp


class ImageLogoResource:
    def on_get(self, req: falcon.Request, resp: falcon.Response):
        logo_ids: list[str] = req.get_param_as_list(
            "logo_ids", transform=int, required=True
        )
        logos = []
        for logo in (
            LogoAnnotation.select()
            .join(ImagePrediction)
            .join(ImageModel)
            .where(LogoAnnotation.id.in_(logo_ids))
            .iterator()
        ):
            logo_dict = logo.to_dict()
            image_prediction = logo_dict.pop("image_prediction")
            logo_dict["image"] = image_prediction["image"]
            logos.append(logo_dict)

        resp.media = {"logos": logos, "count": len(logos)}


class ImageLogoSearchResource:
    def on_get(self, req: falcon.Request, resp: falcon.Response):
        server_type = get_server_type_from_req(req)
        count: int = req.get_param_as_int(
            "count", min_value=1, max_value=2000, default=25
        )
        type_: Optional[str] = req.get_param("type")
        barcode: Optional[str] = req.get_param("barcode")
        value: Optional[str] = req.get_param("value")
        taxonomy_value: Optional[str] = req.get_param("taxonomy_value")
        min_confidence: Optional[float] = req.get_param_as_float("min_confidence")
        random: bool = req.get_param_as_bool("random", default=False)
        annotated: Optional[bool] = req.get_param_as_bool("annotated")

        if type_ is None and (value is not None or taxonomy_value is not None):
            raise falcon.HTTPBadRequest(
                description="a type is required if `value` or `taxonomy_value` is provided"
            )

        if value is not None and taxonomy_value is not None:
            raise falcon.HTTPBadRequest(
                description="`value` and `taxonomy_value` are mutually exclusive parameters"
            )

        if type_ == "label" and taxonomy_value is None and value is not None:
            raise falcon.HTTPBadRequest(
                description="you should provide a `taxonomy_value` and not a `value` "
                "for label type"
            )

        where_clauses = [ImageModel.server_type == server_type.name]
        if annotated is not None:
            where_clauses.append(LogoAnnotation.annotation_value.is_null(not annotated))

        if min_confidence is not None:
            where_clauses.append(LogoAnnotation.score >= min_confidence)

        if barcode is not None:
            where_clauses.append(ImageModel.barcode == barcode)

        if type_ is not None:
            where_clauses.append(LogoAnnotation.annotation_type == type_)

        if value is not None:
            value_tag = get_tag(value)
            where_clauses.append(LogoAnnotation.annotation_value_tag == value_tag)

        if taxonomy_value is not None:
            where_clauses.append(LogoAnnotation.taxonomy_value == taxonomy_value)

        query = LogoAnnotation.select()
        query = query.join(ImagePrediction).join(ImageModel)

        if where_clauses:
            query = query.where(*where_clauses)

        query_count = query.count()

        if random:
            query = query.order_by(peewee.fn.Random())

        query = query.limit(count)
        items = [item.to_dict() for item in query.iterator()]

        for item in items:
            image_prediction = item.pop("image_prediction")
            item["image"] = image_prediction["image"]

        resp.media = {"logos": items, "count": query_count}


def check_logo_annotation(type_: str, value: Optional[str] = None):
    if value is not None:
        if type_ == "label" and not is_prefixed_value(value):
            raise falcon.HTTPBadRequest(
                description=f"language-prefixed value are required for label type (here: {value})"
            )
    elif type_ in ("brand", "category", "label", "store"):
        raise falcon.HTTPBadRequest(description=f"value required for type {type_})")


class ImageLogoDetailResource:
    def on_get(self, req: falcon.Request, resp: falcon.Response, logo_id: int):
        logo = LogoAnnotation.get_or_none(id=logo_id)

        if logo is None:
            resp.status = falcon.HTTP_404
            return

        logo_dict = logo.to_dict()
        image_prediction = logo_dict.pop("image_prediction")
        logo_dict["image"] = image_prediction["image"]
        resp.media = logo_dict

    @jsonschema.validate(schema.UPDATE_LOGO_SCHEMA)
    def on_put(self, req: falcon.Request, resp: falcon.Response, logo_id: int):
        auth = parse_auth(req)
        if auth is None:
            raise falcon.HTTPForbidden(
                description="authentication is required to annotate logos"
            )

        with db.atomic():
            logo: Optional[LogoAnnotation] = LogoAnnotation.get_or_none(id=logo_id)
            if logo is None:
                resp.status = falcon.HTTP_404
                return

            type_ = req.media["type"]
            value = req.media.get("value") or None
            check_logo_annotation(type_, value)

            if type_ != logo.annotation_type or value != logo.annotation_value:
                annotated_logos = update_logo_annotations(
                    [(type_, value, logo)],
                    username=auth.get_username() or "unknown",
                    completed_at=datetime.datetime.utcnow(),
                )
                server_type = ServerType[logo.image_prediction.image.server_type]
                generate_insights_from_annotated_logos(
                    annotated_logos, auth, server_type
                )

        resp.status = falcon.HTTP_204


class ImageLogoResetResource:
    def on_post(self, req: falcon.Request, resp: falcon.Response, logo_id: int):
        with db.atomic():
            logo = LogoAnnotation.get_or_none(id=logo_id)
            if logo is None:
                resp.status = falcon.HTTP_404
                return

            annotation_type = logo.annotation_type
            logo.annotation_value = None
            logo.annotation_value_tag = None
            logo.taxonomy_value = None
            logo.annotation_type = None
            logo.username = None
            logo.completed_at = None
            logo.save()

            if annotation_type in ("brand", "label"):
                prediction_deleted = (
                    Prediction.delete()
                    .where(
                        # Speed-up filtering by providing additional filters
                        Prediction.barcode == logo.barcode,
                        Prediction.type == annotation_type,
                        Prediction.predictor == "universal-logo-detector",
                        Prediction.data["logo_id"] == str(logo.id),
                    )
                    .execute()
                )
                insights_deleted = (
                    ProductInsight.delete()
                    .where(
                        ProductInsight.barcode == logo.barcode,
                        ProductInsight.type == annotation_type,
                        # never delete annotated insights
                        ProductInsight.annotation.is_null(),
                        ProductInsight.predictor == "universal-logo-detector",
                        ProductInsight.data["logo_id"] == str(logo_id),
                    )
                    .execute()
                )
                logger.info(
                    "prediction deleted: %s, insight deleted: %s",
                    prediction_deleted,
                    insights_deleted,
                )

        resp.status = falcon.HTTP_204


class ImageLogoAnnotateResource:
    @jsonschema.validate(schema.ANNOTATE_LOGO_SCHEMA)
    def on_post(self, req: falcon.Request, resp: falcon.Response):
        auth = parse_auth(req)
        if auth is None:
            raise falcon.HTTPForbidden(
                description="authentication is required to annotate logos"
            )
        server_type: ServerType = ServerType[req.media.get("server_type", "off")]
        annotations = req.media["annotations"]
        completed_at = datetime.datetime.utcnow()
        annotation_logos = []

        with db.atomic():
            for annotation in annotations:
                logo_id = annotation["logo_id"]
                type_ = annotation["type"]
                value = annotation["value"] or None
                check_logo_annotation(type_, value)

                try:
                    logo = LogoAnnotation.get_by_id(logo_id)
                except LogoAnnotation.DoesNotExist:
                    raise falcon.HTTPNotFound(description=f"logo {logo_id} not found")

                if logo.annotation_type is None:
                    # Don't annotate already annotated logos
                    annotation_logos.append((type_, value, logo))

            if annotation_logos:
                annotated_logos = update_logo_annotations(
                    annotation_logos,
                    username=auth.get_username() or "unknown",
                    completed_at=completed_at,
                )
            else:
                annotated_logos = []

        if annotated_logos:
            logo_ids = [logo.id for logo in annotated_logos]
            enqueue_job(
                generate_insights_from_annotated_logos_job,
                high_queue,
                {"result_ttl": 0, "timeout": "5m"},
                logo_ids=logo_ids,
                server_type=server_type,
                auth=auth,
            )
        resp.media = {"annotated": len(annotated_logos)}


class ImageLogoUpdateResource:
    def on_post(self, req: falcon.Request, resp: falcon.Response):
        """Bulk update logo annotations: change type and value of logos that have specific
        types and values.

        Because this endpoint mass-update annotations, leave it out of API documentation."""
        source_value = req.get_param("source_value", required=True)
        source_type = req.get_param("source_type", required=True)
        target_value = req.get_param("target_value", required=True)
        target_type = req.get_param("target_type", required=True)

        auth = parse_auth(req)
        username = None if auth is None else auth.get_username()
        completed_at = datetime.datetime.utcnow()

        target_value_tag = get_tag(target_value)
        source_value_tag = get_tag(source_value)
        taxonomy_value = match_taxonomized_value(target_value_tag, target_type)

        query = LogoAnnotation.update(
            {
                LogoAnnotation.annotation_type: target_type,
                LogoAnnotation.annotation_value: target_value,
                LogoAnnotation.annotation_value_tag: target_value_tag,
                LogoAnnotation.taxonomy_value: taxonomy_value,
                LogoAnnotation.username: username,
                LogoAnnotation.completed_at: completed_at,
            }
        ).where(
            LogoAnnotation.annotation_type == source_type,
            LogoAnnotation.annotation_value_tag == source_value_tag,
        )
        updated = query.execute()
        resp.media = {"updated": updated}


class ANNResource:
    def on_get(
        self, req: falcon.Request, resp: falcon.Response, logo_id: Optional[int] = None
    ):
        """Search for nearest neighbors of:
        - a random logo (if logo_id not provided)
        - a specific logo otherwise
        """
        count = req.get_param_as_int("count", min_value=1, max_value=500, default=100)
        server_type = get_server_type_from_req(req)

        if logo_id is None:
            logo_embeddings = list(
                LogoEmbedding.select().order_by(peewee.fn.Random()).limit(1)
            )

            if not logo_embeddings:
                resp.media = {"results": [], "count": 0, "query_logo_id": None}
                return

            logo_embedding = logo_embeddings[0]
            logo_id = logo_embedding.logo_id
        else:
            logo_embedding = LogoEmbedding.get_or_none(logo_id=logo_id)

            if logo_embedding is None:
                resp.status = falcon.HTTP_404
                return

        raw_results = [
            item
            for item in knn_search(
                es_client, logo_embedding.embedding, count, server_type=server_type
            )
            if item[0] != logo_id
        ][:count]
        results = [{"logo_id": item[0], "distance": item[1]} for item in raw_results]
        resp.media = {
            "results": results,
            "count": len(results),
            "query_logo_id": logo_id,
        }


SERVER_DOMAIN_REGEX = re.compile(
    r"api(\.pro)?\.open(food|beauty|product|petfood)facts\.(org|net)"
)


def check_server_domain(server_domain: str):
    if not SERVER_DOMAIN_REGEX.fullmatch(server_domain):
        raise falcon.HTTPBadRequest(f"invalid `server_domain`: {server_domain}")

    tld = server_domain.rsplit(".", maxsplit=1)[-1]
    instance_tld = settings._get_tld()
    if tld != instance_tld:
        raise falcon.HTTPBadRequest(
            f"invalid `server_domain`, expected '{instance_tld}' tld, got '{tld}'"
        )


class WebhookProductResource:
    """This handles requests from product opener
    that act as webhooks on product update or deletion.
    """

    def on_post(self, req: falcon.Request, resp: falcon.Response):
        barcode = req.get_param("barcode", required=True)
        action = req.get_param("action", required=True)
        server_domain: str = req.get_param("server_domain", required=True)
        check_server_domain(server_domain)
        logger.info(
            "New webhook event received for product %s (action: %s, domain: %s)",
            barcode,
            action,
            server_domain,
        )
        if action not in ("updated", "deleted"):
            raise falcon.HTTPBadRequest(
                title="invalid_action",
                description="action must be one of " "`deleted`, `updated`",
            )

        server_type = ServerType.get_from_server_domain(server_domain)
        product_id = ProductIdentifier(barcode, server_type)

        # Only add the update insight job to the queue for Open Food Facts,
        # as we don't have MongoDB connection for other projects yet
        if action == "updated" and server_type == ServerType.off:
            enqueue_in_job(
                update_insights_job,
                high_queue,
                settings.UPDATED_PRODUCT_WAIT,
                job_kwargs={"result_ttl": 0},
                product_id=product_id,
            )
        elif action == "deleted":
            enqueue_job(
                delete_product_insights_job,
                high_queue,
                job_kwargs={"result_ttl": 0},
                product_id=product_id,
            )

        resp.media = {
            "status": "scheduled",
        }


class ProductQuestionsResource:
    """Get questions about a product to confirm/infirm an insight

    see also doc/explanation/questions.md
    """

    def on_get(self, req: falcon.Request, resp: falcon.Response, barcode: str):
        response: JSONType = {}
        count: int = req.get_param_as_int("count", min_value=1, default=25)
        lang: str = req.get_param("lang", default="en")
        server_type = get_server_type_from_req(req)
        # If the device_id is not provided as a request parameter, we use the
        # hash of the IPs as a backup.
        device_id = device_id_from_request(req)

        auth: Optional[OFFAuthentication] = parse_auth(req)

        keep_types = QuestionFormatterFactory.get_default_types()

        insights = list(
            get_insights(
                barcode=barcode,
                server_type=server_type,
                keep_types=keep_types,
                limit=count,
                order_by="n_votes",
                avoid_voted_on=_get_skip_voted_on(auth, device_id),
                automatically_processable=False,
            )
        )

        if not insights:
            response["questions"] = []
            response["status"] = "no_questions"
        else:
            questions: list[JSONType] = []

            for insight in insights:
                formatter_cls = QuestionFormatterFactory.get(insight.type)
                formatter: QuestionFormatter = formatter_cls(TRANSLATION_STORE)
                question = formatter.format_question(insight, lang)
                questions.append(question.serialize())

            response["questions"] = questions
            response["status"] = "found"

        resp.media = response


class RandomQuestionsResource:
    def on_get(self, req: falcon.Request, resp: falcon.Response):
        get_questions_resource_on_get(req, resp, "random")


class PopularQuestionsResource:
    def on_get(self, req: falcon.Request, resp: falcon.Response):
        get_questions_resource_on_get(req, resp, "popularity")


class QuestionsCollectionResource:
    def on_get(self, req: falcon.Request, resp: falcon.Response):
        order_by = req.get_param("order_by", default="popularity")

        if order_by not in ("random", "popularity", "confidence"):
            raise falcon.HTTPBadRequest(
                description=f"invalid '{order_by}' value for `order_by` parameter"
            )
        get_questions_resource_on_get(req, resp, order_by)


def get_questions_resource_on_get(
    req: falcon.Request,
    resp: falcon.Response,
    order_by: Literal["random", "popularity", "confidence"],
):
    response: JSONType = {}
    page: int = req.get_param_as_int("page", min_value=1, default=1)
    count: int = req.get_param_as_int("count", min_value=1, default=25)
    lang: str = req.get_param("lang", default="en")
    keep_types: Optional[list[str]] = req.get_param_as_list(
        "insight_types", required=False
    )
    country: Optional[str] = req.get_param("country")
    value_tag: str = req.get_param("value_tag")
    brands = req.get_param_as_list("brands") or None
    reserved_barcode: Optional[bool] = req.get_param_as_bool(
        "reserved_barcode", default=False
    )
    server_type = get_server_type_from_req(req)
    # filter by annotation campaigns
    campaigns: Optional[list[str]] = req.get_param_as_list("campaigns") or None

    if campaigns is None:
        # `campaign` is a deprecated field, use campaigns now instead
        campaign: Optional[str] = req.get_param("campaign")
        campaigns = [campaign] if campaign is not None else None

    predictor = req.get_param("predictor")

    # If the device_id is not provided as a request parameter, we use the
    # hash of the IPs as a backup.
    device_id = device_id_from_request(req)

    auth: Optional[OFFAuthentication] = parse_auth(req)

    if reserved_barcode:
        # Include all results, including non reserved barcodes
        reserved_barcode = None

    if keep_types is None:
        keep_types = QuestionFormatterFactory.get_default_types()
    else:
        # Limit the number of types to prevent slow SQL queries
        keep_types = keep_types[:10]

    if brands is not None:
        # Limit the number of brands to prevent slow SQL queries
        brands = brands[:10]

    get_insights_ = functools.partial(
        get_insights,
        server_type=server_type,
        keep_types=keep_types,
        country=country,
        value_tag=value_tag,
        brands=brands,
        order_by=order_by,
        reserved_barcode=reserved_barcode,
        avoid_voted_on=_get_skip_voted_on(auth, device_id),
        automatically_processable=False,
        campaigns=campaigns,
        predictor=predictor,
    )

    offset: int = (page - 1) * count
    insights = list(get_insights_(limit=count, offset=offset))
    response["count"] = get_insights_(count=True)
    # This code should be merged with the one in ProductQuestionsResource.get
    if not insights:
        response["questions"] = []
        response["status"] = "no_questions"
    else:
        questions: list[JSONType] = []

        for insight in insights:
            formatter_cls = QuestionFormatterFactory.get(insight.type)

            if formatter_cls is None:
                continue

            formatter: QuestionFormatter = formatter_cls(TRANSLATION_STORE)
            question = formatter.format_question(insight, lang)
            questions.append(question.serialize())

        response["questions"] = questions
        response["status"] = "found"

    resp.media = response


class StatusResource:
    def on_get(self, req: falcon.Request, resp: falcon.Response):
        resp.media = {
            "status": "running",
        }


class HealthResource:
    def on_get(self, req: falcon.Request, resp: falcon.Response):
        from robotoff.health import health

        message, status, headers = health.run()
        resp.media = {
            "message": orjson.loads(message),
            "status": status,
            "headers": headers,
        }
        resp.status = str(status)


class DumpResource:
    def on_get(self, req: falcon.Request, resp: falcon.Response):
        keep_types: list[str] = req.get_param_as_list(
            "insight_types", required=False, default=[]
        )[:10]
        barcode = req.get_param("barcode")
        annotated = req.get_param_as_bool("annotated", blank_as_true=False)
        value_tag = req.get_param("value_tag")
        count = req.get_param_as_int("count", min_value=0, max_value=10_000)
        server_type = get_server_type_from_req(req)

        get_insights_ = functools.partial(
            get_insights,
            server_type=server_type,
            barcode=barcode,
            keep_types=keep_types,
            annotated=annotated,
            value_tag=value_tag,
        )
        insight_count: int = get_insights_(count=True)  # type: ignore
        if insight_count > 10_000 and count is None:
            raise falcon.HTTPBadRequest(
                description=f"more than 10 000 insights matching criteria (here: {insight_count}), "
                "use more specific criteria or use count parameter"
            )

        insights_iter = get_insights_(limit=count, as_dict=True)
        writer = None

        with tempfile.TemporaryFile("w+", newline="") as temp_f:
            for insight_dict in insights_iter:
                serial = orjson.loads(orjson.dumps(insight_dict))

                if writer is None:
                    writer = csv.DictWriter(temp_f, fieldnames=serial.keys())
                    writer.writeheader()

                writer.writerow(serial)

            temp_f.seek(0)
            content = temp_f.read()

        if content:
            resp.content_type = "text/csv"
            resp.body = content
        else:
            resp.status = falcon.HTTP_204


class UserStatisticsResource:
    def on_get(self, req: falcon.Request, resp: falcon.Response, username: str):
        annotation_count = (
            ProductInsight.select().where(ProductInsight.username == username).count()
        )
        resp.media = {"count": {"annotations": annotation_count}}


class ImageCollection:
    def on_get(self, req: falcon.Request, resp: falcon.Response):
        response: JSONType = {}
        count: int = req.get_param_as_int("count", min_value=1, default=25)
        page: int = req.get_param_as_int("page", min_value=1, default=1)
        with_predictions: Optional[bool] = req.get_param_as_bool(
            "with_predictions", default=False
        )
        barcode: Optional[str] = req.get_param("barcode")
        server_type = get_server_type_from_req(req)

        get_images_ = functools.partial(
            get_images,
            with_predictions=with_predictions,
            barcode=barcode,
            server_type=server_type,
        )

        offset: int = (page - 1) * count
        images = [i.to_dict() for i in get_images_(limit=count, offset=offset)]
        response["count"] = get_images_(count=True)

        if not images:
            response["images"] = []
            response["status"] = "no_images"
        else:
            response["images"] = images
            response["status"] = "found"

        resp.media = response


class PredictionCollection:
    def on_get(self, req: falcon.Request, resp: falcon.Response):
        page: int = req.get_param_as_int("page", min_value=1, default=1)
        count: int = req.get_param_as_int("count", min_value=1, default=25)
        barcode: Optional[str] = req.get_param("barcode")
        value_tag: str = req.get_param("value_tag")
        keep_types: Optional[list[str]] = req.get_param_as_list("types", required=False)
        server_type = get_server_type_from_req(req)

        if keep_types:
            # Limit the number of types to prevent slow SQL queries
            keep_types = keep_types[:10]

        get_predictions_ = functools.partial(
            get_predictions,
            keep_types=keep_types,
            value_tag=value_tag,
            barcode=barcode,
            server_type=server_type,
        )

        offset: int = (page - 1) * count
        predictions = [
            i.to_dict() for i in get_predictions_(limit=count, offset=offset)
        ]

        response: JSONType = {"count": get_predictions_(count=True)}

        if not predictions:
            response["predictions"] = []
            response["status"] = "no_predictions"
        else:
            response["predictions"] = list(predictions)
            response["status"] = "found"

        resp.media = response


class UnansweredQuestionCollection:
    def on_get(self, req: falcon.Request, resp: falcon.Response):
        response: JSONType = {}
        page: int = req.get_param_as_int("page", min_value=1, default=1)
        count: int = req.get_param_as_int("count", min_value=1, default=25)
        insight_type: str = req.get_param("type")
        country: Optional[str] = req.get_param("country")
        reserved_barcode: Optional[bool] = req.get_param_as_bool(
            "reserved_barcode", default=False
        )
        server_type = get_server_type_from_req(req)
        # filter by annotation campaigns
        campaigns: Optional[list[str]] = req.get_param_as_list("campaigns") or None
        if campaigns is None:
            # `campaign` is a deprecated field, use campaigns now instead
            campaign: Optional[str] = req.get_param("campaign")
            campaigns = [campaign] if campaign is not None else None

        predictor = req.get_param("predictor")

        get_insights_ = functools.partial(
            get_insights,
            server_type=server_type,
            keep_types=[insight_type] if insight_type else None,
            group_by_value_tag=True,
            limit=count,
            country=country,
            automatically_processable=False,
            reserved_barcode=reserved_barcode,
            campaigns=campaigns,
            predictor=predictor,
        )

        offset: int = (page - 1) * count
        insights = [i for i in get_insights_(limit=count, offset=offset)]

        response["count"] = get_insights_(count=True)

        if not insights:
            response["questions"] = []
            response["status"] = "no_questions"
        else:
            response["questions"] = insights
            response["status"] = "found"

        resp.media = response


class ImagePredictionCollection:
    def on_get(self, req: falcon.Request, resp: falcon.Response):
        response: JSONType = {}
        count: int = req.get_param_as_int("count", min_value=1, default=25)
        page: int = req.get_param_as_int("page", min_value=1, default=1)
        with_logo: Optional[bool] = req.get_param_as_bool("with_logo", default=False)
        barcode: Optional[str] = req.get_param("barcode")
        type: Optional[str] = req.get_param("type")
        server_type = get_server_type_from_req(req)

        query_parameters = {
            "with_logo": with_logo,
            "barcode": barcode,
            "type": type,
            "server_type": server_type,
        }

        get_image_predictions_ = functools.partial(
            get_image_predictions, **query_parameters
        )

        offset: int = (page - 1) * count
        image_predictions = [
            i.to_dict() for i in get_image_predictions_(limit=count, offset=offset)
        ]
        response["count"] = get_image_predictions_(count=True)

        if not image_predictions:
            response["image_predictions"] = []
            response["status"] = "no_image_predictions"
        else:
            response["image_predictions"] = image_predictions
            response["status"] = "found"

        resp.media = response


class LogoAnnotationCollection:
    def on_get(self, req: falcon.Request, resp: falcon.Response):
        response: JSONType = {}
        barcode: Optional[str] = req.get_param("barcode")
        server_type = get_server_type_from_req(req)
        keep_types: Optional[list[str]] = req.get_param_as_list("types", required=False)
        value_tag: str = req.get_param("value_tag")
        page: int = req.get_param_as_int("page", min_value=1, default=1)
        count: int = req.get_param_as_int("count", min_value=1, default=25)

        if keep_types:
            # Limit the number of types to prevent slow SQL queries
            keep_types = keep_types[:10]

        query_parameters = {
            "barcode": barcode,
            "keep_types": keep_types,
            "value_tag": value_tag,
            "server_type": server_type,
        }

        get_annotation_ = functools.partial(get_logo_annotation, **query_parameters)

        offset: int = (page - 1) * count
        annotation = [i.to_dict() for i in get_annotation_(limit=count, offset=offset)]
        response["count"] = get_annotation_(count=True)

        if not annotation:
            response["annotation"] = []
            response["status"] = "no_annotation"
        else:
            response["annotation"] = annotation
            response["status"] = "found"

        resp.media = response


cors = CORS(
    allow_all_origins=True,
    allow_all_headers=True,
    allow_all_methods=True,
    allow_credentials_all_origins=True,
    max_age=600,
)

api = falcon.API(
    middleware=[cors.middleware, MultipartMiddleware(), DBConnectionMiddleware()]
)

json_handler = falcon.media.JSONHandler(dumps=orjson.dumps, loads=orjson.loads)
extra_handlers = {
    "application/json": json_handler,
}

api.resp_options.media_handlers.update(extra_handlers)

# Parse form parameters
api.req_options.auto_parse_form_urlencoded = True
api.req_options.strip_url_path_trailing_slash = True
api.req_options.auto_parse_qs_csv = True
# defines urls
api.add_route("/api/v1/insights/{barcode}", ProductInsightResource())
api.add_route("/api/v1/insights/detail/{insight_id:uuid}", ProductInsightDetail())
api.add_route("/api/v1/insights", InsightCollection())
api.add_route("/api/v1/insights/random", RandomInsightResource())
api.add_route("/api/v1/insights/annotate", AnnotateInsightResource())
api.add_route("/api/v1/insights/dump", DumpResource())
api.add_route("/api/v1/predict/ingredients/spellcheck", IngredientSpellcheckResource())
api.add_route("/api/v1/predict/nutrient", NutrientPredictorResource())
api.add_route("/api/v1/predict/ocr_insights", OCRInsightsPredictorResource())
api.add_route("/api/v1/predict/category", CategoryPredictorResource())
api.add_route("/api/v1/products/dataset", UpdateDatasetResource())
api.add_route("/api/v1/webhook/product", WebhookProductResource())
api.add_route("/api/v1/images", ImageCollection())
api.add_route("/api/v1/images/import", ImageImporterResource())
api.add_route("/api/v1/images/crop", ImageCropResource())
api.add_route("/api/v1/image_predictions", ImagePredictionResource())
api.add_route("/api/v1/image_predictions/import", ImagePredictionImporterResource())
api.add_route("/api/v1/images/predict", ImagePredictorResource())
api.add_route("/api/v1/images/logos", ImageLogoResource())
api.add_route("/api/v1/images/logos/search", ImageLogoSearchResource())
api.add_route("/api/v1/images/logos/{logo_id:int}", ImageLogoDetailResource())
api.add_route("/api/v1/images/logos/{logo_id:int}/reset", ImageLogoResetResource())
api.add_route("/api/v1/images/logos/annotate", ImageLogoAnnotateResource())
api.add_route("/api/v1/images/logos/update", ImageLogoUpdateResource())
api.add_route("/api/v1/ann/{logo_id:int}", ANNResource())
api.add_route("/api/v1/ann", ANNResource())
api.add_route("/api/v1/ann/search/{logo_id:int}", ANNResource())
api.add_route("/api/v1/ann/search", ANNResource())
api.add_route("/api/v1/questions/{barcode}", ProductQuestionsResource())
api.add_route("/api/v1/questions", QuestionsCollectionResource())
api.add_route("/api/v1/questions/random", RandomQuestionsResource())
api.add_route("/api/v1/questions/popular", PopularQuestionsResource())
api.add_route("/api/v1/questions/unanswered", UnansweredQuestionCollection())
api.add_route("/api/v1/status", StatusResource())
api.add_route("/api/v1/health", HealthResource())
api.add_route("/api/v1/users/statistics/{username}", UserStatisticsResource())
api.add_route("/api/v1/predictions", PredictionCollection())
api.add_route("/api/v1/annotation/collection", LogoAnnotationCollection())
