"""
This module is present to handle everything related to input data that are not structural.

In the Grid2Op vocabulary a "GridValue" or "Chronics" is something that provides data to change the input parameter
of a power flow between 1 time step and the other.

It is a more generic terminology. Modification that can be performed by :class:`GridValue` object includes, but
are not limited to:

  - injections such as:

    - productions active production setpoint
    - generators voltage setpoint
    - loads active consumption
    - loads reactive consumption

  - structural informations such as:

    - planned outage: powerline disconnection anticipated in advance
    - hazards: powerline disconnection that cannot be anticipated, for example due to a windstorm.

All powergrid modification that can be performed using an :class:`grid2op.Action` can be implemented as form of a
:class:`GridValue`.

The same mechanism than for :class:`grid2op.Action` or :class:`grid2op.Observation` is pursued here. All states
modifications made by the :class:`grid2op.Environment` must derived from the :class:`GridValue`. It is not
recommended to instanciate them directly, but rather to use the :class:`ChronicsHandler` for such a purpose.

Note that the values returned by a :class:`GridValue` are **backend dependant**. A GridValue object should always
return the data in the order expected by the :class:`grid2op.Backend`, regardless of the order in which data are given
in the files or generated by the data generator process.

This implies that changing the backend will change the output of :class:`GridValue`. More information about this
is given in the description of the :func:`GridValue.initialize` method.

Finally, compared to other Reinforcement Learning problems, is the possibility to use "forecast". This optional feature
can be accessed via the :class:`grid2op.Observation` and mainly the :func:`grid2op.Observation.simulate` method. The
data that are used to generate this forecasts come from the :class:`grid2op.GridValue` and are detailed in the
:func:`GridValue.forecasts` method.

"""
import os
import copy

import numpy as np
import pandas as pd

import warnings

from datetime import datetime, timedelta

try:
    from .Exceptions import *
except (ModuleNotFoundError, ImportError):
    from Exceptions import *

from abc import ABC, abstractmethod

import pdb


# TODO sous echantillonner ou sur echantilloner les scenario
# TODO max_iter is not properly handled in the example of GridValue now
# TODO add a class to sample "online" the data.
# TODO this is weird that maintenance and hazards are in the action,
# but not maintenance_time and maintenance_duration...
# TODO in FromFile, add the possibility to change the timestamps (for now it's not modified at all...)


class GridValue(ABC):
    """
    This is the base class for every kind of data for the _grid.

    It allows the :class:`grid2op.Environment` to perform powergrid modification that make the "game" time dependant.

    It is not recommended to directly create :class:`GridValue` object, but to use the
    :attr:`grid2op.Environment.chronics_handler" for such a purpose. This is made in an attempt to make sure the
    :func:`GridValue.initialize` is called. Before this initialization, it is not recommended to use any
    :class:`GridValue` object.

    The method :func:`GridValue.next_chronics` should be used between two epoch of the game. If there are no more
    data to be generated from this object, then :func:`GridValue.load_next` should raise a :class:`StopIteration`
    exception and a call to :func:`GridValue.done` should return True.

    Attributes
    ----------
    time_interval: :class:`.datetime.timedelta`
        Time interval between 2 consecutive timestamps. Default 5 minutes.

    current_datetime: :class:`datetime.datetime`
        The timestamp of the current observation.

    n_gen: ``int``
        Number of generators in the powergrid

    n_load: ``int``
        Number of loads in the powergrid

    n_lines: ``int``
        Number of powerline in the powergrid

    max_iter: ``int``
        Number maximum of data to generate for one episode.

    curr_iter: ``int``
        Duration of the current episode.

    maintenance_time: ``numpy.ndarray``, dtype:``int``
        Number of time steps the next maintenance will take place with the following convention:

            - -1 no maintenance are planned for the forseeable future
            - 0 a maintenance is taking place
            - 1, 2, 3 ... a maintenance will take place in 1, 2, 3, ... time step

        Some examples are given in :func:`GridValue.maintenance_time_1d`.

    maintenance_duration: ``numpy.ndarray``, dtype:``int``
        Duration of the next maintenance. 0 means no maintenance is happening. If a maintenance is planned for a
        given powerline, this number decreases each time step, up until arriving at 0 when the maintenance is over. Note
        that if a maintenance is planned (see :attr:`GridValue.maintenance_time`) this number indicates how long
        the maintenance will last, and does not suppose anything on the maintenance taking place or not (= there can be
        positive number here without a powerline being removed from the grid for maintenance reason). Some examples are
        given in :func:`GridValue.maintenance_duration_1d`.

    hazard_duration: ``numpy.ndarray``, dtype:``int``
        Duration of the next hzard. 0 means no maintenance is happening. If a hazard is taking place for a
        given powerline, this number decreases each time step, up until arriving at 0 when the maintenance is over. On
        the contrary to :attr:`GridValue.maintenance_duration`, if a component of this vector is higher than 1, it
        means that the powerline is out of service. Some examples are
        given in :func:`GridValue.get_hazard_duration_1d`.


    """
    def __init__(self, time_interval=timedelta(minutes=5), max_iter=-1):
        # TODO complete that with a real datetime
        self.time_interval = time_interval
        self.current_datetime = datetime(year=2019, month=1, day=1)
        self.n_gen = None
        self.n_load = None
        self.n_lines = None
        self.max_iter = max_iter
        self.curr_iter = 0

        self.maintenance_time = None
        self.maintenance_duration = None
        self.hazard_duration = None

    @abstractmethod
    def initialize(self, order_backend_loads, order_backend_prods, order_backend_lines, order_backend_subs,
                   names_chronics_to_backend):
        """
        This function is used to initialize the data generator.
        It can be use to load scenarios, or to initialize noise if scenarios are generated on the fly. It must also
        initialize :attr:`GridValue.maintenance_time`, :attr:`GridValue.maintenance_duration` and
        :attr:`GridValue.hazard_duration`.

        This function should also increment :attr:`GridValue.curr_iter` of 1 each time it is called.

        The :class:`GridValue` is what makes the connection between the data (generally in a shape of files on the
        hard drive) and the power grid. One of the main advantage of the Grid2Op package is its ability to change
        the tool that computes the load flows. Generally, such :class:`grid2op.Backend` expects data in a specific
        format that is given by the way their internal powergrid is represented, and in particular, the "same"
        objects can have different name and different position. To ensure that the same chronics would
        produce the same results on every backend (**ie** regardless of the order of which the Backend is expecting
        the data, the outcome of the powerflow is the same) we encourage the user to provide a file that maps the name
        of the object in the chronics to the name of the same object in the backend.

        This is done with the "names_chronics_to_backend" dictionnary that has the following keys:

          - "loads"
          - "prods"
          - "lines"

        The value associated to each of these keys is in turn a mapping dictionnary from the chronics to the backend.
        This means that each *keys* of these subdictionnary is a name of one column in the files, and each values
        is the corresponding name of this same object in the dictionnary. An example is provided bellow.

        Parameters
        ----------
        order_backend_loads: ``numpy.ndarray``, dtype:str
            Ordered name, in the Backend, of the loads. It is required that a :class:`grid2op.Backend` object always
            output the informations in the same order. This array gives the name of the loads following this order.
            See the documentation of :mod:`grid2op.Backend` for more information about this.

        order_backend_prods: ``numpy.ndarray``, dtype:str
            Same as order_backend_loads, but for generators.

        order_backend_lines: ``numpy.ndarray``, dtype:str
            Same as order_backend_loads, but for powerline.

        order_backend_subs: ``numpy.ndarray``, dtype:str
            Same as order_backend_loads, but for powerline.

        names_chronics_to_backend: ``dict``
            See in the description of the method for more information about its format.

        Returns
        -------
        ``None``

        Examples
        --------
        For example, suppose we have a :class:`grid2op.Backend` with:

          - substations ids strart from 0 to N-1 (N being the number of substations in the powergrid)
          - loads named "load_i" with "i" the subtations to which it is connected
          - generators units named "gen_i" (i still the substation id to which it is connected)
          - powerlnes are named "i_j" if it connected substations i to substation j

        And on the other side, we have some files with the following conventions:

          - substations are numbered from 1 to N
          - loads are named "i_C" with i being the substation to which it is connected
          - generators are named "i_G" with is being the id of the substations to which it is connected
          - powerlines are namesd "i_j_k" where i is the origin substation, j the extremity substations and "k"
            is a unique identifier of this powerline in the powergrid.

        In this case, instead of renaming the powergrid (in the backend) of the data files, it is advised to build the
        following elements and initialize the object gridval of type :class:`GridValue` with:

        >>> gridval = GridValue()  # Note: this code won't execute because "GridValue" is an abstract class
        >>> order_backend_loads = ['load_1', 'load_2', 'load_13', 'load_3', 'load_4', 'load_5', 'load_8', 'load_9',
        >>>                         'load_10', 'load_11', 'load_12']
        >>> order_backend_prods = ['gen_1', 'gen_2', 'gen_5', 'gen_7', 'gen_0']
        >>> order_backend_lines = ['0_1', '0_4', '8_9', '8_13', '9_10', '11_12', '12_13', '1_2', '1_3', '1_4', '2_3',
        >>>                            '3_4', '5_10', '5_11', '5_12', '3_6', '3_8', '4_5', '6_7', '6_8']
        >>> order_backend_subs = ['sub_0', 'sub_1', 'sub_10', 'sub_11', 'sub_12', 'sub_13', 'sub_2', 'sub_3', 'sub_4',
        >>>                           'sub_5', 'sub_6', 'sub_7', 'sub_8', 'sub_9']
        >>> names_chronics_to_backend = {"loads": {"2_C": 'load_1', "3_C": 'load_2',
        >>>                                            "14": 'load_13', "4_C": 'load_3', "5_C": 'load_4',
        >>>                                            "6_C": 'load_5', "9_C": 'load_8', "10_C": 'load_9',
        >>>                                            "11_C": 'load_10', "12_C": 'load_11',
        >>>                                            "13_C": 'load_12'},
        >>>                                  "lines": {'1_2_1': '0_1', '1_5_2': '0_4', '9_10_16': '8_9', '9_14_17': '8_13',
        >>>                                            '10_11_18': '9_10', '12_13_19': '11_12', '13_14_20': '12_13',
        >>>                                            '2_3_3': '1_2', '2_4_4': '1_3', '2_5_5': '1_4', '3_4_6': '2_3',
        >>>                                            '4_5_7': '3_4', '6_11_11': '5_10', '6_12_12': '5_11',
        >>>                                            '6_13_13': '5_12', '4_7_8': '3_6', '4_9_9': '3_8', '5_6_10': '4_5',
        >>>                                           '7_8_14': '6_7', '7_9_15': '6_8'},
        >>>                                  "prods": {"1_G": 'gen_0', "3_G": "gen_2", "6_G": "gen_5",
        >>>                                            "2_G": "gen_1", "8_G": "gen_7"},
        >>>                                 }
        >>> gridval.initialize(order_backend_loads, order_backend_prods, order_backend_lines, names_chronics_to_backend)

        """
        pass

    @staticmethod
    def get_maintenance_time_1d(maintenance):
        """
        This function allows to transform a 1d numpy aarray maintenance, where is specify:

            - 0 there is no maintenance at this time step
            - 1 there is a maintenance at this time step

        Into the representation in terms of "next maintenance time" as specified in
        :attr:`GridValue.maintenance_time` which is:

            - `-1` no foreseeable maintenance operation will be performed
            - `0` a maintenance operation is being performed
            - `1`, `2` etc. is the number of time step the next maintenance will be performed.

        Parameters
        ----------
        maintenance: ``numpy.ndarray``
            1 dimensional array representing the time series of the maintenance (0 there is no maintenance, 1 there
            is a maintenance at this time step)

        Returns
        -------
        maintenance_duration: ``numpy.ndarray``
            Array representing the time series of the duration of the next maintenance forseeable.

        Examples
        --------

        If no maintenance are planned:

        >>> maintenance_time = GridValue.get_maintenance_time_1d(np.array([0 for _ in range(10)]))
        >>> assert np.all(maintenance_time == np.array([-1  for _ in range(10)]))

        If a maintenance planned of 3 time steps starting at timestep 6 (index 5 - index starts at 0)

        >>> maintenance = np.array([0,0,0,0,0,1,1,1,0,0,0,0,0,0,0,0])
        >>> maintenance_time = GridValue.get_maintenance_time_1d(maintenance)
        >>> assert np.all(maintenance_time == np.array([5,4,3,2,1,0,0,0,-1,-1,-1,-1,-1,-1,-1,-1]))

        If a maintenance planned of 3 time steps starting at timestep 6
        (index 5 - index starts at 0), and a second one for 2 time steps at time step 13

        >>> maintenance = np.array([0,0,0,0,0,1,1,1,0,0,0,0,1,1,0,0,0])
        >>> maintenance_time = GridValue.get_maintenance_time_1d(maintenance)
        >>> assert np.all(maintenance_time == np.array([5,4,3,2,1,0,0,0,4,3,2,1,0,0,-1,-1,-1]))

        """

        res = np.full(maintenance.shape, fill_value=np.NaN, dtype=np.int)
        maintenance = np.concatenate((maintenance, (0, 0)))
        a = np.diff(maintenance)
        # +1 is because numpy does the diff `t+1` - `t` so to get index of the initial array
        # I need to "+1"
        start = np.where(a == 1)[0] + 1  # start of maintenance
        end = np.where(a == -1)[0] + 1  # end of maintenance
        prev_ = 0
        # it's efficient here as i do a loop only on the number of time there is a maintenance
        # and maintenance are quite rare
        for beg_, end_ in zip(start, end):
            res[prev_:beg_] = list(range(beg_ - prev_, 0, -1))
            res[beg_:end_] = 0
            prev_ = end_

        # no maintenance are planned in the forseeable future
        res[prev_:] = -1
        return res

    @staticmethod
    def get_maintenance_duration_1d(maintenance):
        """
        This function allows to transform a 1d numpy aarray maintenance (or hazards), where is specify:

            - 0 there is no maintenance at this time step
            - 1 there is a maintenance at this time step

        Into the representation in terms of "next maintenance duration" as specified in
        :attr:`GridValue.maintenance_duration` which is:

            - `0` no forseeable maintenance operation will be performed
            - `1`, `2` etc. is the number of time step the next maintenance will last (it can be positive even in the
                case that no maintenance is currently being performed.

        Parameters
        ----------
        maintenance: ``numpy.ndarray``
            1 dimensional array representing the time series of the maintenance (0 there is no maintenance, 1 there
            is a maintenance at this time step)

        Returns
        -------
        maintenance_duration: ``numpy.ndarray``
            Array representing the time series of the duration of the next maintenance forseeable.

        Examples
        --------

        If no maintenance are planned:

        >>> maintenance = np.array([0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0])
        >>> maintenance_duration = GridValue.get_maintenance_duration_1d(maintenance)
        >>> assert np.all(maintenance_duration == np.array([0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0]))

        If a maintenance planned of 3 time steps starting at timestep 6 (index 5 - index starts at 0)

        >>> maintenance = np.array([0,0,0,0,0,1,1,1,0,0,0,0,0,0,0,0])
        >>> maintenance_duration = GridValue.get_maintenance_duration_1d(maintenance)
        >>> assert np.all(maintenance_duration == np.array([3,3,3,3,3,3,2,1,0,0,0,0,0,0,0,0]))

        If a maintenance planned of 3 time steps starting at timestep 6
        (index 5 - index starts at 0), and a second one for 2 time steps at time step 13

        >>> maintenance = np.array([0,0,0,0,0,1,1,1,0,0,0,0,1,1,0,0,0])
        >>> maintenance_duration = GridValue.get_maintenance_duration_1d(maintenance)
        >>> assert np.all(maintenance_duration == np.array([3,3,3,3,3,3,2,1,2,2,2,2,2,1,0,0,0]))

        """

        res = np.full(maintenance.shape, fill_value=np.NaN, dtype=np.int)
        maintenance = np.concatenate((maintenance, (0,0)))
        a = np.diff(maintenance)
        # +1 is because numpy does the diff `t+1` - `t` so to get index of the initial array
        # I need to "+1"
        start = np.where(a == 1)[0] + 1  # start of maintenance
        end = np.where(a == -1)[0] + 1  # end of maintenance
        prev_ = 0
        # it's efficient here as i do a loop only on the number of time there is a maintenance
        # and maintenance are quite rare
        for beg_, end_ in zip(start, end):
            res[prev_:beg_] = end_ - beg_
            res[beg_:end_] = list(range(end_ - beg_, 0, -1))
            prev_ = end_

        # no maintenance are planned in the foreseeable future
        res[prev_:] = 0
        return res

    @staticmethod
    def get_hazard_duration_1d(hazard):
        """
        This function allows to transform a 1d numpy aarray maintenance (or hazards), where is specify:

            - 0 there is no maintenance at this time step
            - 1 there is a maintenance at this time step

        Into the representation in terms of "hzard duration" as specified in
        :attr:`GridValue.maintenance_duration` which is:

            - `0` no forseeable hazard operation will be performed
            - `1`, `2` etc. is the number of time step the next hzard will last (it is positive only when a hazard
                affect a given powerline)

        Compared to :func:`GridValue.get_maintenance_duration_1d` we only know when the hazard occurs how long it
        will last.

        Parameters
        ----------
        maintenance: ``numpy.ndarray``
            1 dimensional array representing the time series of the maintenance (0 there is no maintenance, 1 there
            is a maintenance at this time step)

        Returns
        -------
        maintenance_duration: ``numpy.ndarray``
            Array representing the time series of the duration of the next maintenance forseeable.

        Examples
        --------

        If no maintenance are planned:

        >>> hazard = np.array([0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0])
        >>> hazard_duration = GridValue.get_hazard_duration_1d(hazard)
        >>> assert np.all(hazard_duration == np.array([0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0]))

        If a maintenance planned of 3 time steps starting at timestep 6 (index 5 - index starts at 0)

        >>> hazard = np.array([0,0,0,0,0,1,1,1,0,0,0,0,0,0,0,0])
        >>> hazard_duration = GridValue.get_hazard_duration_1d(hazard)
        >>> assert np.all(hazard_duration == np.array([0,0,0,0,0,3,2,1,0,0,0,0,0,0,0,0]))

        If a maintenance planned of 3 time steps starting at timestep 6
        (index 5 - index starts at 0), and a second one for 2 time steps at time step 13

        >>> hazard = np.array([0,0,0,0,0,1,1,1,0,0,0,0,1,1,0,0,0])
        >>> hazard_duration = GridValue.get_hazard_duration_1d(hazard)
        >>> assert np.all(hazard_duration == np.array([0,0,0,0,0,3,2,1,0,0,0,0,2,1,0,0,0]))

        """

        res = np.full(hazard.shape, fill_value=np.NaN, dtype=np.int)
        hazard = np.concatenate((hazard, (0, 0)))
        a = np.diff(hazard)
        # +1 is because numpy does the diff `t+1` - `t` so to get index of the initial array
        # I need to "+1"
        start = np.where(a == 1)[0] + 1  # start of maintenance
        end = np.where(a == -1)[0] + 1  # end of maintenance
        prev_ = 0
        # it's efficient here as i do a loop only on the number of time there is a maintenance
        # and maintenance are quite rare
        for beg_, end_ in zip(start, end):
            res[prev_:beg_] = 0
            res[(beg_):(end_)] = list(range(end_ - beg_, 0, -1))
            prev_ = end_

        # no maintenance are planned in the forseeable future
        res[prev_:] = 0
        return res

    @abstractmethod
    def load_next(self):
        """
        Generate the next values, either by reading from a file, or by generating on the fly and return a dictionnary
        compatible with the :class:`grid2op.Action` class allowed for the :class:`Environment`.

        More information about this dictionnary can be found at :func:`grid2op.Action.update`.

        As a (quick) reminder: this dictionnary has for keys:

          - "injection" (optional): a dictionnary with keys (optional) "load_p", "load_q", "prod_p", "prod_v"
          - "hazards" (optional) : the outage suffered from the _grid
          - "maintenance" (optional) : the maintenance operations planned on the grid for the current time step.

        Returns
        -------
        timestamp: ``datetime.datetime``
            The current timestamp for which the modifications have been generated.

        dict_: ``dict``
            A dictionnary understandable by the ::func:`grid2op.Action.update` method. **NB** this function should
            return the dictionnary that will be converted, is should not, in any case, return an action.

        maintenance_time: ``numpy.ndarray``, dtype:``int``
            Information about the next planned maintenance. See :attr:`GridValue.maintenance_time` for more information.

        maintenance_duration: ``numpy.ndarray``, dtype:``int``
            Information about the duration of next planned maintenance. See :attr:`GridValue.maintenance_duration`
            for more information.

        hazard_duration: ``numpy.ndarray``, dtype:``int``
            Information about the current hazard. See :attr:`GridValue.hazard_duration`
            for more information.

        Raises
        ------
        StopIteration
            if the chronics is over
        """
        self.current_datetime += self.time_interval
        return self.current_datetime, {}, self.maintenance_time, self.maintenance_duration, self.hazard_duration

    @abstractmethod
    def check_validity(self, backend):
        """
        To make sure that the data returned by this class are of the proper dimension, a call to this method
        must be performed before actually using the data generated by this class.

        In the grid2op framework, this is ensure because the :class:`grid2op.Environment` calls this method
        in its initialization.

        Parameters
        ----------
        backend: :class:`grid2op.Backend`
            The backend used by the :class;`Environment`.

        Returns
        -------

        """
        raise EnvError("check_validity not implemented")

    def done(self):
        """
        Whether the episode is over or not.

        Returns
        -------
        done: ``bool``
            ``True`` means the episode has arrived to the end (no more data to generate) ``False`` means that the episode
            is not over yet.

        """
        if self.max_iter >= 0:
            return self.curr_iter >= self.max_iter
        else:
            return False

    def forecasts(self):
        """
        This method is used to generate the forecasts that are made available to the :class:`grid2op.Agent`.
        This forecasts are behaving the same way than a list of tuple as the one returned by
        :func:`GridValue.load_next` method.

        The way they are generated depends on the GridValue class. If not forecasts are made available, then
        the empty list should be returned.

        Returns
        -------
        res: ``list``
            Each element of this list having the same type as what is returned by :func:`GridValue.load_next`.
        """
        return []

    @abstractmethod
    def next_chronics(self):
        """
        Load the next batch of chronics. This function is called after an episode has finished by the
        :class:`grid2op.Environment` or the :class:`grid2op.Runner`.

        A call to this function should also reset :attr:`GridValue.curr_iter` to 0.
        Returns
        -------
        ``None``
        """
        pass

    def tell_id(self, id_num):
        """
        Tell the backend to use one folder for the chronics in particular. This method is mainly use when the GridValue
        object can deal with many folder. In this case, this method is used by the :class:`grid2op.Runner` to indicate
        which chronics to load for the current simulated episode.

        This is important to ensure reproducibility, especially in parrallel computation settings.

        This should also be used in case of generation "on the fly" of the chronics to ensure the same property.

        By default it does nothing.

        Returns
        -------
        ``None``
        """
        warnings.warn("Class {} doesn't handle different input folder. \"tell_id\" method has no impact.".format(type(self).__name__))

    def get_id(self) -> str:
        """
        Utility to get the current name of the path of the data are looked at, if data are files.

        This could also be used to return a unique identifier to the generated chronics even in the case where they are
        generated on the fly, for example by return a hash of the seed.

        Returns
        -------
        res: ``str``
            A unique identifier of the chronics generated for this episode. For example, if the chronics comes from a
            specific folder, this could be the path to this folder.

        """
        warnings.warn("Class {} doesn't handle different input folder. \"get_id\" method will return \"\".".format(type(self).__name__))
        return ""

    def max_timestep(self):
        """
        This method returned the maximum timestep that the current episode can last.
        Note that if the :class:`grid2op.Agent` performs a bad action that leads to a game over, then the episode
        can lasts less.

        Returns
        -------
        res: int
            -1 if possibly infinite length of a positive integer representing the maximum duration of this episode

        """
        # warnings.warn("Class {} has possibly and infinite duration.".format(type(self).__name__))
        return self.max_iter


class ChangeNothing(GridValue):
    """
    This class is the most basic class to modify a powergrid values.
    It does nothing exceptie increasing :attr:`GridValue.max_iter` and the :attr:`GridValue.current_datetime`.
    """
    def __init__(self, time_interval=timedelta(minutes=5), max_iter=-1):
        GridValue.__init__(self, time_interval=time_interval, max_iter=max_iter)

    def initialize(self, order_backend_loads, order_backend_prods, order_backend_lines, order_backend_subs,
                   names_chronics_to_backend=None):
        self.n_gen = len(order_backend_prods)
        self.n_load = len(order_backend_loads)
        self.n_lines = len(order_backend_lines)
        self.curr_iter = 0

        self.maintenance_time = np.zeros(shape=(self.n_lines, ), dtype=np.int) - 1
        self.maintenance_duration = np.zeros(shape=(self.n_lines, ), dtype=np.int)
        self.hazard_duration = np.zeros(shape=(self.n_lines, ), dtype=np.int)

    def load_next(self):
        """
        This function does nothing but the two requirements of load_next ie:

          - increasing :attr:`GridValue.curr_iter` of 1
          - increasing :attr:`GridValue.current_datetime`

        Returns
        -------
        timestamp: ``datetime.datetime``
            The current timestamp for which the modifications have been generated.

        dict_: ``dict``
            Always empty, indicating i do nothing.

        maintenance_time: ``numpy.ndarray``, dtype:``int``
            Information about the next planned maintenance. See :attr:`GridValue.maintenance_time` for more information.

        maintenance_duration: ``numpy.ndarray``, dtype:``int``
            Information about the duration of next planned maintenance. See :attr:`GridValue.maintenance_duration`
            for more information.

        hazard_duration: ``numpy.ndarray``, dtype:``int``
            Information about the current hazard. See :attr:`GridValue.hazard_duration`
            for more information.

        """
        self.current_datetime += self.time_interval
        self.curr_iter += 1
        return self.current_datetime, {}, self.maintenance_time, self.maintenance_duration, self.hazard_duration

    def check_validity(self, backend):
        """

        Parameters
        ----------
        backend: :class:`grid2op.Backend`
            The backend, not used here.

        Returns
        -------
        res: ``bool``
            Always ``True``. As this doesn't change the powergird, there is no way to make invalid changed.
        """
        return True

    def next_chronics(self):
        """
        Restarts:

          - :attr:`GridValue.current_datetime` to its origin value ( 2019 / 01 / 01)
          - :attr:`GridValue.curr_iter` to 0

        Returns
        -------

        """
        self.current_datetime = datetime(year=2019, month=1, day=1)
        self.curr_iter = 0


# TODO handle case of missing files: equivalent to "change nothing" probably
# TODO change date time: should be read from file!
# TODO initialize datetime properly ! probably in a file (i just need start_date and time_interval
class GridStateFromFile(GridValue):
    """
    Read the injections values from a file stored on hard drive. More detailed about the files is provided in the
    :func:`GridStateFromFile.initialize` method.

    This class reads only files stored as csv. The header of the csv is mandatory and should represent the name of
    the objects. This names should either be matched to the name of the same object in the backend using the
    `names_chronics_to_backend` argument pass into the :func:`GridStateFromFile.initialize` (see
    :func:`GridValue.initialize` for more information) or match the names of the object in the backend.

    When the grid value is initialized, all present csv are read, sorted in order compatible with the backend and
    extracted as numpy array.

    For now, the current date and times are not read from file. It is mandatory that the chronics starts at 00:00 and
    its first time stamps is corresponds to January, 1st 2019.

    Chronics read from this files don't implement the "forecast" value.

    In this values, only 1 episode is stored. If the end of the episode is reached and another one should start, then
    it will loop from the beginning.

    Attributes
    ----------
    path: ``str``
        The path of the folder where the data are stored. It is recommended to set absolute path, and not relative
        paths.

    load_p: ``numpy.ndarray``, dtype: ``float``
        All the values of the load active values

    load_q: ``numpy.ndarray``, dtype: ``float``
        All the values of the load reactive values

    prod_p: ``numpy.ndarray``, dtype: ``float``
        All the productions setpoint active values.

    prod_v: ``numpy.ndarray``, dtype: ``float``
        All the productions setpoint voltage magnitude values.

    hazards: ``numpy.ndarray``, dtype: ``bool``
        This vector represents the possible hazards. It is understood as: ``True`` there is a hazard
        for the given powerline, ``False`` there is not.

    maintenance: ``numpy.ndarray``, dtype: ``bool``
        This vector represents the possible maintenance. It is understood as: ``True`` there is a maintenance
        for the given powerline, ``False`` there is not.

    current_index: ``int``
        The index of the last observation sent to the :class:`grid2op.Environment`.

    sep: ``str``, optional
        The csv columns separator. By defaults it's ";"

    names_chronics_to_backend: ``dict``
        This directory matches the name of the objects (line extremity, generator or load) to the same object in the
        backed. See the help of :func:`GridValue.initialize` for more information).
    """
    def __init__(self, path, sep=";", time_interval=timedelta(minutes=5), max_iter=-1):
        """
        Build an instance of GridStateFromFile. Such an instance should be built before an :class:`grid2op.Environment`
        is created.

        Parameters
        ----------
        path: ``str``
            Used to initialize :attr:`GridStateFromFile.path`

        sep: ``str``, optional
            Used to initialize :attr:`GridStateFromFile.sep`

        time_interval: ``datetime.timedelta``
            Used to initialize :attr:`GridValue.time_interval`

        max_iter: int, optional
            Used to initialize :attr:`GridValue.max_iter`

        """
        GridValue.__init__(self, time_interval=time_interval, max_iter=max_iter)

        self.path = path

        self.load_p = None  # numpy array corresponding to the current active load values in the power _grid. It has the same size as the number of loads
        self.load_q = None  # numpy array corresponding to the current reactive load values in the power _grid. It has the same size as the number of loads
        self.prod_p = None  # numpy array corresponding to the current active production values in the power _grid. It has the same size as the number of generators
        self.prod_v = None   # numpy array corresponding to the current voltage production setpoint values in the power _grid. It has the same size as the number of generators

        # for the two following vector, the convention is the following: False(line is disconnected) / True(line is connected)
        self.hazards = None   # numpy array representing the outage (unplanned), same size as the number of powerlines on the _grid.
        self.maintenance = None  # numpy array representing the _maintenance (planned withdrawal of a powerline), same size as the number of powerlines on the _grid.

        self.current_index = -1
        self.sep = sep

        self.names_chronics_to_backend = None

    def _assert_correct(self, dict_convert, order_backend):
        len_backend = len(order_backend)
        len_dict_keys = len(dict_convert)
        vals = set(dict_convert.values())
        lend_dict_values = len(vals)

        if len_dict_keys != len_backend:
            err_msg = "Conversion mismatch between backend data {} elements and converter data {} (keys)"
            raise IncorrectNumberOfElements(err_msg.format(len_backend, len_dict_keys))
        if lend_dict_values != len_backend:
            err_msg = "Conversion mismatch between backend data {} elements and converter data {} (values)"
            raise IncorrectNumberOfElements(err_msg.format(len_backend, lend_dict_values))

        for el in order_backend:
            if not el in vals:
                raise ChronicsError("Impossible to find element \"{}\" in the original converter data".format(el))

    def _assert_correct_second_stage(self, pandas_name, dict_convert, key, extra=""):
        for i, el in enumerate(pandas_name):
            if not el in dict_convert[key]:
                raise ChronicsError("Element named {} is found in the data (column {}) but it is not found on the powergrid for data of type \"{}\".\nData in files  are: {}\nConverter data are: {}".format(el, i+1, key, sorted(list(pandas_name)), sorted(list(dict_convert[key].keys()))))

    def initialize(self, order_backend_loads, order_backend_prods, order_backend_lines, order_backend_subs,
                   names_chronics_to_backend=None):
        """
        In this function, the numpy arrays are read from the csv using the panda.dataframe engine.

        In order to be valid, the folder located at :attr:`GridStateFromFile.path` should at least contain:

          - a file named "load_p.csv" used to initialize :attr:`GridStateFromFile.load_p`
          - a file named "load_q.csv" used to initialize :attr:`GridStateFromFile.load_q`
          - a file named "prod_p.csv" used to initialize :attr:`GridStateFromFile.prod_p`
          - a file named "prod_v.csv" used to initialize :attr:`GridStateFromFile.prod_v`
          - a file named "hazards.csv" used to initialize :attr:`GridStateFromFile.hazards`
          - a file named "_maintenance.csv" used to initialize :attr:`GridStateFromFile._maintenance`

        All these csv must have the same separator specified by :attr:`GridStateFromFile.sep`.

        The first row of these csv is understood as the name of the object concerned by the column. Either this name is
        present in the :class:`grid2op.Backend`, in this case no modification is performed, or in case the name
        is not found in the backend and in this case it must be specified in the "names_chronics_to_backend"
        _parameters how to understand it. See the help of :func:`GridValue.initialize` for more information
        about this dictionnary.

        All files should have the same number of rows.

        Parameters
        ----------
        See help of :func:`GridValue.initialize` for a detailed help about the _parameters.

        Returns
        -------
        ``None``

        """
        self.n_gen = len(order_backend_prods)
        self.n_load = len(order_backend_loads)
        self.n_lines = len(order_backend_lines)

        self.names_chronics_to_backend = copy.deepcopy(names_chronics_to_backend)
        if self.names_chronics_to_backend is None:
            self.names_chronics_to_backend = {}
        if not "loads" in self.names_chronics_to_backend:
            self.names_chronics_to_backend["loads"] = {k: k for k in order_backend_loads}
        else:
            self._assert_correct(self.names_chronics_to_backend["loads"], order_backend_loads)
        if not "prods" in self.names_chronics_to_backend:
            self.names_chronics_to_backend["prods"] = {k: k for k in order_backend_prods}
        else:
            self._assert_correct(self.names_chronics_to_backend["prods"], order_backend_prods)
        if not "lines" in self.names_chronics_to_backend:
            self.names_chronics_to_backend["lines"] = {k: k for k in order_backend_lines}
        else:
            self._assert_correct(self.names_chronics_to_backend["lines"], order_backend_lines)
        if not "subs" in self.names_chronics_to_backend:
            self.names_chronics_to_backend["subs"] = {k: k for k in order_backend_subs}
        else:
            self._assert_correct(self.names_chronics_to_backend["subs"], order_backend_subs)

        read_compressed = ".csv"
        if not os.path.exists(os.path.join(self.path, "load_p.csv")):
            # try to read compressed data
            if os.path.exists(os.path.join(self.path, "load_p.csv.bz2")):
                read_compressed = ".csv.bz2"
            elif os.path.exists(os.path.join(self.path, "load_p.zip")):
                read_compressed = ".zip"
            elif os.path.exists(os.path.join(self.path, "load_p.csv.gzip")):
                read_compressed = ".csv.gzip"
            elif os.path.exists(os.path.join(self.path, "load_p.csv.xz")):
                read_compressed = ".csv.xz"
            else:
                raise RuntimeError(
                    "GridStateFromFile: unable to locate the data files that should be at \"{}\"".format(self.path))
        load_p = pd.read_csv(os.path.join(self.path, "load_p{}".format(read_compressed)), sep=self.sep)
        load_q = pd.read_csv(os.path.join(self.path, "load_q{}".format(read_compressed)), sep=self.sep)
        prod_p = pd.read_csv(os.path.join(self.path, "prod_p{}".format(read_compressed)), sep=self.sep)
        prod_v = pd.read_csv(os.path.join(self.path, "prod_v{}".format(read_compressed)), sep=self.sep)
        hazards = pd.read_csv(os.path.join(self.path, "hazards{}".format(read_compressed)), sep=self.sep)
        maintenance = pd.read_csv(os.path.join(self.path, "maintenance{}".format(read_compressed)), sep=self.sep)

        order_backend_loads = {el: i for i, el in enumerate(order_backend_loads)}
        order_backend_prods = {el: i for i, el in enumerate(order_backend_prods)}
        order_backend_lines = {el: i for i, el in enumerate(order_backend_lines)}

        self._assert_correct_second_stage(load_p.columns, self.names_chronics_to_backend, "loads", "active")
        order_chronics_load_p = np.array([order_backend_loads[self.names_chronics_to_backend["loads"][el]]
                                          for el in load_p.columns]).astype(np.int)

        self._assert_correct_second_stage(load_q.columns, self.names_chronics_to_backend, "loads", "reactive")
        order_backend_load_q = np.array([order_backend_loads[self.names_chronics_to_backend["loads"][el]]
                                         for el in load_q.columns]).astype(np.int)

        self._assert_correct_second_stage(prod_p.columns, self.names_chronics_to_backend, "prods", "active")
        order_backend_prod_p = np.array([order_backend_prods[self.names_chronics_to_backend["prods"][el]]
                                         for el in prod_p.columns]).astype(np.int)

        self._assert_correct_second_stage(prod_v.columns, self.names_chronics_to_backend, "prods", "voltage magnitude")
        order_backend_prod_v = np.array([order_backend_prods[self.names_chronics_to_backend["prods"][el]]
                                         for el in prod_v.columns]).astype(np.int)

        self._assert_correct_second_stage(hazards.columns, self.names_chronics_to_backend, "lines", "hazards")
        order_backend_hazards = np.array([order_backend_lines[self.names_chronics_to_backend["lines"][el]]
                                          for el in hazards.columns]).astype(np.int)

        self._assert_correct_second_stage(maintenance.columns, self.names_chronics_to_backend, "lines", "maintenance")
        order_backend_maintenance = np.array([order_backend_lines[self.names_chronics_to_backend["lines"][el]]
                                              for el in maintenance.columns]).astype(np.int)
        self.load_p = copy.deepcopy(load_p.values[:, np.argsort(order_chronics_load_p)])
        self.load_q = copy.deepcopy(load_q.values[:, np.argsort(order_backend_load_q)])
        self.prod_p = copy.deepcopy(prod_p.values[:, np.argsort(order_backend_prod_p)])
        self.prod_v = copy.deepcopy(prod_v.values[:, np.argsort(order_backend_prod_v)])
        self.hazards = copy.deepcopy(hazards.values[:, np.argsort(order_backend_hazards)])
        self.maintenance = copy.deepcopy(maintenance.values[:, np.argsort(order_backend_maintenance)])

        self.maintenance_time = np.zeros(shape=(self.load_p.shape[0], self.n_lines), dtype=np.int) - 1
        self.maintenance_duration = np.zeros(shape=(self.load_p.shape[0], self.n_lines), dtype=np.int)
        self.hazard_duration = np.zeros(shape=(self.load_p.shape[0], self.n_lines), dtype=np.int)

        for line_id in range(self.n_lines):
            self.maintenance_time[:, line_id] = self.get_maintenance_time_1d(self.maintenance[:, line_id])
            self.maintenance_duration[:, line_id] = self.get_maintenance_duration_1d(self.maintenance[:, line_id])
            self.hazard_duration[:, line_id] = self.get_hazard_duration_1d(self.hazards[:, line_id])
            # if line_id == 17 and np.sum(self.hazards) > 0:
            #     pdb.set_trace()

        # there are _maintenance and hazards only if the value in the file is not 0.
        self.maintenance = self.maintenance != 0.
        self.hazards = self.hazards != 0.

        # TODO

        self.curr_iter = 0
        if self.max_iter == -1:
            # if the number of maximum time step is not set yet, we set it to be the number of
            # data in the chronics (number of rows of the files) -1.
            # the -1 is present because the initial grid state doesn't count as a "time step" but is read
            # from these data.
            self.max_iter = self.load_p.shape[0]-1

    def done(self):
        """
        Compare to :func:`GridValue.done` an episode can be over for 2 main reasons:

          - :attr:`GridValue.max_iter` has been reached
          - There are no data in the csv.

        The episode is done if one of the above condition is met.

        Returns
        -------
        res: ``bool``
            Whether the episode has reached its end or not.

        """
        res = False
        if self.current_index+1 >= self.load_p.shape[0]:
            res = True
        elif self.max_iter > 0:
            if self.curr_iter > self.max_iter:
                res = True

        return res

    def load_next(self):
        self.current_index += 1

        if self.current_index >= self.load_q.shape[0]:
            raise StopIteration
        if self.max_iter > 0:
            if self.curr_iter > self.max_iter:
                raise StopIteration

        res = {}
        res["injection"] = {"load_p": self.load_p[self.current_index, :], "load_q": self.load_q[self.current_index, :],
                            "prod_p": self.prod_p[self.current_index, :], "prod_v": self.prod_v[self.current_index, :]}
        res["maintenance"] = self.maintenance[self.current_index, :]
        res["hazards"] = self.hazards[self.current_index, :]

        self.current_datetime += self.time_interval
        self.curr_iter += 1

        maintenance_time = self.maintenance_time[self.current_index, :]
        maintenance_duration = self.maintenance_duration[self.current_index, :]
        hazard_duration = self.hazard_duration[self.current_index, :]

        return self.current_datetime, res, maintenance_time, maintenance_duration, hazard_duration

    def check_validity(self, backend):
        """
        A call to this method ensure that the action that will be sent to the current :class:`grid2op.Environment`
        can be properly implemented by its :class:`grid2op.Backend`.
        This specific method check that the dimension of all vectors are consistent

        Parameters
        ----------
        backend: :class:`grid2op.Backend.Backend`
            The backend used by the :class:`grid2op.Environment.Environment`

        Returns
        -------
        ``None``
        """

        if self.load_p.shape[1] != backend.n_loads:
            msg_err = "for the active part. It should be {} but is in fact {}"
            raise IncorrectNumberOfLoads(msg_err.format(backend.n_loads, self.load_p.shape[1]))
        if self.load_q.shape[1] != backend.n_loads:
            msg_err = "for the reactive part. It should be {} but is in fact {}"
            raise IncorrectNumberOfLoads(msg_err.format(backend.n_loads, self.load_q.shape[1]))

        if self.prod_p.shape[1] != backend.n_generators:
            msg_err = "for the active part. It should be {} but is in fact {}"
            raise IncorrectNumberOfGenerators(msg_err.format(backend.n_generators, self.prod_p.shape[1]))
        if self.prod_v.shape[1] != backend.n_generators:
            msg_err = "for the voltage part. It should be {} but is in fact {}"
            raise IncorrectNumberOfGenerators(msg_err.format(backend.n_generators, self.prod_v.shape[1]))

        if self.hazards.shape[1] != backend.n_lines:
            msg_err = "for the outage. It should be {} but is in fact {}"
            raise IncorrectNumberOfLines(msg_err.format(backend.n_lines, self.hazards.shape[1]))
        if self.maintenance.shape[1] != backend.n_lines:
            msg_err = "for the maintenance. It should be {} but is in fact {}"
            raise IncorrectNumberOfLines(msg_err.format(backend.n_lines, self.maintenance.shape[1]))

        if self.maintenance_time.shape[1] != backend.n_lines:
            msg_err = "for the maintenance times. It should be {} but is in fact {}"
            raise IncorrectNumberOfLines(msg_err.format(backend.n_lines, self.maintenance_time.shape[1]))

        if self.maintenance_duration.shape[1] != backend.n_lines:
            msg_err = "for the maintenance durations. It should be {} but is in fact {}"
            raise IncorrectNumberOfLines(msg_err.format(backend.n_lines, self.maintenance_duration.shape[1]))

        if self.hazard_duration.shape[1] != backend.n_lines:
            msg_err = "for the hazard durations. It should be {} but is in fact {}"
            raise IncorrectNumberOfLines(msg_err.format(backend.n_lines, self.hazard_duration.shape[1]))

        n = self.load_p.shape[0]
        for name_arr, arr in zip(["load_q", "load_p", "prod_v", "prod_p", "maintenance", "hazards",
                                  "maintenance time", "maintenance duration", "hazard duration"],
                                 [self.load_q, self.load_p, self.prod_v, self.prod_p, self.maintenance, self.hazards,
                                  self.maintenance_time, self.maintenance_duration, self.hazard_duration]):
            if arr.shape[0] != n:
                msg_err = "Array {} has not the same number of rows of load_p. The chronics cannot be loaded properly."
                raise EnvError(msg_err.format(name_arr))

        if self.max_iter > 0:
            if self.max_iter > self.load_p.shape[0]:
                # TODO make a test for that!
                # is this > or >= ????
                msg_err = "Files count {} rows and you ask this episode to last at {} timestep."
                raise InsufficientData(msg_err.format(self.load_p.shape[0], self.max_iter))

    def next_chronics(self):
        self.current_datetime = datetime(year=2019, month=1, day=1)
        self.current_index = -1
        self.curr_iter = 0

    def get_id(self) -> str:
        return self.path


class GridStateFromFileWithForecasts(GridStateFromFile):
    """
    An extension of :class:`GridStateFromFile` that implements the "forecast" functionality.

    Forecast are also read from a file. For this class, only 1 forecast per timestep is read. The "forecast"
    present in the file at row $i$ is the one available at the corresponding time step, so valid for the grid state
    at the next time step.

    To have more advanced forecasts, this class could be overridden.

    Attributes
    ----------
    load_p_forecast: ``numpy.ndarray``, dtype: ``float``
        Array used to store the forecasts of the load active values.

    load_q_forecast: ``numpy.ndarray``, dtype: ``float``
        Array used to store the forecasts of the load reactive values.

    prod_p_forecast: ``numpy.ndarray``, dtype: ``float``
        Array used to store the forecasts of the generator active production setpoint.

    prod_v_forecast: ``numpy.ndarray``, dtype: ``float``
        Array used to store the forecasts of the generator voltage magnitude setpoint.

    maintenance_forecast: ``numpy.ndarray``, dtype: ``float``
        Array used to store the forecasts of the _maintenance operations.

    """
    def __init__(self, path, sep=";", time_interval=timedelta(minutes=5), max_iter=-1):
        GridStateFromFile.__init__(self, path, sep=sep, time_interval=time_interval, max_iter=max_iter)

        self.load_p_forecast = None
        self.load_q_forecast = None
        self.prod_p_forecast = None
        self.prod_v_forecast = None
        self.maintenance_forecast = None

    def initialize(self, order_backend_loads, order_backend_prods, order_backend_lines, order_backend_subs,
                   names_chronics_to_backend=None):
        """
        The same condition as :class:`GridStateFromFile.initialize` applies also for
        :attr:`GridStateFromFileWithForecasts.load_p_forecast`,  :attr:`GridStateFromFileWithForecasts.load_q_forecast`,
        :attr:`GridStateFromFileWithForecasts.prod_p_forecast`,
        :attr:`GridStateFromFileWithForecasts.prod_v_forecast` and
        :attr:`GridStateFromFileWithForecasts.maintenance_forecast`.

        Parameters
        ----------
        See help of :func:`GridValue.initialize` for a detailed help about the _parameters.

        Returns
        -------
        ``None``

        """
        super().initialize(order_backend_loads, order_backend_prods, order_backend_lines, order_backend_subs,
                           names_chronics_to_backend)
        read_compressed = ".csv"
        if not os.path.exists(os.path.join(self.path, "load_p.csv")):
            # try to read compressed data
            if os.path.exists(os.path.join(self.path, "load_p.csv.bz2")):
                read_compressed = ".csv.bz2"
            elif os.path.exists(os.path.join(self.path, "load_p.zip")):
                read_compressed = ".zip"
            elif os.path.exists(os.path.join(self.path, "load_p.csv.gzip")):
                read_compressed = ".csv.gzip"
            elif os.path.exists(os.path.join(self.path, "load_p.csv.xz")):
                read_compressed = ".csv.xz"
            else:
                raise RuntimeError(
                    "GridStateFromFileWithForecasts: unable to locate the data files that should be at \"{}\"".format(self.path))

        load_p = pd.read_csv(os.path.join(self.path, "load_p_forecasted{}".format(read_compressed)), sep=self.sep)
        load_q = pd.read_csv(os.path.join(self.path, "load_q_forecasted{}".format(read_compressed)), sep=self.sep)
        prod_p = pd.read_csv(os.path.join(self.path, "prod_p_forecasted{}".format(read_compressed)), sep=self.sep)
        prod_v = pd.read_csv(os.path.join(self.path, "prod_v_forecasted{}".format(read_compressed)), sep=self.sep)
        maintenance = pd.read_csv(os.path.join(self.path, "maintenance_forecasted{}".format(read_compressed)),
                                  sep=self.sep)

        order_backend_loads = {el: i for i, el in enumerate(order_backend_loads)}
        order_backend_prods = {el: i for i, el in enumerate(order_backend_prods)}
        order_backend_lines = {el: i for i, el in enumerate(order_backend_lines)}

        order_chronics_load_p = np.array([order_backend_loads[self.names_chronics_to_backend["loads"][el]]
                                          for el in load_p.columns]).astype(np.int)
        order_backend_load_q = np.array([order_backend_loads[self.names_chronics_to_backend["loads"][el]]
                                         for el in load_q.columns]).astype(np.int)
        order_backend_prod_p = np.array([order_backend_prods[self.names_chronics_to_backend["prods"][el]]
                                         for el in prod_p.columns]).astype(np.int)
        order_backend_prod_v = np.array([order_backend_prods[self.names_chronics_to_backend["prods"][el]]
                                         for el in prod_v.columns]).astype(np.int)
        order_backend_maintenance = np.array([order_backend_lines[self.names_chronics_to_backend["lines"][el]]
                                              for el in maintenance.columns]).astype(np.int)

        self.load_p_forecast = copy.deepcopy(load_p.values[:, np.argsort(order_chronics_load_p)])
        self.load_q_forecast = copy.deepcopy(load_q.values[:, np.argsort(order_backend_load_q)])
        self.prod_p_forecast = copy.deepcopy(prod_p.values[:, np.argsort(order_backend_prod_p)])
        self.prod_v_forecast = copy.deepcopy(prod_v.values[:, np.argsort(order_backend_prod_v)])
        self.maintenance_forecast = copy.deepcopy(maintenance.values[:, np.argsort(order_backend_maintenance)])

        # there are _maintenance and hazards only if the value in the file is not 0.
        self.maintenance_forecast = self.maintenance != 0.

    def check_validity(self, backend):
        super(GridStateFromFileWithForecasts, self).check_validity(backend)

        if self.load_p_forecast.shape[1] != backend.n_loads:
            raise IncorrectNumberOfLoads("for the active part. It should be {} but is in fact {}".format(backend.n_loads, len(self.load_p)))
        if self.load_q_forecast.shape[1] != backend.n_loads:
            raise IncorrectNumberOfLoads("for the reactive part. It should be {} but is in fact {}".format(backend.n_loads, len(self.load_q)))

        if self.prod_p_forecast.shape[1] != backend.n_generators:
            raise IncorrectNumberOfGenerators("for the active part. It should be {} but is in fact {}".format(backend.n_generators, len(self.prod_p)))
        if self.prod_v_forecast.shape[1] != backend.n_generators:
            raise IncorrectNumberOfGenerators("for the voltage part. It should be {} but is in fact {}".format(backend.n_generators, len(self.prod_v)))

        if self.maintenance_forecast.shape[1] != backend.n_lines:
            raise IncorrectNumberOfLines("for the _maintenance. It should be {} but is in fact {}".format(backend.n_lines, len(self.maintenance)))

        n = self.load_p.shape[0]
        for name_arr, arr in zip(["load_q", "load_p", "prod_v", "prod_p", "maintenance", "outage"],
                                 [self.load_q_forecast, self.load_p_forecast, self.prod_v_forecast,
                                  self.prod_p_forecast, self.maintenance_forecast]):
            if arr.shape[0] < n:
                raise EnvError("Array for forecast {}_forecasted as not the same number of rows of load_p. The chronics cannot be loaded properly.".format(name_arr))

    def forecasts(self):
        """
        This is the major difference between :class:`GridStateFromFileWithForecasts` and :class:`GridStateFromFile`.
        It returns non empty forecasts.

        As explained in the :func:`GridValue.forecasts`, forecasts are made of list of tuple. Each tuple having
        exactly 2 elements:

          1. Is the time stamp of the forecast
          2. An :class:`grid2op.Action` representing the modification of the powergrid after the forecast.

        For this class, only the forecast of the next time step is given, and only for the injections and maintenance.

        Returns
        -------
        See :func:`GridValue.forecasts` for more information.

        """
        res = {}
        res["injection"] = {"load_p": self.load_p_forecast[self.current_index, :],
                            "load_q": self.load_q_forecast[self.current_index, :],
                            "prod_p": self.prod_p_forecast[self.current_index, :],
                            "prod_v": self.prod_v_forecast[self.current_index, :]}
        res["maintenance"] = self.maintenance_forecast[self.current_index, :]

        forecast_datetime = self.current_datetime + self.time_interval
        return [(forecast_datetime, res)]

    def get_id(self) -> str:
        return self.path


# TODO change the order of the reading.
class Multifolder(GridValue):
    """
    The classes :class:`GridStateFromFile` and :class:`GridStateFromFileWithForecasts` implemented the reading of a
    single folder representing a single episode.

    This class is here to "loop" between different episode. Each one being stored in a folder readable by
    :class:`GridStateFromFile` or one of its derivate (eg. :class:`GridStateFromFileWithForecasts`).

    Chronics are always read in the alpha-numeric order for this class. This means that if the folder is not modified,
    the data are always loaded in the same order, regardless of the :class:`grid2op.Backend`, :class:`grid2op.Agent` or
    :class:`grid2op.Environment`.

    Attributes
    -----------
    gridvalueClass: ``type``, optional
        Type of class used to read the data from the disk. It defaults to :class:`GridStateFromFile`.

    data: :class:`GridStateFromFile`
        Data that will be loaded and used to produced grid state and forecasted values.


    path: ``str``
        Path where the folders of the episodes are stored.

    sep: ``str``
        Columns separtor, forwarded to :attr:`Multifolder.data` when it's built at the beginning of each episode.

    subpaths: ``list``
        List of all the episode that can be "played". It's a sorted list of all the directory in
        :attr:`Multifolder.path`. Each one should contain data in a format that is readable by
        :attr:`MultiFolder.gridvalueClass`.

    id_chron_folder_current: ``int``
        Id (in :attr:`MultiFolder.subpaths`) for which data are generated in the current episode.
    """
    def __init__(self, path,
                 time_interval=timedelta(minutes=5),
                 gridvalueClass=GridStateFromFile,
                 sep=";", max_iter=-1):
        GridValue.__init__(self, time_interval=time_interval, max_iter=max_iter)
        self.gridvalueClass = gridvalueClass
        self.data = None
        self.path = os.path.abspath(path)
        self.sep = sep
        self.subpaths = [os.path.join(self.path, el) for el in os.listdir(self.path)
                         if os.path.isdir(os.path.join(self.path, el))]
        self.subpaths.sort()
        if len(self.subpaths) == 0:
            raise ChronicsNotFoundError("Not chronics are found in \"{}\". Make sure there are at least "
                                        "1 chronics folder there.".format(self.path))
        # np.random.shuffle(self.subpaths)
        self.id_chron_folder_current = 0

    def initialize(self, order_backend_loads, order_backend_prods, order_backend_lines, order_backend_subs,
                   names_chronics_to_backend=None):

        self.n_gen = len(order_backend_prods)
        self.n_load = len(order_backend_loads)
        self.n_lines = len(order_backend_lines)
        # print("max_iter: {}".format(self.max_iter))
        self.data = self.gridvalueClass(time_interval=self.time_interval,
                                        sep=self.sep,
                                        path=self.subpaths[self.id_chron_folder_current],
                                        max_iter=self.max_iter)
        self.data.initialize(order_backend_loads, order_backend_prods, order_backend_lines, order_backend_subs,
                             names_chronics_to_backend=names_chronics_to_backend)
        # print(" self.data.max_iter: {}".format(self.data.max_iter))
        # print(" self.gridvalueClass: {}".format(self.gridvalueClass))

    def done(self):
        """
        Tells the :class:`grid2op.Environment` if the episode is over.

        Returns
        -------
        res: ``bool``
            Whether or not the episode, represented by :attr:`MultiFolder.data` is over.

        """
        return self.data.done()

    def load_next(self):
        """
        Load the next data from the current episode. It loads the next time step for the current episode.

        Returns
        -------
        See the return type of  :class:`GridStateFromFile.load_next` (or of :attr:`MultiFolder.gridvalueClass` if it
        has been changed) for more information.

        """
        return self.data.load_next()

    def check_validity(self, backend):
        """
        This method check that the data loaded can be properly read and understood by the :class:`grid2op.Backend`.

        Parameters
        ----------
        backend: :class:`grid2op.Backend`
            The backend used for the experiment.

        Returns
        -------
        See the return type of  :class:`GridStateFromFile.check_validity` (or of :attr:`MultiFolder.gridvalueClass` if it
        has been changed) for more information.
        """
        return self.data.check_validity(backend)

    def forecasts(self):
        """
        The representation of the forecasted grid state(s), if any.

        Returns
        -------
        See the return type of  :class:`GridStateFromFile.forecasts` (or of :attr:`MultiFolder.gridvalueClass` if it
        has been changed) for more information.
        """
        return self.data.forecasts()

    def next_chronics(self):
        """
        Load the next episode.

        Note that :func:`MultiFolder.initialize` must be called after a call to this method has been performed. This is
        either done by the :class:`grid2op.Environemnt` or by the :class:`grid2op.Runner`.

        Returns
        -------
        ``None``

        """
        self.id_chron_folder_current += 1
        self.id_chron_folder_current %= len(self.subpaths)

    def tell_id(self, id_num):
        """
        This tells this chronics to load for the next episode.
        By default, if id_num is greater than the number of episode, it is equivalent at restarting from the first
        one: episode are played indefinitely in the same order.

        Parameters
        ----------
        id_num: ``int``
            Id of the chronics to load.

        Returns
        -------

        """
        self.id_chron_folder_current = id_num
        self.id_chron_folder_current %= len(self.subpaths)
        # print("Chronics handler: going to chronics {}".format(self.id_chron_folder_current))

    def get_id(self) -> str:
        """
        Full absolute path of the current folder used for the current episode.

        Returns
        -------
        res: ``str``
            Path from which the data are generated for the current episode.
        """
        return self.subpaths[self.id_chron_folder_current]

    def max_timestep(self):
        return self.data.max_timestep()


class ChronicsHandler:
    """
    Represents a Chronics handler that returns a grid state.

    As stated previously, it is not recommended to make an directly an object from the class :class:`GridValue`. This
    utility will ensure that the creation of such objects are properly made.

    The types of chronics used can be specified in the :attr:`ChronicsHandler.chronicsClass` attribute.

    Attributes
    ----------
    chronicsClass: ``type``, optional
        Type of chronics that will be loaded and generated. Default is :class:`ChangeNothing` (*NB* the class, and not
        an object / instance of the class should be send here.) This should be a derived class from :class:`GridValue`.

    kwargs: ``dict``, optional
        key word arguments that will be used to build new chronics.

    max_iter: ``int``, optional
        Maximum number of iterations per episode.

    real_data: :class:`GridValue`
        An instance of type given by :attr:`ChronicsHandler.chronicsClass`.

    seed: ``float``
        Seed to use for reproducible experiments (currently not implemented)
    """
    def __init__(self, chronicsClass=ChangeNothing, time_interval=timedelta(minutes=5), max_iter=-1,
                 **kwargs):
        if not isinstance(chronicsClass, type):
            raise Grid2OpException("Parameter \"chronicsClass\" used to build the ChronicsHandler should be a type (a class) and not an object (an instance of a class). It is currently \"{}\"".format(type(legalActClass)))
        if not issubclass(chronicsClass, GridValue):
            raise ChronicsError("ChronicsHandler: the \"chronicsClass\" argument should be a derivative of the \"Grid2Op.GridValue\" type and not {}.".format(type(chronicsClass)))
        self.chronicsClass = chronicsClass
        self.kwargs = kwargs
        self.max_iter = max_iter

        self.real_data = None
        try:
            self.real_data = self.chronicsClass(time_interval=time_interval, max_iter=self.max_iter,
                                                **self.kwargs)
        except TypeError:
            raise ChronicsError("Impossible to build a chronics of type {} with arguments in {}".format(chronicsClass, self.kwargs))
        self.seed = None

    def initialize(self, order_backend_loads, order_backend_prods, order_backend_lines, order_backend_subs,
                   names_chronics_to_backend=None):
        """
        After being loaded, this method will initialize the data.

        See definition of :func:`GridValue.initialize` for more information about this method.

        Returns
        -------
        ``None``

        """
        self.real_data.initialize(order_backend_loads, order_backend_prods, order_backend_lines, order_backend_subs,
                                  names_chronics_to_backend)

    def check_validity(self, backend):
        """
        This method ensure the data are valid and compatible with the backend used.

        See definition of :func:`GridValue.check_validity` for more information about this method.

        Returns
        -------
        ``None``

        """
        self.real_data.check_validity(backend)

    def next_time_step(self):
        """
        This method returns the modification of the powergrid at the next time step for the same episode.

        See definition of :func:`GridValue.load_next` for more information about this method.

        """
        res = self.real_data.load_next()
        return res

    def done(self):
        """
        This method returns whether or not the episode is done.

        See definition of :func:`GridValue.done` for more information about this method.

        """
        return self.real_data.done()

    def forecasts(self):
        """
        This method returns the forecasts of the data.

        See definition of :func:`GridValue.forecasts` for more information about this method.

        """
        return self.real_data.forecasts()

    def next_chronics(self):
        """
        This method is called when changing the episode after game over or after it has reached the end.

        See definition of :func:`GridValue.next_chronics` for more information about this method.

        """
        self.real_data.next_chronics()

    def tell_id(self, id_num):
        """
        This method is called when setting a given episode after game over or after it has reached the end.

        See definition of :func:`GridValue.tell_id` for more information about this method.

        """
        self.real_data.tell_id(id_num=id_num)

    def max_timestep(self):
        """
        This method gives the maximum number of time step an episode can last.

        See definition of :func:`GridValue.max_timestep` for more information about this method.

        """
        return self.real_data.max_timestep()

    def get_id(self):
        """
        This method gives a unique identifier for the current episode.

        See definition of :func:`GridValue.get_id` for more information about this method.

        """
        return self.real_data.get_id()

    def seed(self, seed):
        """
        Use to set the seed in case of non determinitics chronics.

        Attributes
        -----------
        seed: ``float``
            The seed to set

        """
        self.seed = seed
        #TODO set seed in the data
