# The MountainSort MDA file format is described here:
# https://mountainsort.readthedocs.io/en/latest/mda_file_format.html

# SpikeInterface can read this format easily.

# In Tim's data for IM-1478/2022-07-25_15-30-00, the `firings.mda` file contains 145 units and 3,040,410 spikes.

# To create test data of a reasonable size, we will trim the spike times to only those in the first 30,000 samples.
# This results in 63 units and 462 spikes.

# To run this script, copy Tim's MountainSort output file `firings.mda` for IM-1478/2022-07-25_15-30-00
# to your computer and adjust the path in this script to point to the location of the data on your computer.

# The `firings.mda` file should be 72,969,860 bytes. The test checks specific properties of the file generated by this
# script.

# Then run this script from the command line from the root of the repo:
#   python tests/test_data/create_processed_ephys_test_data.py

from pathlib import Path

from spikeinterface.extractors import read_mda_sorting, MdaSortingExtractor

# NOTE: Adjust this path to point to the location of Tim's sorted data for IM-1478/2022-07-25_15-30-00
firings_mda_file_path = Path("/Users/rly/Documents/NWB/berke-lab-to-nwb/data/ephys/mntsort_output/firings.mda")
sampling_frequency = 30_000

# Read the .mda file
sorting = read_mda_sorting(firings_mda_file_path, sampling_frequency=sampling_frequency)

# Trim the spike times to only those in the first 30,000 samples
sorting_trimmed = sorting.frame_slice(start_frame=0, end_frame=3_000)

# Create a new directory to store the trimmed data
new_data_dir = Path("./tests/test_data/processed_ephys")
new_data_dir.mkdir(parents=True, exist_ok=True)
output_file_path = new_data_dir / "firings.mda"

# Write the trimmed spike sorting data to a new .mda file
MdaSortingExtractor.write_sorting(sorting=sorting_trimmed, save_path=output_file_path)

print(f"Trimmed spike sorting data saved to: {output_file_path}")
