from greenflow.dataframe_flow import (
    ConfSchema, PortsSpecSchema, MetaDataSchema)
from greenflow.dataframe_flow import Node
from greenflow.dataframe_flow.template_node_mixin import TemplateNodeMixin
from ..node_hdf_cache import NodeHDFCacheMixin

__all__ = ['MinNode']


class MinNode(TemplateNodeMixin, NodeHDFCacheMixin, Node):

    def init(self):
        TemplateNodeMixin.init(self)
        self.INPUT_PORT_NAME = 'in'
        self.OUTPUT_PORT_NAME = 'out'
        port_type = PortsSpecSchema.port_type
        port_inports = {
            self.INPUT_PORT_NAME: {
                port_type: [
                    "pandas.DataFrame", "cudf.DataFrame",
                    "dask_cudf.DataFrame", "dask.dataframe.DataFrame"
                ]
            },
        }
        port_outports = {
            self.OUTPUT_PORT_NAME: {
                port_type: "${port:in}"
            }
        }
        cols_required = {"asset": "int64"}
        if 'column' in self.conf:
            retention = {self.conf['column']: "float64",
                         "asset": "int64"}
        else:
            retention = {"asset": "int64"}
        meta_inports = {
            self.INPUT_PORT_NAME: cols_required
        }
        meta_outports = {
            self.OUTPUT_PORT_NAME: {
                MetaDataSchema.META_OP: MetaDataSchema.META_OP_RETENTION,
                MetaDataSchema.META_DATA: retention
            }
        }
        self.template_ports_setup(
            in_ports=port_inports,
            out_ports=port_outports
        )
        self.template_meta_setup(
            in_ports=meta_inports,
            out_ports=meta_outports
        )

    def update(self):
        TemplateNodeMixin.update(self)
        meta_inports = self.template_meta_setup().inports
        required = meta_inports[self.INPUT_PORT_NAME]
        if 'column' in self.conf:
            col_name = self.conf['column']
            input_meta = self.get_input_meta()
            if self.INPUT_PORT_NAME not in input_meta:
                required[col_name] = None
            else:
                col_from_inport = input_meta[self.INPUT_PORT_NAME]
                if col_name in col_from_inport:
                    required[col_name] = col_from_inport[col_name]
                else:
                    required[col_name] = None
        meta_inports[self.INPUT_PORT_NAME] = required
        self.template_meta_setup(in_ports=meta_inports, out_ports=None)

    def conf_schema(self):
        json = {
            "title": "Minimum Value Node configure",
            "type": "object",
            "description": "Compute the minimum value of the key column",
            "properties": {
                "column":  {
                    "type": "string",
                    "description": "column to calculate the minimum value"
                }
            },
            "required": ["column"],
        }
        input_meta = self.get_input_meta()
        if self.INPUT_PORT_NAME in input_meta:
            col_from_inport = input_meta[self.INPUT_PORT_NAME]
            enums = [col for col in col_from_inport.keys()]
            json['properties']['column']['enum'] = enums
            ui = {
            }
            return ConfSchema(json=json, ui=ui)
        else:
            ui = {
                "column": {"ui:widget": "text"}
            }
            return ConfSchema(json=json, ui=ui)

    def process(self, inputs):
        """
        Compute the minium value of the key column which is defined in the
        `column` of the node's conf

        Arguments
        -------
         inputs: list
            list of input dataframes.
        Returns
        -------
        dataframe
        """
        input_df = inputs[self.INPUT_PORT_NAME]
        min_column = self.conf['column']
        volume_df = input_df[[min_column,
                              "asset"]].groupby(["asset"]).min().reset_index()
        volume_df.columns = ['asset', min_column]
        return {self.OUTPUT_PORT_NAME: volume_df}
