"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToKinesisStreams = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
// Imports
const api = require("aws-cdk-lib/aws-apigateway");
const iam = require("aws-cdk-lib/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
// Note: To ensure CDKv2 compatibility, keep the import statement for Construct separate
const constructs_1 = require("constructs");
/**
 * @summary The ApiGatewayToKinesisStreams class.
 */
class ApiGatewayToKinesisStreams extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of the ApiGatewayToKinesisStreams class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {ApiGatewayToKinesisStreamsProps} props - user provided props for the construct.
     * @since 1.62.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        defaults.CheckProps(props);
        // Setup the Kinesis stream
        this.kinesisStream = defaults.buildKinesisStream(scope, {
            existingStreamObj: props.existingStreamObj,
            kinesisStreamProps: props.kinesisStreamProps
        });
        // Setup the API Gateway
        const globalRestApiResponse = defaults.GlobalRestApi(this, props.apiGatewayProps, props.logGroupProps);
        this.apiGateway = globalRestApiResponse.api;
        this.apiGatewayCloudWatchRole = globalRestApiResponse.role;
        this.apiGatewayLogGroup = globalRestApiResponse.logGroup;
        // Setup the API Gateway role
        this.apiGatewayRole = new iam.Role(this, 'api-gateway-role', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
        });
        this.kinesisStream.grantWrite(this.apiGatewayRole);
        // Setup API Gateway methods
        const requestValidator = this.apiGateway.addRequestValidator('request-validator', {
            requestValidatorName: 'request-body-validator',
            validateRequestBody: true
        });
        // PutRecord
        const putRecordResource = this.apiGateway.root.addResource('record');
        defaults.addProxyMethodToApiResource({
            service: 'kinesis',
            action: 'PutRecord',
            apiGatewayRole: this.apiGatewayRole,
            apiMethod: 'POST',
            apiResource: putRecordResource,
            requestTemplate: this.buildPutRecordTemplate(props.putRecordRequestTemplate),
            additionalRequestTemplates: this.buildAdditionalPutRecordTemplates(props.additionalPutRecordRequestTemplates),
            contentType: "'x-amz-json-1.1'",
            requestValidator,
            requestModel: { 'application/json': this.buildPutRecordModel(props.putRecordRequestModel) },
            integrationResponses: props.putRecordIntegrationResponses
        });
        // PutRecords
        const putRecordsResource = this.apiGateway.root.addResource('records');
        defaults.addProxyMethodToApiResource({
            service: 'kinesis',
            action: 'PutRecords',
            apiGatewayRole: this.apiGatewayRole,
            apiMethod: 'POST',
            apiResource: putRecordsResource,
            requestTemplate: this.buildPutRecordsTemplate(props.putRecordsRequestTemplate),
            additionalRequestTemplates: this.buildAdditionalPutRecordTemplates(props.additionalPutRecordsRequestTemplates),
            contentType: "'x-amz-json-1.1'",
            requestValidator,
            requestModel: { 'application/json': this.buildPutRecordsModel(props.putRecordsRequestModel) },
            integrationResponses: props.putRecordsIntegrationResponses
        });
        if (props.createCloudWatchAlarms === undefined || props.createCloudWatchAlarms) {
            // Deploy best practices CW Alarms for Kinesis Stream
            this.cloudwatchAlarms = defaults.buildKinesisStreamCWAlarms(this);
        }
    }
    /**
     * This method transforms the value of each request template by replacing the stream name placeholder value with the
     * actual name of the stream resource
     *
     * @param templates The additional request templates to transform.
     */
    buildAdditionalPutRecordTemplates(templates) {
        const transformedTemplates = {};
        for (const key in templates) {
            if (templates[key] !== undefined) {
                transformedTemplates[key] = templates[key].replace("${StreamName}", this.kinesisStream.streamName);
            }
        }
        return transformedTemplates;
    }
    buildPutRecordTemplate(putRecordTemplate) {
        if (putRecordTemplate !== undefined) {
            return putRecordTemplate.replace("${StreamName}", this.kinesisStream.streamName);
        }
        return `{ "StreamName": "${this.kinesisStream.streamName}", "Data": "$util.base64Encode($input.json('$.data'))", "PartitionKey": "$input.path('$.partitionKey')" }`;
    }
    buildPutRecordModel(putRecordModel) {
        let modelProps;
        if (putRecordModel !== undefined) {
            modelProps = putRecordModel;
        }
        else {
            modelProps = {
                contentType: 'application/json',
                modelName: 'PutRecordModel',
                description: 'PutRecord proxy single-record payload',
                schema: {
                    schema: api.JsonSchemaVersion.DRAFT4,
                    title: 'PutRecord proxy single-record payload',
                    type: api.JsonSchemaType.OBJECT,
                    required: ['data', 'partitionKey'],
                    properties: {
                        data: { type: api.JsonSchemaType.STRING },
                        partitionKey: { type: api.JsonSchemaType.STRING }
                    }
                }
            };
        }
        return this.apiGateway.addModel('PutRecordModel', modelProps);
    }
    buildPutRecordsTemplate(putRecordsTemplate) {
        if (putRecordsTemplate !== undefined) {
            return putRecordsTemplate.replace("${StreamName}", this.kinesisStream.streamName);
        }
        return `{ "StreamName": "${this.kinesisStream.streamName}", "Records": [ #foreach($elem in $input.path('$.records')) { "Data": "$util.base64Encode($elem.data)", "PartitionKey": "$elem.partitionKey"}#if($foreach.hasNext),#end #end ] }`;
    }
    buildPutRecordsModel(putRecordsModel) {
        let modelProps;
        if (putRecordsModel !== undefined) {
            modelProps = putRecordsModel;
        }
        else {
            modelProps = {
                contentType: 'application/json',
                modelName: 'PutRecordsModel',
                description: 'PutRecords proxy payload data',
                schema: {
                    schema: api.JsonSchemaVersion.DRAFT4,
                    title: 'PutRecords proxy payload data',
                    type: api.JsonSchemaType.OBJECT,
                    required: ['records'],
                    properties: {
                        records: {
                            type: api.JsonSchemaType.ARRAY,
                            items: {
                                type: api.JsonSchemaType.OBJECT,
                                required: ['data', 'partitionKey'],
                                properties: {
                                    data: { type: api.JsonSchemaType.STRING },
                                    partitionKey: { type: api.JsonSchemaType.STRING }
                                }
                            }
                        }
                    }
                }
            };
        }
        return this.apiGateway.addModel('PutRecordsModel', modelProps);
    }
}
exports.ApiGatewayToKinesisStreams = ApiGatewayToKinesisStreams;
_a = JSII_RTTI_SYMBOL_1;
ApiGatewayToKinesisStreams[_a] = { fqn: "@aws-solutions-constructs/aws-apigateway-kinesisstreams.ApiGatewayToKinesisStreams", version: "2.34.0" };
//# sourceMappingURL=data:application/json;base64,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