# AUTOGENERATED! DO NOT EDIT! File to edit: notebooks/retraitement_erfs-fpr/modules/calibration_01_dev_cal.ipynb (unless otherwise specified).

__all__ = [
    "config",
    "tc",
    "seuil_chomage",
    "get_minimal_frontiers",
    "nb_zero",
    "DF_quantiles",
    "Bucket_Base",
    "Bucket_Pote",
    "distrib_to_quantiles",
    "generate_title",
    "compare_distributions",
    "ajout_gens_en_haut",
    "create_ppl_HR",
    "save_made_up_people",
    "Tracker",
    "merge_and_replace",
    "prepare_for_calib",
    "compute_min_max_calib",
    "data_avant_calib",
    "recalage_calib",
    "init_calib_zero",
    "create_ppl_zero",
    "variables_independantes",
    "calib_empty_bucket",
    "calib_bucket_lim",
    "optimization_quantile",
    "amelioration_quantile",
    "calibration_quantiles",
    "calibration",
    "dico",
]

# Cell
import copy
import json
import random

# export
import unittest

import matplotlib.pyplot as plt
import pandas as pd
import seaborn as sns
from leximpact_socio_fisca_simu_etat.config import Configuration

from .calib_and_copules import bucket_merge_with_above, reduce_bucket_number
from .toolbase import foyers_fiscaux_to_individus, individus_to_foyers_fiscaux

config = Configuration(project_folder="leximpact-prepare-data")
tc = unittest.TestCase()

# Cell
""" Supprime les buckets de calib qui sont au-dessus du plafond d'allocations chomage """
# Sources
# https://www.data.gouv.fr/fr/datasets/montant-dallocation-chomage-et-salaires-de-reference-des-allocataires-de-lassurance-chomage/


def seuil_chomage(calib_in):
    m = len(calib_in["buckets"])
    plafond_annuel = 365 * 241.22  # 2021
    calib_out = copy.deepcopy(calib_in)

    # On cherche les buckets max
    for idx, bucket in enumerate(calib_in["buckets"]):
        if bucket["seuil_var_inf"] >= plafond_annuel:
            print("Seuil inf", bucket["seuil_var_inf"])
            idmax = idx
            break
    # On supprime les buckets supérieurs
    calib_out["buckets"] = calib_out["buckets"][:idmax]

    # Gens supprimés
    suppr = calib_in["buckets"][idmax:]
    print(suppr)
    print(idmax)
    total = 0
    for i in range(len(suppr)):
        total += suppr[i]["nombre_ff_tranche"]

    # On ajuste la frontière max
    calib_out["buckets"][-1]["seuil_var_supp"] = plafond_annuel
    print(
        "On a supprimé les",
        m - len(calib_out["buckets"]),
        " buckets au-dessus du plafond de chomage :",
        plafond_annuel,
        " €, soit ",
        total,
        " personnes",
    )
    return calib_out


# Cell
""" Obtention des frontières minimales (non jointives) de la distribution issue de POTE """


def get_minimal_frontiers(calib, erfs, var_name):

    # 0 - Pour les frontières entre 0 et 1: pour le cas des revenus du capital, il faut les distinguer
    # On initialise le 2e bucket à la premiere valeur > 0
    for i in range(len(calib["buckets"])):
        if calib["buckets"][i]["seuil_var_inf"] > 0:
            frontiere_supp_initiale = calib["buckets"][i]["seuil_var_inf"]
            break
    # Attention ! Dans les fichiers calib le seuil_supp du bucket zero n'est pas le seuil_inf du bucket 1

    # 1 - On fusionne les buckets qui ont les mêmes frontières
    last = False
    while last is False:
        buckets = calib["buckets"]
        for idx, bucket in enumerate(buckets):
            # On s'arrête 2 buckets avant la fin
            if idx + 2 == len(buckets):
                last = True
                break
            # Si les buckets contiennent les mêmes frontières
            elif bucket["seuil_var_inf"] == calib["buckets"][idx + 1]["seuil_var_inf"]:
                calib = bucket_merge_with_above(calib, idx)
                break

    # 2 - On initialise le 1er bucket de gens nuls (pour réparer le fichier calib)
    frontieres_var = [0, frontiere_supp_initiale]
    calib["buckets"][0]["seuil_var_supp"] = frontiere_supp_initiale
    calib["buckets"][1]["seuil_var_inf"] = frontiere_supp_initiale

    # 3 - On recupere toutes les frontieres
    for i in range(1, len(calib["buckets"]) - 1):
        frontieres_var.append(calib["buckets"][i]["seuil_var_supp"])

    # 4 - On augmente la hauteur du seuil max (cas où max(ERFS) > max(POTE))
    big_max = (
        max(erfs[var_name].max(), calib["buckets"][-1]["seuil_var_supp"]) + 1
    )  # Pour garder la valeur max dans le bucket
    frontieres_var.append(big_max)

    tc.assertEqual(len(frontieres_var) - 1, len(calib["buckets"]))
    tc.assertEqual(len(frontieres_var), len(set(frontieres_var)))

    print(
        "On étudie la distribution en ",
        len(frontieres_var) - 1,
        " buckets, avec un min de ",
        frontieres_var[0],
        " et un max de ",
        frontieres_var[-1],
        "€ de",
        var_name,
    )

    return calib, frontieres_var


# Cell
""" Calcule le nombre de gens à zéro dans POTE et dans l'ERFS avant calibration """


def nb_zero(erfs_ff, var_name, calib):

    nb_zero_erfs = erfs_ff[erfs_ff[var_name] < calib["buckets"][1]["seuil_var_inf"]][
        "wprm"
    ].sum()
    nb_zero_pote = calib["nb_foyer"]["zero"]

    print(
        "Nombre de foyers de ",
        var_name,
        "à zéro, dans l'ERFS : ",
        nb_zero_erfs,
        "et dans POTE : ",
        nb_zero_pote,
        "soit un écart de : ",
        100 * (nb_zero_erfs - nb_zero_pote) / nb_zero_pote,
        "%",
    )

    return nb_zero_erfs, nb_zero_pote


# Cell
""" DataFrame qui contient la distribution des données en buckets """


class DF_quantiles:
    def __init__(self, frontiers):
        self.columns = ["seuil_inf", "seuil_max", "middle", "nb_ff", "sum", "mean"]
        self.index = [str(i) for i in range(0, len(frontiers) - 1)]
        self.df = pd.DataFrame(columns=self.columns, index=self.index)
        self.bucket_list = {key: None for key in self.index}

    def update_quantile(self, i, bucket):
        self.df["seuil_inf"][i] = bucket.seuil_inf
        self.df["seuil_max"][i] = bucket.seuil_max
        self.df["middle"][i] = bucket.middle
        self.df["nb_ff"][i] = bucket.nb_ff
        self.df["sum"][i] = bucket.sum_
        self.df["mean"][i] = bucket.mean_
        # On sauvegarde l'ensemble des buckets
        self.bucket_list[str(i)] = bucket

        return self


# Cell
""" Objet qui contient toutes les informations d'un bucket """


class Bucket_Base:
    def __init__(self, i, frontiers, erfs_ff, var_name):
        self.nb = i
        self.seuil_inf = frontiers[self.nb]
        self.seuil_max = frontiers[self.nb + 1]
        self.middle = self.seuil_inf + (self.seuil_max - self.seuil_inf) / 2
        # Echantillon du bucket
        self.sample = erfs_ff.loc[
            (erfs_ff[var_name] >= self.seuil_inf) & (erfs_ff[var_name] < self.seuil_max)
        ].copy(deep=True)
        self.sample_pondere = (self.sample[var_name] * self.sample["wprm"]).copy(
            deep=True
        )
        # Infos
        self.nb_ff = round(self.sample["wprm"].sum())
        self.sum_ = self.sample_pondere.sum()
        self.mean_ = self.sample_pondere.mean()


class Bucket_Pote:
    def __init__(self, i, frontiers, calib):
        self.nb = i
        self.seuil_inf = frontiers[self.nb]
        self.seuil_max = frontiers[self.nb + 1]
        self.middle = self.seuil_inf + (self.seuil_max - self.seuil_inf) / 2
        # Infos
        self.nb_ff = calib["buckets"][self.nb]["nombre_ff_tranche"]
        self.sum_ = calib["buckets"][self.nb]["sum_tranche_var"]
        self.mean_ = calib["buckets"][self.nb]["mean_tranche_var"]


# Cell
""" Génère la distribution en buckets de la base ERFS, calée sur POTE """


def distrib_to_quantiles(erfs_ff, var_name, calib):

    # I - Nombre de foyers à zéro
    nb_zero_erfs, nb_zero_pote = nb_zero(erfs_ff, var_name, calib)

    # II - Obtention des frontières de distribution
    calib, frontieres_var = get_minimal_frontiers(calib, erfs_ff, var_name)

    # III - On passe à zéro tous les gens du 1er quantile
    erfs_ff.loc[
        erfs_ff[var_name] < calib["buckets"][1]["seuil_var_inf"], var_name
    ] = 0.0
    sample_zero = erfs_ff[erfs_ff[var_name] < calib["buckets"][1]["seuil_var_inf"]]
    print(
        "Dans l'ERFS, on a ", sample_zero["wprm"].sum(), " foyers de ", var_name, " = 0"
    )
    # print(frontieres_var)

    # III - Création d'objets pour enregistrer les quantiles
    Distrib_BASE = DF_quantiles(frontieres_var)
    Distrib_POTE = DF_quantiles(frontieres_var)

    # IV - Création des quantiles
    for i in range(0, len(frontieres_var) - 1):
        # Distribution de l'ERFS
        bucket_erfs = Bucket_Base(i, frontieres_var, erfs_ff, var_name)
        Distrib_BASE = Distrib_BASE.update_quantile(i, bucket_erfs)
        # Distribution de POTE
        bucket_pote = Bucket_Pote(i, frontieres_var, calib)
        Distrib_POTE = Distrib_POTE.update_quantile(i, bucket_pote)

    # IV - Tests
    df_erfs = Distrib_BASE.df
    df_pote = Distrib_POTE.df
    # Vérification des frontières
    tc.assertAlmostEqual(df_erfs["seuil_inf"].sum(), df_pote["seuil_inf"].sum())
    tc.assertAlmostEqual(df_erfs["seuil_max"].sum(), df_pote["seuil_max"].sum())
    # Vérification que les frontières de buckets sont bien distinctes
    tc.assertAlmostEqual(df_erfs["seuil_inf"].nunique(), len(df_erfs["seuil_inf"]))
    # Vérification du nombre total de foyers fiscaux
    tc.assertAlmostEqual(df_erfs["nb_ff"].sum() / erfs_ff["wprm"].sum(), 1, places=0)
    # Vérification de la somme totale
    tc.assertAlmostEqual(
        df_erfs["sum"].sum() / (erfs_ff[var_name] * erfs_ff["wprm"]).sum(), 1, places=3
    )  # Pour avoir plus de marge à cause des arrondis de wprm

    return Distrib_BASE, Distrib_POTE, calib


# Cell
""" Génère un titre adapté pour les différents plots issus de la calibration """


def generate_title(
    var_name, annee_erfs, annee_pote, title_suffix, log=False, cal=False
):
    # Avec ou sans calibration
    if cal is False:
        title = (
            "Comparaison des distributions de "
            + var_name
            + "\n ERFS "
            + annee_erfs
            + " (en rouge) et  POTE  "
            + annee_pote
            + " (en bleu) "
        )
    else:
        title = (
            "Calibration de la distribution de "
            + var_name
            + "\n ERFS "
            + annee_erfs
            + " (en rouge), calibration (en vert) et  POTE  "
            + annee_pote
            + " (en bleu) "
        )

    # Details
    title = title + title_suffix

    return title


# Cell
""" Plot et compare les distributions de l'ERFS et de POTE """


def compare_distributions(
    df_erfs, df_pote, var_name, annee_erfs, annee_pote, log, title_suffix, df_cal=[]
):

    # Pour traiter le cas des buckets vides, on remplace les NaN par zéro
    df_pote["mean"] = df_pote["mean"].fillna(0)
    df_erfs["mean"] = df_erfs["mean"].fillna(0)
    if len(df_cal) != 0:
        df_cal["mean"] = df_cal["mean"].fillna(0)

    # Création d'une figure
    fig = plt.figure(figsize=(20, 8), facecolor="white", clear=True)
    ax = sns.barplot(data=df_pote, y="sum", x=df_pote.index, alpha=0.5, color="blue")
    ax = sns.barplot(data=df_erfs, y="sum", x=df_erfs.index, alpha=0.5, color="red")
    outname = (
        var_name + "_ERFS_" + annee_erfs + "_POTE_" + annee_pote + "_" + title_suffix
    )

    # S'il y a une calibration
    if len(df_cal) != 0:
        ax = sns.barplot(data=df_cal, y="sum", x=df_cal.index, alpha=0.5, color="green")
        title = generate_title(var_name, annee_erfs, annee_pote, title_suffix, cal=True)
        outname = "Distributions_de_" + outname
        df_base = df_cal.copy()

    else:
        title = generate_title(
            var_name, annee_erfs, annee_pote, title_suffix, cal=False
        )
        outname = "Calibration_de_" + outname
        df_base = df_erfs.copy()

    # Si échelle logarithmique
    if log:
        _ = ax.set_yscale("log")
        _ = ax.set_title(
            (title + "\n Echelle de population logarithmique"), fontsize=18
        )
    else:
        _ = ax.set_title(title, fontsize=18)

    # Axis Setup
    xticks = [i for i in range(len(df_pote["middle"]))]
    xlabels = [f"{str(round(j / (10 ** 3), 2))} k€" for j in df_pote["mean"]]
    _ = ax.set_xticks(xticks)
    _ = ax.set_xticklabels(xlabels, rotation=75)
    _ = ax.set_xlabel(
        " ' " + var_name + " ' " + " moyen pour chaque quantile (POTE)", fontsize=16
    )
    _ = ax.set_ylabel("Somme de " + var_name + " dans chaque quantile", fontsize=16)

    # Plotting the sum on top of the bars
    # ax.margins(y=0.1)
    # for bars in ax.containers:
    #    ax.bar_label(bars, fmt='%.1f')

    # Saving the figure
    figpath = config.get("PLOTS")
    plt.savefig((figpath + outname), bbox_inches="tight")

    # On calcule l'erreur comme la moyenne des erreurs de chaque bucket
    error_df = 100 * pd.Series(abs(df_pote["sum"] - df_base["sum"]) / df_pote["sum"])
    # Pour le premier bucket, les sommes sont à zéro donc on mesure l'erreur en Nb de foyers
    error_df[0] = (
        100 * abs(df_pote["nb_ff"][0] - df_base["nb_ff"][0]) / df_pote["nb_ff"][0]
    )
    # On exclut les quantiles nuls du calcul d'erreur
    final_error = (error_df.sum()) / (len(df_base))
    print("Erreur moyenne des buckets de ", var_name, " : ", final_error, " %")
    print("Erreur min : ", min(error_df), "erreur max: ", max(error_df))

    return fig, error_df, final_error


# Cell

"""
    Ajout de foyers dans le haut de la distribution
"""


def ajout_gens_en_haut(erfs, erfs_ind, var_name, Distrib_POTE, Distrib_ERFS):

    buckets = Distrib_ERFS.bucket_list
    list_missing = []
    # On parcourt les buckets
    for i in range(len(buckets)):
        idx = buckets[str(i)].nb
        # Si on a un bucket vide alors que la calib n'est pas vide, et qu'on n'est pas dans le 1er bucket
        if (
            (buckets[str(i)].nb_ff == 0)
            and (Distrib_POTE.bucket_list[str(i)].nb != 0)
            and (idx != 0)
        ):
            list_missing.append(idx)

    # Si on a des buckets qui sont vides:
    if len(list_missing) > 0:
        erfsn, erfs_indn = create_ppl_HR(
            erfs_ind, var_name, list_missing, Distrib_POTE, Distrib_ERFS
        )
        tc.assertGreater(len(erfsn), len(erfs))
    else:
        erfsn = erfs.copy(deep=True)
        erfs_indn = erfs_ind.copy(deep=True)

    tc.assertEqual(erfsn["fake_id"].isna().sum(), 0)

    return erfsn, erfs_indn


# Cell
"""
    Création de nouvelles personnes dans la base
    On notera qu'on ajoute uniquement des déclarants principaux, donc ici foyer=gens
"""


def create_ppl_HR(erfs_ind, var_name, list_missing, Distrib_POTE, Distrib_ERFS):

    # On décide de sélectionner des profils "proches", donc dans les 2 buckets précédant le 1er bucket vide
    id_min = min(list_missing) - 3
    id_max = min(list_missing) - 1
    echantillon_ou_piocher = erfs_ind.loc[
        (erfs_ind[var_name] >= Distrib_POTE.bucket_list[str(id_min)].seuil_inf)
        & (erfs_ind[var_name] < Distrib_POTE.bucket_list[str(id_max)].seuil_max)
    ].copy(deep=True)
    # On ne prend que les declarants principaux
    echantillon_ou_piocher = echantillon_ou_piocher[
        echantillon_ou_piocher["quifoy"] == 0
    ]

    # On pioche pour chaque bucket le nombre de personnes manquantes
    for id_x in list_missing:
        # 1 - Choix du nombre de foyers que l'on doit ajouter
        Nb_wanted = Distrib_POTE.bucket_list[str(id_x)].nb_ff
        seuil_inf = Distrib_POTE.bucket_list[str(id_x)].seuil_inf
        seuil_max = Distrib_POTE.bucket_list[str(id_x)].seuil_max

        # Si c'est trop, on en prend au max 0.5% de la population
        if Nb_wanted > 1000:
            Nb = int(round(len(erfs_ind) / 200, 0))
            print("inf 1000", Nb)
        else:
            Nb = int(Nb_wanted)
            print("Else", Nb)

        # 2 - Sélection au hasard de gens
        print(
            "On ajoute ",
            Nb,
            " personnes dans le bucket ",
            id_x,
            "entre ",
            seuil_inf,
            " et ",
            seuil_max,
        )
        # Si on peut, on essaye de ne pas prendre plusieurs fois la meme personne
        if Nb >= len(echantillon_ou_piocher):
            new_people = echantillon_ou_piocher.sample(n=Nb, replace=True).reset_index(
                drop=True
            )
        else:
            new_people = echantillon_ou_piocher.sample(n=Nb, replace=False).reset_index(
                drop=True
            )
            # print("echantillon \n", echantillon_ou_piocher)
        # TODO : on fixe le random-state pour plus de stabilité?

        # 3 - On calibre leurs poids
        rate = Nb_wanted / (new_people["wprm"].sum())
        new_people.loc[:, "wprm"] = new_people["wprm"].apply(lambda row: row * rate)
        tc.assertAlmostEqual(new_people["wprm"].sum(), Nb_wanted)

        # 4 - On calibre leur variable d'intéret
        sum_wanted = Distrib_POTE.bucket_list[str(id_x)].sum_
        # 4.1 - On leur attribue une valeur comprise dans le bucket (+/- 5% pour avoir une marge)
        new_people.loc[:, var_name] = [
            random.randint(
                round(seuil_inf, 0) + 10, round(seuil_max, 0) - 10
            )  # round(1.05*seuil_inf), round(0.95*seuil_max))
            for ppl in range(len(new_people))
        ]
        sum_ponderee = (new_people[var_name] * new_people["wprm"]).sum()

        # 4.2 - On recalibre
        error = 1
        while error > 0.001:
            coef = sum_wanted / sum_ponderee
            # On update les valeurs
            new_people.loc[:, var_name] = new_people[var_name].apply(
                lambda row: max(min((row * coef), seuil_max - 1), seuil_inf + 1)
            )
            sum_ponderee = (new_people[var_name] * new_people["wprm"]).sum()
            error = (sum_ponderee - sum_wanted) / sum_wanted

        tc.assertAlmostEqual(error, 0, places=0)

        # Mise en forme et intégration dans les bases
        erfs_ind = save_made_up_people(erfs_ind, new_people)
        erfs = individus_to_foyers_fiscaux(erfs_ind)

    return erfs, erfs_ind


# Cell
""" Mise en forme des gens créés dans les bases en foyers ET en individus """


def save_made_up_people(erfs, new_people):

    new_people = new_people.copy().reset_index(drop=True)

    # Mise en forme de nos 'nouveaux foyers'
    # 1 - Fake_id
    new_people.loc[:, "fake_id"] = 1.0
    tc.assertEqual(new_people["fake_id"].isna().sum(), 0)

    # 2 - Idfoy, Idfam, Idmen
    max_idfoy = max(erfs["idfoy"]) + 1  # S'update à chaque ajout de bucket

    dict_ = {}
    for i in range(len(new_people)):
        dict_[new_people.loc[i, "idfoy"]] = max_idfoy + i

        new_people.loc[i, "idfoy"] = max_idfoy + i
        new_people.loc[i, "idfam"] = max_idfoy + i
        new_people.loc[i, "idmen"] = max_idfoy + i

    tc.assertEqual(new_people.isna().sum().sum(), 0)

    # 3 - On ajoute ces gens dans notre base en individus
    # new_people.drop(['index'], axis=1)
    erfs_av = erfs.copy()  # Pour le test
    erfs_ind = pd.concat([erfs, new_people])
    erfs_ind = erfs_ind.reset_index(drop=True)
    tc.assertEqual(len(erfs_ind), len(erfs_av) + len(new_people))

    return erfs_ind


# Cell
""" Objet permettant de sauvegarder les informations propres au process de calibration """


class Tracker:
    def __init__(self, frontiers):
        self.columns = [
            "error_type",
            "wanted",
            "sum_av",
            "sum_ap",
            "max_possible",
            "min_possible",
            "var_max",
            "var_min",
            "final_error",
        ]
        self.index = [str(i) for i in range(0, len(frontiers) - 1)]
        self.df = pd.DataFrame(columns=self.columns, index=self.index)

    def track(self, i, key, value):
        self.df[key].iloc[i] = value
        return self.df


# Cell
""" Fonction qui merge un échantillon dans la base (sur idfoy) et remplace les valeurs par celles de l'échantillon """


def merge_and_replace(erfss, sample_to_merge, var_list):

    # On utilise les 'idfoy' comme index sans perdre la colonne
    erfssn = erfss.copy(deep=True)
    erfssn.set_index("idfoy", drop=False, inplace=True)

    # On utilise les 'idfoy' comme index (inutile de garder la colonne)
    sample_to_merge = sample_to_merge[["idfoy"] + var_list]
    sample_to_merge.set_index("idfoy", drop=True, inplace=True)

    # Le update se fait sur les 2 index (i.e. les 'idfoy')
    erfssn.update(sample_to_merge, join="left", overwrite=True)

    # Check : pas d'ajout de foyers
    tc.assertEqual(len(erfssn), len(erfss))
    # On reset les index
    erfssn.reset_index(inplace=True, drop=True)
    return erfssn


# Cell


def prepare_for_calib(erfs, var_name):
    # On trie l'ERFS par RFR croissant
    erfss = erfs.sort_values(by=var_name).copy(deep=True).reset_index(drop=True)

    # On vérifie qu'on a une liste unique de 'idfoy',
    # qu'on va utiliser comme index du dataframe
    erfss["index"] = erfss["idfoy"].copy()
    tc.assertEqual(erfss["idfoy"].nunique(), len(erfss["idfoy"]))
    erfss = erfss.set_index("index")

    return erfss


# Cell
"""
Calcul des sommes max et min possibles
"""


def compute_min_max_calib(var_name, bucket):

    sample = bucket.sample
    # On distingue les foyers fake des vrais
    sample_true = sample.loc[sample["fake_id"] == 0].copy()
    sample_fake = sample.loc[sample["fake_id"] == 1].copy()
    not_fake_sum = (sample_true[var_name] * sample_true["wprm"]).sum()

    # On calcule la somme pondérée max qu'on peut générer si on updatait tous les fake
    # Attention, ici i1 faut un certain écart (>= 1) sinon les arrondis de calcul font passer certains au-dessus du seuil
    var_max = bucket.seuil_max - 1
    max_possible = not_fake_sum + (var_max * sample_fake["wprm"]).sum()
    # print("Sum max possible ponderee: ", max_possible)

    # On calcule la somme pondérée min qu'on peut générer si on updatait tous les fake
    # Pour rester supérieur ou égal au seuil_inf
    var_min = bucket.seuil_inf
    min_possible = not_fake_sum + (var_min * sample_fake["wprm"]).sum()
    # print("Sum min possible ponderee: ", min_possible)

    tc.assertGreaterEqual(var_max, var_min)
    # C'est égal si tout le monde a le même revenu

    return min_possible, max_possible, var_min, var_max


# Cell


def data_avant_calib(bucket, tracker, var_name, wanted):

    # Etat des agrégats de somme
    tracker.track(bucket.nb, "wanted", wanted)
    tracker.track(bucket.nb, "sum_av", bucket.sum_)
    tc.assertAlmostEqual(
        (bucket.sample[var_name] * bucket.sample["wprm"]).sum(), bucket.sum_
    )
    error = abs(bucket.sum_ - wanted) / wanted
    print(
        "wanted pondérée: ",
        wanted,
        "somme en cours pondérée:",
        bucket.sum_,
        "erreur avant calibration : ",
        round(100 * error, 2),
    )

    # On regarde ce qui est possible
    min_possible, max_possible, var_min, var_max = compute_min_max_calib(
        var_name, bucket
    )
    tracker.track(bucket.nb, "max_possible", max_possible)
    tracker.track(bucket.nb, "min_possible", min_possible)
    tracker.track(bucket.nb, "var_max", var_max)
    tracker.track(bucket.nb, "var_min", var_min)

    return tracker, error


# Cell

""" Cette fonction n'est pas utilisée mais permet de recaler chaque décile sur la somme expected de POTE
On ne l'utilise pas car le coef fait sortir les foyers fiscaux de leurs buckets respectifs """


def recalage_calib(erfs_cal, var_name, sum_wanted):

    sum_av = (erfs_cal[var_name] * erfs_cal["wprm"]).sum()
    erfs_cal[var_name] = erfs_cal[var_name] * sum_wanted / sum_av

    print(
        "On recale le "
        + var_name
        + " sur POTE. Aggrégat utilisé : "
        + f"{ sum_wanted :,}"
    )

    tc.assertEqual(sum_wanted, (erfs_cal[var_name] * erfs_cal["wprm"]).sum())

    return erfs_cal


# Cell

"""
On va créer artificiellement des gens dans le bucket var = 0
en utilisant une version adaptée de la fonction create_ppl_HR (cf. plus bas)
"""

# Initialisation du premier bucket
def init_calib_zero(var_name, erfs_ind, bucket_erfs, bucket_pote, tracker):

    frontier_min = bucket_pote.seuil_max
    tc.assertEqual(frontier_min, bucket_erfs.seuil_max)

    # On les met à zéro tout pile dans la base individus
    erfs_ind.loc[erfs_ind[var_name] < frontier_min, "var_cal"] = 0.0
    sample_zero_ind = erfs_ind[erfs_ind[var_name] < frontier_min]
    sample_zero_ff = bucket_erfs.sample
    print(
        "Calibration du 1er bucket \n",
        "Dans l'ERFS, on a ",
        sample_zero_ff["wprm"].sum(),
        " foyers, soit ",
        sample_zero_ind["wprm"].sum(),
        " individus de ",
        var_name,
        " = 0",
    )

    # Infos sur les foyers à zéro
    nb_zero_pote = bucket_pote.nb_ff
    nb_zero = bucket_erfs.nb_ff
    tc.assertEqual(nb_zero, round((sample_zero_ff["wprm"]).sum()))
    print("Poids total des foyers à zéro AVANT : ", nb_zero, "vs POTE", nb_zero_pote)

    # On crée de nouvelles personnes s'il en manque
    if nb_zero_pote > nb_zero:
        max_idfoy = max(erfs_ind["idfoy"]) + 1
        new_people = create_ppl_zero(sample_zero_ind, nb_zero_pote, nb_zero, max_idfoy)

        # On ajoute les nouveaux dans la base
        erfs_cal_ind = pd.concat([erfs_ind, new_people])
        tc.assertEqual(len(erfs_cal_ind), len(erfs_ind) + len(new_people))
        # On vérifie qu'on a bien une ligne par foyer
        tc.assertEqual(
            erfs_ind["idfoy"].nunique(), len(individus_to_foyers_fiscaux(erfs_ind))
        )
    else:
        erfs_cal_ind = erfs_ind.copy()
    # TODO: et s'il y a TROP de personnes à zéro ?

    # Recalcul final
    sample_zero_cal = erfs_cal_ind[erfs_cal_ind["var_cal"] < frontier_min]
    nb_zero_cal = (sample_zero_cal["wprm"]).sum()

    # Trackers de la calibration
    error_zero = (nb_zero_cal - nb_zero_pote) / nb_zero_pote
    tracker.track(0, "error_type", f"Erreur Nb gens à zéro :{error_zero}")
    tracker.track(0, "sum_av", 0)
    tracker.track(0, "sum_ap", 0)
    tracker.track(0, "max_possible", 0)
    tracker.track(0, "min_possible", 0)
    tracker.track(0, "wanted", bucket_pote.sum_)
    tracker.track(0, "final_error", error_zero)

    return tracker, erfs_cal_ind


# Cell

variables_independantes = [
    "chomage_brut",
    "pensions_alimentaires_percues",
    "rag",
    "ric",
    "rnc",
    "salaire_de_base",
    "f4ba",
    "retraite_brute",
    "rfr",
    "revenus_capitaux_prelevement_bareme",
    "revenus_capitaux_prelevement_liberatoire",
    "revenus_capitaux_prelevement_forfaitaire_unique_ir",
    "salaire_imposable",
]


def create_ppl_zero(sample_zero, Nb_wanted, Nb_actuel, max_idfoy):
    """
    Ajout de foyers de rfr nul
    On notera qu'on ajoute uniquement des déclarants principaux, donc ici foyer=gens
    NB: attention ici on ajoute pas mal de monde dans la population (mais ce sera régulé dans l'étape d'inflation_ff)
    """
    # On décide de sélectionner des profils "proches", donc parmi ceux déjà à zéro
    echantillon_ou_piocher = sample_zero.copy(deep=True)
    # On ne prend que des declarants principaux
    echantillon_ou_piocher = echantillon_ou_piocher[
        echantillon_ou_piocher["quifoy"] == 0
    ]

    # 1 - Choix du nombre de foyers que l'on ajoute
    # Pas plus de 1000 ou 10_000 ?
    if Nb_wanted > 1000:
        Nb = 1000
    elif Nb_wanted > len(sample_zero):
        Nb = len(sample_zero)
    else:
        Nb = Nb_wanted

    # 2 - Sélection au hasard de gens
    new_people = echantillon_ou_piocher.sample(n=Nb).reset_index(drop=True)
    # On les met à zéro tout pile
    new_people.loc[:, "var_cal"] = 0.0

    # TODO : on fixe le random-state pour plus de stabilité?

    # 3 - Mise en forme de nos 'nouveaux foyers'
    # 3.1 - On indique qu'ils sont fake
    new_people["fake_id"] = 1
    # 3.2 - On ajuste le 'idfoy' pour ne pas avoir de doublons
    for i in range(Nb):
        new_people.loc[i, "idfoy"] = max_idfoy + i
        new_people.loc[i, "idfam"] = max_idfoy + i
        new_people.loc[i, "idmen"] = max_idfoy + i

    # 4 - On calibre leurs poids
    rate = (Nb_wanted - Nb_actuel) / (new_people["wprm"].sum())
    new_people["wprm"] = rate * new_people["wprm"]
    tc.assertAlmostEqual(new_people["wprm"].sum() + Nb_actuel, Nb_wanted)

    return new_people


# Cell


def calib_empty_bucket(i, tracker):
    print("On n'a personne à calibrer dans le bucket", i, "\n")
    # Trackers de la calibration
    tracker.track(
        i, "error_type", "empty"
    )  # Arbitraire, pour répérer dans quel cas on est
    tracker.track(i, "sum_av", 0)
    tracker.track(i, "sum_ap", 0)
    tracker.track(i, "max_possible", 0)
    tracker.track(i, "min_possible", 0)
    tracker.track(i, "final_error", 1)

    return tracker


# Cell


def calib_bucket_lim(erfs, i, sample_fake, not_fake_sum, tracker, limit):
    # On update toutes les valeurs des fake
    sample_fake["var_cal"] = sample_fake["var_cal"].apply(lambda row: limit)
    # Réintégration dans la base
    erfss = merge_and_replace(erfs, sample_fake, ["var_cal"])
    # Tracker
    tracker.track(i, "error_type", "lim")  # Arbitraire
    tracker.track(
        i, "sum_ap", not_fake_sum + (sample_fake["var_cal"] * sample_fake["wprm"]).sum()
    )
    tracker.track(
        i,
        "final_error",
        (tracker.df["sum_ap"][i] - tracker.df["wanted"][i]) / tracker.df["wanted"][i],
    )

    return erfss, tracker


# Cell


def optimization_quantile(
    erfss, i, tracker, sample_fake, not_fake_sum, error_in, var_name, err_max
):

    # On calibre une fois et on update la base
    improvement, current_sum, error, sample_fake = amelioration_quantile(
        i, tracker, sample_fake, not_fake_sum, error_in, var_name
    )
    # print("Recalcul de l'erreur du bucket après calib précise", error)
    erfss = merge_and_replace(erfss, sample_fake, ["var_cal"])

    # Si ça ne suffit pas on recommence sur les valeurs qu'on vient de calibrer
    while (abs(error) > err_max) and (improvement is True):
        print("On calibre à nouveau")
        improvement, current_sum, error, sample_fake = amelioration_quantile(
            i, tracker, sample_fake, not_fake_sum, error, "var_cal"
        )
        # Une fois qu'on a la meilleure solution, on remplace
        erfss = merge_and_replace(erfss, sample_fake, ["var_cal"])

    # Trackers de la calibration
    tracker.track(i, "error_type", "optimization")
    tracker.track(i, "sum_ap", current_sum)
    tracker.track(
        i,
        "final_error",
        (current_sum - tracker.df["wanted"][i]) / tracker.df["wanted"][i],
    )

    # print('final error opt', tracker.df["final_error"][i])

    return erfss, tracker, sample_fake


# Cell


def amelioration_quantile(
    i, tracker, sample_fake_in, not_fake_sum, error_in, changing_var_name
):

    # Initialisation
    improvement = True
    fake_sum_en_cours = (
        sample_fake_in[changing_var_name] * sample_fake_in["wprm"]
    ).sum()
    sample_fake_en_cours = sample_fake_in.copy(deep=True)

    # Le coefficient multiplicateur pour obtenir la sum wanted est
    coef = (tracker.df["wanted"][i] - not_fake_sum) / fake_sum_en_cours

    # On update les valeurs
    sample_fake_en_cours.loc[:, "var_cal"] = sample_fake_en_cours[
        changing_var_name
    ].apply(
        lambda row: max(
            min((row * coef), tracker.df["var_max"][i]), tracker.df["var_min"][i]
        )
    )
    current_sum = (
        not_fake_sum
        + (sample_fake_en_cours["var_cal"] * sample_fake_en_cours["wprm"]).sum()
    )
    # print(current_sum)
    error = (current_sum - tracker.df["wanted"][i]) / tracker.df["wanted"][i]

    # Si on stagne ou si ça empire la calib, on garde l'état précédent et on s'arrête:
    if abs(error) >= abs(error_in):
        improvement = False
        return improvement, (not_fake_sum + fake_sum_en_cours), error_in, sample_fake_in

    return improvement, current_sum, error, sample_fake_en_cours


# Cell
"""
Script global de calibration
"""


def calibration_quantiles(var_name, erfs_ind, Distrib_ERFS, Distrib_POTE, err_max):

    # Liste des frontières des quantiles
    frontieres = list(Distrib_ERFS.df["seuil_inf"])
    frontieres.append(list(Distrib_ERFS.df["seuil_max"])[-1])
    tc.assertEqual(len(frontieres) - 1, len(Distrib_POTE.df["seuil_inf"]))

    # On crée un tracker qui va nous permettre de checker la calibration
    tracker = Tracker(frontieres)

    # On crée une colonne où l'on mettra les variables calibrées :
    # (le 0.1 permettra de retrouver ceux qui n'ont pas été touchés)
    erfs_ind["var_cal"] = 0.1

    # Calibration du bucket de gens à zéro
    tracker, erfs_cal_ind = init_calib_zero(
        var_name,
        erfs_ind,
        Distrib_ERFS.bucket_list["0"],
        Distrib_POTE.bucket_list["0"],
        tracker,
    )

    # On prépare la base pour la calibration
    erfs_ff = individus_to_foyers_fiscaux(erfs_cal_ind)
    print(
        "Nb gens à zéro post-calib : ",
        erfs_ff[erfs_ff["var_cal"] < frontieres[1]]["wprm"].sum(),
    )
    erfss = prepare_for_calib(erfs_ff, var_name)
    # Itération : on parcourt les buckets
    for i in range(1, len(frontieres) - 1):
        print("\n \n", "bucket", i)
        bucket = Distrib_ERFS.bucket_list[str(i)]
        # On considère le sous-échantillon minimal des foyers de ce bucket
        sample = bucket.sample[["idfoy", "fake_id", "wprm", var_name]]

        # Analyse du bucket avant calib
        tracker, error = data_avant_calib(
            bucket, tracker, var_name, Distrib_POTE.df["sum"].iloc[bucket.nb]
        )

        # Cas 1 : le bucket est vide
        if len(sample) == 0:
            tracker = calib_empty_bucket(i, tracker)
            print("On a calibré", i, "erreur finale du bucket : ", 1)

        # Cas 2 : un bucket non vide
        else:
            sample.loc[:, "var_cal"] = 0.1
            # On distingue les foyers fake des vrais
            sample_true = sample.loc[sample["fake_id"] == 0].copy(deep=True)
            sample_fake = sample.loc[sample["fake_id"] == 1].copy(deep=True)
            not_fake_sum = (sample_true[var_name] * sample_true["wprm"]).sum()
            # On vérifie qu'on n'a pas de bug sur les fake_id
            tc.assertEqual(sample["fake_id"].max(), 1)

            # Cas 2.1 : inutile de calibrer
            if error <= err_max:
                sample.loc[:, "var_cal"] = sample[var_name]
                erfss = merge_and_replace(erfss, sample, ["var_cal"])
                # Trackers de la calibration
                tracker.track(i, "error_type", "no calib needed")
                tracker.track(i, "final_error", error)
                tracker.track(i, "sum_ap", (sample["var_cal"] * sample["wprm"]).sum())

            # Cas 2.2 : il faut calibrer
            else:
                # Cas 2.2.1 : on passe tout le monde au MAX
                if tracker.df["wanted"][i] >= tracker.df["max_possible"][i]:
                    erfss, tracker = calib_bucket_lim(
                        erfss,
                        i,
                        sample_fake,
                        not_fake_sum,
                        tracker,
                        tracker.df["var_max"][i],
                    )
                # Cas 2.2.2 : on passe tout le monde au MIN
                elif tracker.df["wanted"][i] <= tracker.df["min_possible"][i]:
                    erfss, tracker = calib_bucket_lim(
                        erfss,
                        i,
                        sample_fake,
                        not_fake_sum,
                        tracker,
                        tracker.df["var_min"][i],
                    )
                # Cas 2.2.3 : calibration iterative
                else:
                    erfss, tracker, sample_fake = optimization_quantile(
                        erfss,
                        i,
                        tracker,
                        sample_fake,
                        not_fake_sum,
                        error,
                        var_name,
                        err_max,
                    )
                    assert sample_fake[sample_fake["var_cal"] == 0.1].empty is True

            # Dans tous les cas, on sauvegarde les valeurs antérieures des sample_true
            sample_true.loc[:, "var_cal"] = sample_true[var_name]
            erfss = merge_and_replace(erfss, sample_true, ["var_cal"])

            # On vérifie qu'on a bien amélioré la calib
            ecart_av = round(
                100
                * (tracker.df["sum_av"][i] - tracker.df["wanted"][i])
                / tracker.df["wanted"][i],
                2,
            )
            ecart_ap = round(100 * tracker.df["final_error"][i], 2)
            tc.assertLessEqual(abs(ecart_ap), abs(ecart_av))
            print(
                "Pour le bucket ",
                i,
                "on est passé d'un écart de ",
                ecart_av,
                "% à un écart de ",
                ecart_ap,
                "%",
            )

            # On reinitialise les variables avant de continuer la boucle
            del sample_true
            del sample_fake
            del sample

    # On vérifie qu'on a bien calibré tout le monde
    tc.assertEqual(erfss[erfss["var_cal"] == 0.1].empty, True)
    # On remplace la colonne d'intérêt par sa version calibrée
    erfss.loc[:, var_name] = erfss["var_cal"]

    # Mise en forme
    tracker.df["final_error"] = [100 * i for i in tracker.df["final_error"]]

    # On fait un recalage final sur agrégat  - ou pas, car sinon on sort des buckets.. ( ou par décile?) TODO
    # erfss = recalage_calib(erfss, var_name, tracker.df['wanted'][i])

    total_sum = (erfss[var_name] * erfss["wprm"]).sum()
    print(
        "Somme finale pondérée de :",
        var_name,
        " : ",
        round(total_sum, 2),
        "€, soit un écart de ",
        round(
            abs(total_sum - tracker.df["wanted"].sum()) / tracker.df["wanted"].sum(), 2
        ),
        "% avec POTE",
    )

    # On repasse en individus
    erfs_ind = foyers_fiscaux_to_individus(
        erfs_cal_ind,
        erfss,
        [var_name],
        [var_name],
    )

    return tracker.df, erfss, erfs_ind


# Cell

# Attribution: fichier calib <> variable openfisca
dico = {
    "rfr": "revkire",
    "salaire_imposable": "salaire_imposable",
    "revenus_capitaux_prelevement_bareme": "revenus_capitaux_prelevement_bareme",
    "revenus_capitaux_prelevement_liberatoire": "revenus_capitaux_prelevement_liberatoire",
    "revenus_capitaux_prelevement_forfaitaire_unique_ir": "revenus_capitaux_prelevement_forfaitaire_unique_ir",
    "retraite_brute": "retraite_brute",  # retraites
    "chomage_brut": "chomage_et_indemnites",
    "f4ba": "revenu_categoriel_foncier",
}


def calibration(erfs_to_cal_ind, var_name, annee_erfs, annee_pote, calib=None):

    erfs_to_cal_ff = individus_to_foyers_fiscaux(erfs_to_cal_ind)

    Distribs = []
    # 1 - Import de la distribution issue de POTE 2019
    if calib is None:
        file = (
            config.get("CALIB")
            + "CalibPote-"
            + config.get("YEAR_POTE")
            + "-"
            + dico[var_name]
            + ".json"
        )
        with open(file) as f:
            calib = json.load(f)

    # 1.5 Cas du chomage
    # Séparation des gens au-dessus du plafond annuel (93_738) : ils touchent des indemnités de la fonction publique
    if var_name == "chomage_brut":
        calib = seuil_chomage(calib)

    # 2 - Réduction des buckets
    print("\n \n On réduit le nombre de buckets")
    calib_new = reduce_bucket_number(calib, max_gap=0.05)
    # On fusionne les buckets dont la moyenne est à moins de 5% d'écart

    # 3 - Calcul de la distribution de l'ERFS (calée sur calib)
    # NB: ici on fusionne les buckets qui ont les mêmes frontières
    print("\n \n On fusionne les buckets qui ont les mêmes frontières")
    Distrib_ERFS, Distrib_POTE, calib_final = distrib_to_quantiles(
        erfs_to_cal_ff, var_name, calib_new
    )
    Distribs.append(Distrib_ERFS)
    Distribs.append(Distrib_POTE)

    # 4 - Ajout de gens dans les buckets vides
    print("\n \n Ajout de gens dans les buckets vides")
    erfs_to_cal, erfs_to_cal_ind = ajout_gens_en_haut(
        erfs_to_cal_ff, erfs_to_cal_ind, var_name, Distrib_POTE, Distrib_ERFS
    )

    # 5 - Recalcul de la distribution
    print("\n \n Calcul de la distribution")
    Distrib_ERFS, Distrib_POTE, calib_final = distrib_to_quantiles(
        erfs_to_cal, var_name, calib_new
    )

    # 6 - Distributions avant calibration
    fig_var_to_cal, error_avant_calib, final_error_av = compare_distributions(
        Distrib_ERFS.df,
        Distrib_POTE.df,
        var_name,
        annee_erfs,
        annee_pote,
        log=False,
        title_suffix="avant_calibration",
        df_cal=[],
    )

    print(
        "Somme avant calibration", (erfs_to_cal["wprm"] * erfs_to_cal[var_name]).sum()
    )

    # 7 - Calibration
    erreur_max = 0.01  # On choisit de calibrer à 1%
    tracking, erfs_cal_ff, erfs_cal_ind = calibration_quantiles(
        var_name, erfs_to_cal_ind, Distrib_ERFS, Distrib_POTE, erreur_max
    )
    erfs_cal_ff.drop(["var_cal"], axis=1, inplace=True)
    print(
        "Somme apres calibration", (erfs_cal_ff["wprm"] * erfs_cal_ff[var_name]).sum()
    )

    # 8 - Nouvelle distribution
    Distrib_CAL, Distrib_POTE, calib_final = distrib_to_quantiles(
        erfs_cal_ff, var_name, calib_final
    )
    Distribs.append(Distrib_CAL)

    # 9 - Amélioration
    fig_var_cal, error_post_calib, final_error = compare_distributions(
        Distrib_ERFS.df,
        Distrib_POTE.df,
        var_name,
        annee_erfs,
        annee_pote,
        log=False,
        title_suffix="apres_calib",
        df_cal=Distrib_CAL.df,
    )

    # Conclusion
    print(
        "Après calibration de la variable ",
        var_name,
        "sur POTE ",
        annee_pote,
        ", on est passé d'une erreur de ",
        final_error_av,
        "% à une erreur de ",
        final_error,
        "%",
        "soit une amélioration de ",
        round(final_error - final_error_av, 2),
        "% (doit être >0)",
    )

    # On repasse en individus
    erfs_cal_ind = foyers_fiscaux_to_individus(
        erfs_cal_ind,
        erfs_cal_ff,
        [var_name],
        [var_name],
    )

    erfs_cal_ind.drop(["var_cal"], axis=1, inplace=True)

    del fig_var_to_cal
    del Distrib_ERFS
    del Distrib_POTE
    del Distrib_CAL

    return erfs_cal_ff, erfs_cal_ind, Distribs, fig_var_cal
