#import inspect
import os
from os.path import join, isfile, isdir
#from html.parser import HTMLParser
import json
import gzip 
from datetime import datetime, timedelta
import pandas as pd
#from random import random
import matplotlib.pyplot as plt
#import numpy as np
import requests
from ourairports import OurAirports

from co2eq.flight_utils import CityDB, FlightDB, GoClimateDB, Flight

plt.rcParams.update({'figure.max_open_warning': 0})

class Meeting:

  def __init__( self, name:str, conf={},  airportDB=True, cityDB=True, \
                flightDB=True, goclimateDB=True ):
    """ Meeting class

    Since conf is used to generate some DB (goclimateDB), DB can either
    be provided as object of set to True to indicate these need to be generated.

    Args:
      name (str): the name of the meeting
      conf (dict): configuration parameters
      airportDB: the airportDB object ( OurAirports ) or True to indicate the DB
        is generated by FlightDB.
        By default it is set to True.
      cityDB: the cityDB object ( CityDB ) or True to indicate the DB is
        generated by FlightDB.
        By default it is set to True.
      goclimateDB: the cityDB object ( GoClimateDB ) or True to indicate the DB
        is generated by FlightDB.
        By default it is set to True. Note that this is the only DB that needs
        the configuration parameters.

    """
    self.name = name
    self.conf = conf
    if airportDB is True:
      self.airportDB = OurAirports()
    else:
      self.airportDB = airportDB
    if  cityDB is True:
      self.cityDB = CityDB( )
    else:
      self.cityDB = cityDB
    if goclimateDB is True:
      self.goclimateDB=GoClimateDB( conf )
    else:
      self.goclimateDB = goclimateDB
    ## human representation IATA, (city, country)( city, state, country)
    self.location = self.get_location( )
    ## iata code for the meeting location - flightDB only consider iata code
    self.iata_location = self.cityDB.best_guess( self.location )[ 'iata' ]
    ## by default meetings have a common destination, so we can optimize
    ## flightDB and designate it by the destination airport
    if flightDB is True:
      flightDB = FlightDB( conf, cityDB=self.cityDB, airportDB=self.airportDB, \
                           goclimateDB=self.goclimateDB)
    self.flightDB = flightDB
    # print( f"meeting iata location: {self.iata_location}" )
    self.cache_base = join( self.conf[ 'CACHE_DIR' ], self.name )
    if isdir( self.cache_base ) is False:
      os.makedirs( self.cache_base )
    self.attendee_list = None

  def get_location( self ):
    pass
    return None

  def get_attendee_list( self, file_attendee_json ):
    """ reads the json file and initializes self.attendee_list """
    with gzip.open( file_attendee_json, 'rt', encoding="utf8") as f:
      self.attendee_list = json.loads( f.read() )
##    pass

  def attendee_location( self, attendee ):
    """ returns the location of the attendee

    This function is specific to the format of the attendees.
    The location can be expressed as ( city_name, country),
    (city_name, state, country), city_name where city_name can be a
    designation or a iata code.
    """
    pass

  def kwargs_to_file_name( self, suffix, extension, kwargs ):
    ## output file_name
    file_name = suffix
    for key, value in kwargs.items():
      file_name += f"-{key}_{value}"
    return file_name + f".{extension}"


  def build_co2eq_data( self, mode='flight', cluster_key=None, co2eq='myclimate' ) -> dict :
    """ co2 equivalent based on real flights including multiple segments)

    Todo:
    * current distance does not work with goclimate as goclimate seems to only
      take iata airport code, while we provide iata city code.
    """

    ## checking if the co2_eq has already been computed.
    ## computing flight co2_eq takes in fact more time than expected, so we prefer
    ## to cache it, in case it is re-used multiple time.
    ## In addition, it provides a way to manually inspect the co2_eq and perform manual operations
    kwargs = locals()
    del kwargs[ 'self' ]
    data_file = join( self.cache_base, self.kwargs_to_file_name( 'co2eq', 'json', kwargs ) )
    if isfile( data_file ) is True:
      with gzip.open( data_file, 'rt', encoding="utf8" ) as f:
        return json.loads( f.read() )

    if cluster_key is None:
      co2eq_data = { 'total co2eq' : 0 }  ## return a number
    else:
      co2eq_data = {} ## returns a dict { cluster_value: co2_eq }
    for attendee in self.attendee_list:
##      print( f"  - attendee: {attendee}" )
      location = self.attendee_location( attendee )
      attendee_iata_city = self.cityDB.best_guess( location )[ 'iata' ]
      if attendee_iata_city == self.iata_location:
        continue

      if mode == 'flight':
        flight = self.flightDB.select_flight( attendee_iata_city , self.iata_location )
      elif mode == 'distance':
        segment_list = [ [ attendee_iata_city , self.iata_location ],  \
                         [ attendee_iata_city , self.iata_location ] ]
        ## co2_eq is computed at instantiation
        flight = Flight( segment_list=segment_list, cityDB=self.cityDB, \
                         airportDB=self.airportDB, goclimateDB=self.goclimateDB)
        flight = flight.export()
      if cluster_key is None:
        co2eq_data [ 'total co2eq' ] += flight[ 'co2eq' ][ co2eq ]
      else:
        if cluster_key == 'flight_segments' :
          key = len( flight[ 'segment_list' ] )
        else:
          key = attendee[ cluster_key ]
        try:
          co2eq_data [ key ] += flight[ 'co2eq' ][ co2eq ]
        except:
          co2eq_data [ key ] = flight[ 'co2eq' ][ co2eq ]

    with gzip.open( data_file, 'wt', encoding="utf8" ) as f:
      f.write( json.dumps( co2eq_data, indent=2) )
    return co2eq_data

  def reduce_and_transpose( self, column_dict, column_keys ):
    """ format the data for a stack diagram

    Args:
      column_dict (dict): is a dictionary that contains the data associated to the column.
    column_dict = { column_label_1 : column_data_1 , ..., column_label_n : column_data_n }
    column_data contains is a dict { column_key_1 : value_1, ..., column_key_m : value_m }

    each column_data is reduced to the 'column_keys' most important keys, other
    values are concatenated into 'others'

    """
    ## sorting keys according to values
    ## merged_column_dict is not necessary when the dictionaries have the same keys
    ## but this is a more generic way to do this.
    ## https://www.pythoncentral.io/how-to-sort-python-dictionaries-by-key-or-value/
    merged_column_dict = {}
    for column_data in column_dict.values():
      for key in column_data.keys():
        try:
          merged_column_dict[ key ] += column_data[ key ]
        except KeyError:
          merged_column_dict[ key ] = column_data[ key ]

    sorted_keys = sorted(merged_column_dict, key=merged_column_dict.__getitem__, reverse=True)
    if column_keys is None:
      plot_sorted_keys = sorted_keys
    else:
      plot_sorted_keys = sorted_keys[ :column_keys ]
      plot_sorted_keys.append('Others')

    for column_label in column_dict.keys():
      column_dict[ column_label ][ 'Others' ] = 0
      column_key_list = list( column_dict[ column_label ].keys() )
      for key in  column_key_list :
        if key not in plot_sorted_keys:
          column_dict[ column_label ][ 'Others' ] += column_dict[ column_label ][ key ]
          del column_dict[ column_label ][ key ]
    ## stack diagrams are line_data stack together
    ## line_data = [ y(X1[0]), y(X2[0]), y(X3[0]) y(X4[0])]
    ## with yi the y value that corresponds to xi
    ## line_list  = [ line_data_0, line_data_1, ..., ]
    line_list = [] ## column_dict data reordered to be plot
    for key in plot_sorted_keys:
      line_data = []
      for column_label in column_dict.keys():
        try:
          line_data.append( column_dict[ column_label ][ key ] )
        except KeyError :
          line_data.append( 0 )
      line_list.append( line_data )
    return line_list, plot_sorted_keys

  def plot_stack( self, line_list:list, stack_label:list=None, column_label:list=None,\
                  column_range:list=None, title:str="", figsize=(6,4)):
    if column_range is None:
      column_range = range( len( line_list[ 0 ] ) )
    if column_label is None:
      column_label = [ "" for column in  line_list[ 0 ] ]

    fig, ax = plt.subplots(figsize=figsize )
    for line_data in line_list:
      if stack_label is None:
        ax.bar( column_range,  line_data )
      else:
        line_index = line_list.index( line_data )
        if line_index == 0:
          bottom = [ 0 for i in column_range ]
        else:
          bottom = [ bottom[ i ] + line_list[ line_index - 1 ][i] for i in column_range ]
        ax.bar( column_range,  line_data, bottom=bottom, label=stack_label[ line_index ] )
    if stack_label is not None :
      ax.legend( labels=stack_label )
    ax.set_ylabel("CO2 Equivalent (kg)" )
    ax.set_xticks( column_range )
    ax.set_xticklabels( column_label )
    ax.set_title( title, pad=20 )
    return fig, ax

  def default_kwargs( self, mode, cluster_key, cluster_nbr, co2eq ) :
    kwargs = locals()
    del kwargs[ 'self' ]
    ## getting co2_eq
    ## depending on cluster_key build_co2eq_data returns a number of a dict
    if mode is None:
      mode_list = [ 'flight', 'distance']
      kwargs[ 'mode' ] = 'flight_distance'
    else:
      mode_list = [ mode ]
    if co2eq is None:
      co2eq_list = [ 'myclimate', 'goclimate']
      kwargs[ 'co2eq' ] = 'myclimate_goclimate'
    else:
      if isinstance( co2eq , str ) :
        co2eq_list = [ co2eq ]
      elif isinstance( co2eq , list ) :
        co2eq_list = co2eq
        kwargs[ 'co2eq' ] = co2eq[ 0 ]
    return mode_list, cluster_key, cluster_nbr, co2eq_list, kwargs

  ## co2eq is represented as a function of ( mode, co2eq)
  def plot_co2eq( self, mode=None, cluster_key=None, cluster_nbr=None, co2eq=None) :
##    https://pylibraries.com/matplotlib/stacked-bar-charts-with-matplotlip-pyplot/
##    https://www.statology.org/matplotlib-stacked-bar-chart/
##
    mode_list, cluster_key, cluster_nbr, co2eq_list, kwargs = \
      self.default_kwargs( mode, cluster_key, cluster_nbr, co2eq )
    fig_file = join( self.cache_base, self.kwargs_to_file_name( 'co2eq', 'svg', kwargs  ) )
    if isfile( fig_file ) is True:
      return

    ## building the CO2 equivalent data.
    ## in our case, the data is a dictionary for each mode
    column_dict = {} ## associates for each labels co2eq dict associated to each column
    for mode in mode_list : #, 'distance' ]:
      for co2eq in co2eq_list:
        co2_eq = self.build_co2eq_data( mode=mode, cluster_key=cluster_key, co2eq=co2eq )
        column_dict[ ( mode, co2eq ) ] = co2_eq
    line_list, stack_label = self.reduce_and_transpose( column_dict, cluster_nbr )

    if len( line_list ) == 1: #one key
      stack_label = None
    column_label = [ f"{mode}\n{co2eq}" for mode, co2eq in column_dict.keys() ]
##    column_range = range( len( column_label ) )
    title = 'CO2 Equivalent (kg)'
    if cluster_key is not None:
      title = title + f" by {cluster_key}"
    title = title + f" for {self.name} ({len(self.attendee_list)} participants)"

    fig, ax = self.plot_stack( line_list, stack_label=stack_label, \
                               column_label=column_label, title=title )
##    fig.subplots_adjust(top=0.1)
    if stack_label is not None : # only when legend exists
      plt.legend(bbox_to_anchor=(1, 1.1), loc='upper left')
    plt.tight_layout()
    plt.savefig( fig_file, bbox_inches='tight' )


## inherite from Meeting
class MeetingList(Meeting):

  def __init__( self, name, conf={}, meeting_list=None  ):
    self.name = name
    self.conf = conf
    self.meeting_list = meeting_list
    self.cache_base = join( conf[ 'CACHE_DIR' ], self.name )
    if isdir( self.cache_base ) is False:
      os.makedirs( self.cache_base )

  def meeting_list_to_obj( self, meeting ):
    """ return a meetin object from the content of meeting_list

    Meeting Object may be quite heavy and as such the meeting_list may not
    contain the object but instead the meeting name for example.
    In that case function generates the meeting object from the name
    """
    if isinstance( meeting, Meeting ):
      return meeting
    else:
      raise ValueError("Unable to return meeting object from meeting_list" )

  def plot_co2eq( self, mode=None, cluster_key=None, cluster_nbr=None, \
                  co2eq=None, figsize=(10,4), column_label=None, \
                  xticks_rotation='vertical', adjust_bottom=0.4, ) :

    kwargs = { 'mode' : mode, 'cluster_key' : cluster_key, \
               'cluster_nbr' : cluster_nbr, 'co2eq' : co2eq }
    if kwargs[ 'cluster_key' ] is not None and not isinstance( kwargs[ 'cluster_key' ], str ):
      raise ValueError( f"Unacceptable arguments. cluster_key must be string or None. {kwargs}" )
    if kwargs[ 'cluster_nbr' ] is not None and not isinstance( kwargs[ 'cluster_nbr' ], int ):
      raise ValueError( f"Unacceptable arguments. cluster_nbr must be int or None. {kwargs}" )

    for k, v in kwargs.items():
      if k in [ 'cluster_key', 'cluster_nbr' ]:
        continue
      if not isinstance( v, str ) :
        raise ValueError( f"Unacceptable arguments. Must be specified with strings except " \
                          f"cluster_key that can be None. {kwargs}" )
    fig_file = join( self.cache_base, self.kwargs_to_file_name( 'co2eq', 'svg', kwargs  ) )
    if isfile( fig_file ) is True:
      return

    column_dict = {}
    for meeting in self.meeting_list:
      meeting = self.meeting_list_to_obj( meeting )
      column_dict[ meeting.name ] = meeting.build_co2eq_data( mode=mode, cluster_key=cluster_key,\
                                                              co2eq=co2eq )
    line_list, stack_label = self.reduce_and_transpose( column_dict, cluster_nbr )
    title = 'CO2 Equivalent (kg)'
    if cluster_key is not None:
      title = title + f" by {cluster_key}"
    title = title + f" by {cluster_key} with {mode} and {co2eq}"

    if column_label is None:
      column_label = column_dict.keys()
    fig, ax = self.plot_stack( line_list, stack_label=stack_label, \
                               column_label=column_label, title=title, \
                               figsize=figsize )
    plt.xticks(rotation = xticks_rotation)
    fig.subplots_adjust(bottom=adjust_bottom)
    plt.legend(bbox_to_anchor=(1, 1.1), loc='upper left')
    plt.tight_layout()
    plt.savefig( fig_file, bbox_inches='tight' )
#    plt.show()




## We may need to specify the remote nature of the meeting and the name when
## it differs from teh closest airport
IETF_LOCATION = {  72 : ( 'Dublin',        'IE' ),
                   73 : ( 'Minneapolis',   'US' ),
                   74 : ( 'San Francisco', 'US' ),
                   75 : ( 'Stockholm',     'SE' ),
                   76 : ( 'Osaka',         'JP' ),
                   77 : ( 'Los Angeles',   'US' ),
                   78 : ( 'Brussels',      'BE' ),
                   79 : ( 'Beijing',       'CN' ),
                   80 : ( 'Prague',        'CZ' ),
                   81 : ( 'Montreal',      'CA' ),
                   82 : ( 'Taipei',        'TW' ),
                   83 : ( 'Paris',         'FR' ),
                   84 : ( 'Vancouver',     'CA' ),
                   85 : ( 'Atlanta',       'US' ),
                   86 : ( 'Orlando',       'US' ),
                   87 : ( 'Berlin',        'DE' ),
                   88 : ( 'Vancouver',     'CA' ),
                   89 : ( 'London',        'GB' ),
                   90 : ( 'Toronto',       'CA' ),
                   91 : ( 'Honolulu',      'US' ),
                   92 : ( 'Dallas/Fort W', 'US' ),
                   93 : ( 'Prague',        'CZ' ),
                   94 : ( 'Tokyo',         'JP' ),
                   95 : ( 'Buenos Aires',  'AR' ),
                   96 : ( 'Berlin',        'DE' ),
                   97 : ( 'Seoul',         'KR' ),
                   98 : ( 'Chicago',       'US' ),
                   99 : ( 'Prague',        'CZ' ),
                  100 : ( 'Singapore',     'SG' ),
                  101 : ( 'London',        'GB' ),
                  102 : ( 'Montreal',      'CA' ),
                  103 : ( 'Bangkok',       'TH' ),
                  104 : ( 'Prague',        'CZ' ),
                  105 : ( 'Montreal',      'CA' ),
                  106 : ( 'Singapore',     'SG' ),
                  107 : ( 'Vancouver',     'CA' ),
                  108 : ( 'Madrid',        'ES' ),
                  109 : ( 'Bangkok',       'TH' ),
                  110 : ( 'Prague',        'CZ' ),
                  111 : ( 'San Francisco', 'US' )
                  }

ORGANIZATION_MATCH = { 'huaw' : "Huawei",
                       'futurewei' : "Huawei",
                       'cisco' : "Cisco",
                       'ericsson' : "Ericsson",
                       'microsoft' : "Microsoft",
                       'google' : "Google",
                       'juniper' : "Juniper",
                       'orange' : "Orange",
                       ( 'france', 'telecom' ) : "Orange",
                       'francetelecom' : "Orange",
                       'oracle' : "Oracle",
                       'isoc' : "ISOC",
                       ( 'internet', 'society' ) : "ISOC",
                       'akama' : "Akamai",
                       'nist' : "NIST",
                       ( 'dehli', 'institute' ) : "Dehli Institute of Advanced studies",
                       ( 'amity', 'university' ) : "Amity University",
                       'intel'  : "Intel",
                       'verisign' : "Verisign",
                       'salesforce' : "Salesforce",
                       'facebook' : "Facebook",
                       'upsa' : "UPSA",
                       'ntt' : "NTT",
                       'apple' : "Apple",
                       'cloudflare' : "Cloudflare",
                       'nokia' : "Nokia",
                       'amsl' : "IETF",
                       'ietf' : "IETF",
                       'interdigital' : "Interdigital",
                       ( 'internet', 'systems', 'consortium' ) : "ISC",
                       'tencent' : "Tencent",
                       'verizon' : "Verizon",
                       'apnic' : 'APNIC',
                       'zte' : 'ZTE',
                       'yokogawa' : 'Yokogawa',
                       'alcatel' : "Alcatel-Lucent",
                       'lucent' : "Alcatel-Lucent",
                       'samsung' : "Samsung",
                       'nortel' : "Nortel",
                       ( 'british', 'telecom' ) : 'BT',
                       ( 'deutsche', 'telekom' ) : 'DT',
                       'tsinghua' : 'Tsinghua University',
                       'hitachi' : 'Hitachi',
                       'siemens' : 'Siemens',
                       ( 'china', 'mobile' ): 'China Mobile',
                       'icann' : 'ICANN',
                       'comcast' : 'Comcast',
                       'mozilla' : 'Mozilla'
                   }

class IETFMeeting ( Meeting ):

  def __init__( self, name:str, conf={},  airportDB=True, cityDB=True, \
                flightDB=True, goclimateDB=True ):
    self.ietf_nbr = int( name[4:] )
    super().__init__( name, conf=conf, cityDB=cityDB, flightDB=flightDB, airportDB=airportDB )
    self.attendee_list_html = join( self.cache_base,  'attendee_list.html.gz' )
    self.attendee_list_json = join( self.cache_base,  'attendee_list.json.gz' )
    self.attendee_list = self.get_attendee_list()

  def get_location( self ):
    return IETF_LOCATION[ self.ietf_nbr ]

  def attendee_location( self, attendee ):
    return attendee[ 'country' ]

  def get_attendee_list_html( self ):
    print( f"{self.name}: Getting HTML file" )
    if self.ietf_nbr >= 108 :
      url = "https://registration.ietf.org/" + str( self.ietf_nbr ) + "/participants/remote/"
    else:
      url = "https://www.ietf.org/registration/ietf" + str( self.ietf_nbr ) + \
          "/attendance.py"
    r = requests.get( url )
    ## while encoding is always utf-8 in some places r.text did not work
    ## while specifying the encoding seemed to work.
    if self.ietf_nbr <= 73 or self.ietf_nbr >= 89:
      txt = r.text
    else:
      txt = r.content.decode('utf8')
    ## note that IETF web pages for 74 - 93 the html file has an error.
    ## The line 31 indicates colspan=2" which should be colspan="2" instead
    if self.ietf_nbr >= 74 and self.ietf_nbr <= 93:
      txt = txt.replace( "colspan=2\"", "colspan\"=2\"" )
    with gzip.open(self.attendee_list_html, 'wt', encoding="utf8" ) as f:
      f.write( txt )

  def parse_htm_remote( self ) :
    """ parses remote meeting  108, 109, 110, 111 """
    with gzip.open( self.attendee_list_html, 'rt', encoding="utf8" ) as f:
      dfs = pd.read_html(f.read(), header=0 )
      json_obj = json.loads( dfs[0].to_json( orient="records" ) )
      for attendee in json_obj:
        try:
          attendee[ 'country' ] = attendee.pop( 'Country' )
          attendee[ 'firstname' ] = attendee.pop( 'First Name' )
          attendee[ 'lastname' ] = attendee.pop( 'Last Name' )
          attendee[ 'organization' ] = attendee.pop( 'Organization' )
##          attendee[ 'presence' ] = attendee.pop( 'On-Site' )
        except:
          print( f"Cannot create attendee: {attendee}" )
      for attendee in json_obj:
        attendee[ 'presence' ] = 'remote'
      return json_obj

  def parse_htm_104( self ):
    """ new IETF 103 meeting """
    with gzip.open( self.attendee_list_html, 'rt', encoding="utf8" ) as f:
      dfs = pd.read_html(f.read(), header=0 )
      json_obj_1 = json.loads( dfs[1].to_json( orient="records" ) )
      for attendee in json_obj_1:
        try:
          attendee[ 'country' ] = attendee.pop( 'In Person Participants - Checked In OnSite.4' )
          attendee[ 'firstname' ] = attendee.pop( 'In Person Participants - Checked In OnSite.2' )
          attendee[ 'lastname' ] = attendee.pop( 'In Person Participants - Checked In OnSite.1' )
          attendee[ 'organization' ] = attendee.pop( 'In Person Participants - Checked In OnSite.3' )
          attendee[ 'presence' ] = attendee.pop( 'In Person Participants - Checked In OnSite' )
          attendee[ 'presence' ] = 'on-site'
        except:
          print( f"Cannot create attendee: {attendee}" )

      print( f"type json_1:{type(json_obj_1)}" )
      json_obj_2 = json.loads( dfs[2].to_json( orient="records" ) )
      for attendee in json_obj_2:
        try:
          attendee[ 'country' ] = attendee.pop( 'In Person Participants - Not Yet Arrived.4' )
          attendee[ 'firstname' ] = attendee.pop( 'In Person Participants - Not Yet Arrived.2' )
          attendee[ 'lastname' ] = attendee.pop( 'In Person Participants - Not Yet Arrived.1' )
          attendee[ 'organization' ] = attendee.pop( 'In Person Participants - Not Yet Arrived.3' )
          attendee[ 'presence' ] = attendee.pop( 'In Person Participants - Not Yet Arrived' )
          attendee[ 'presence' ] = 'not-arrived'
        except:
          print( f"Cannot create attendee: {attendee}" )
      json_obj_3 = json.loads( dfs[3].to_json( orient="records" ) )
      for attendee in json_obj_3:
        try:
          attendee[ 'country' ] = attendee.pop( 'Remote Participants.4' )
          attendee[ 'firstname' ] = attendee.pop( 'Remote Participants.2' )
          attendee[ 'lastname' ] = attendee.pop( 'Remote Participants.1' )
          attendee[ 'organization' ] = attendee.pop( 'Remote Participants.3' )
          attendee[ 'presence' ] = attendee.pop( 'Remote Participants' )
          attendee[ 'presence' ] = 'remote'
        except:
          print( f"Cannot create attendee: {attendee}" )

      ## header may appears in each list as an attendee
      header = { "country": "ISO 3166 Code",
                 "firstname": "First Name",
                 "lastname": "Last Name",
                 "organization": "Organization",
                 "presence": "on-site" }
      json_obj_1.remove( header )
      header[ 'presence' ] = 'not-arrived'
      json_obj_2.remove( header )
      header[ 'presence' ] = 'remote'
      json_obj_3.remove( header )

      json_obj_1.extend( json_obj_2 )
      json_obj_1.extend( json_obj_3 )
      json_obj = json_obj_1
      return json_obj

  def parse_htm_72( self ):
    with gzip.open( self.attendee_list_html, 'rt', encoding="utf8" ) as f:
      dfs = pd.read_html(f.read(), header=0 )
      if len( dfs ) == 3: ## IETF 74 -92 propose a login to view Profiles
        table_index = 2
      else: ## IETF 72 - 73, and IETF 93 - do not have login
        table_index = 1
      json_obj = json.loads( dfs[ table_index ].to_json( orient="records" ) )
      for attendee in json_obj:
        try:
          ## we use pop in order to avoid creating a new filed. JSON objects
          ## breaks when a new field is added.
          attendee[ 'country' ] = attendee.pop( 'ISO 3166 Code' )
          attendee[ 'firstname' ] = attendee.pop( 'First Name' )
          attendee[ 'lastname' ] = attendee.pop( 'Last Name' )
          attendee[ 'organization' ] = attendee.pop( 'Organization' )
          if 'Paid' in attendee.keys(): ## IETF 72 - 79
            attendee[ 'presence' ] = attendee.pop( 'Paid' )
          elif 'On-Site' in attendee.keys(): ## replaces 'Paid' for IETF >= 80
            attendee[ 'presence' ] = attendee.pop( 'On-Site' )
        except:
          print( f"Cannot create attendee: {attendee}" )
      for attendee in json_obj:
        presence = attendee[ 'presence' ]
        if presence in [ 'Yes', 'Comp', 'Comp - Host' ]:
          attendee[ 'presence' ] = 'on-site'
        elif presence == 'Remote':
          attendee[ 'presence' ] = 'remote'
        elif presence == 'No':
          attendee[ 'presence' ] = 'not-arrived'
        else:
          raise ValueError( f"unexpected attendee format {attendee}." \
                            f"Expecting 'Yes', 'No' or 'Remote' for presence" )
      return json_obj

  def get_attendee_list_json( self ):

    print( f"{self.name}: Parsing HTML file" )
    if self.ietf_nbr <= 103 :
      json_obj = self.parse_htm_72()
    elif self.ietf_nbr in [ 108, 109, 110, 111 ] : #remote meetings
      json_obj = self.parse_htm_remote( )
    elif self.ietf_nbr > 103 and self.ietf_nbr <= 107:
      json_obj = self.parse_htm_104( )
    else:
      with gzip.open( self.attendee_list_html, 'rt', encoding="utf8" ) as f:
        dfs = pd.read_html(f.read(), header=0 )
        print( f"   - dfs: {len(dfs)}" )
        for i in range( len( dfs ) ):
          print( f"      - dfs[{i}]: {dfs[i]}" )
        raise ValueError ( f"Unable to parse attendees for IETF{self.ietf_nbr}" )
    with gzip.open( self.attendee_list_json, 'wt', encoding="utf8" ) as f:
      f.write( json.dumps( json_obj, indent=2) )


  def clean_org( self, org_value ):
    """ get a more conventional string for Organization """

    if org_value is None:
      return 'Not Provided'
    org_value = org_value.lower()
    if org_value == 'bt': ## special cases where we look at exact match
      org_value = 'BT'
    elif org_value == 'nec':
      org_value = 'NEC'
    elif org_value == 'isc':
      org_value = 'ISC'
    else:
      for match in ORGANIZATION_MATCH.keys():
        if isinstance( match, str ):
          if match in org_value:
            org_value = ORGANIZATION_MATCH[ match ]
            break
        elif isinstance( match, tuple ):
          ## match all members of the tuple
          for m in match :
            if m not in org_value :
              break
          else : ## no break found
            org_value = ORGANIZATION_MATCH[ match ]
            break
    return org_value

  def get_attendee_list( self ):
    if self.attendee_list is not None:
      return self.attendee_list
    if isfile( self.attendee_list_json ) is False:
      if isfile( self.attendee_list_html ) is False:
        self.get_attendee_list_html( )
      self.get_attendee_list_json( )
    with gzip.open( self.attendee_list_json, 'rt', encoding="utf8" ) as f:
      attendee_list = json.loads( f.read() )
      ## removing countries set to None
      for attendee in attendee_list:
        if attendee[ 'country' ]  == 'None':
          attendee_list.remove( attendee )
        ## This is unexplained to me that 'NA' is replace by None
        ## I suspect, this is interpreted as Not Applicable, but thi sneeds to be checked.
        elif attendee[ 'country' ] is None :
          attendee[ 'country' ] = 'NA'
        attendee[ 'organization' ] = self.clean_org( attendee[ 'organization' ] )
    return attendee_list

class IETFMeetingList(MeetingList):

  def __init__( self, name="IETF Meetings", conf={}, meeting_list=None ):
    super().__init__( name, conf=conf, meeting_list=meeting_list )
    if self.meeting_list is None:
      min_ietf_nbr =  min( IETF_LOCATION.keys() )
      max_ietf_nbr = max( IETF_LOCATION.keys() )
      self.meeting_list = [ min_ietf_nbr + i for i in  range( max_ietf_nbr - min_ietf_nbr + 1 ) ]

  def meeting_list_to_obj( self, meeting ):
    if isinstance( meeting, int ) :
      return IETFMeeting( 'IETF' + str( meeting ), conf=self.conf )
    elif isinstance( meeting, IETFMeeting ) :
      return meeting
    else:
      raise ValueError("Unable to return meeting object from meeting_list" )

  def plot_all( self ):
    for meeting in self.meeting_list:
      meeting = self.meeting_list_to_obj( meeting )
      print( f"{meeting.name}: Processing plot_co2eq" )
      meeting.plot_co2eq( )
      meeting.plot_co2eq( cluster_key='organization', cluster_nbr=15)
      meeting.plot_co2eq( cluster_key='presence' )
      meeting.plot_co2eq( cluster_key='country', cluster_nbr=15 )
      meeting.plot_co2eq( cluster_key='flight_segments',  cluster_nbr=15 )
    print( f"\nprocessing {self.name }\n" )
    if isinstance( self.meeting_list[0], int ):
      column_label = []
      for ietf_nbr in self.meeting_list:
        label = f"{IETF_LOCATION[ ietf_nbr ][ 0 ]}  {ietf_nbr}"
        label = label.replace( 'Osaka', 'Hiroshima' )
        label = label.replace( 'Dallas/Fort W', 'Dallas' )
        column_label.append( label )
    else:
      column_label = None
    figsize=(10,4) ## width, heigh (inch) of the figure
    adjust_bottom = 0.4 ## make sure we have enough space to read the xticks
    xticks_rotation='vertical' ## xticks orientation
    plot_kwargs = { 'mode' : 'flight', 'figsize' : figsize, 'column_label' : column_label, \
                    'adjust_bottom' : adjust_bottom, 'xticks_rotation' : xticks_rotation }
    for co2eq in [ 'myclimate', 'goclimate' ]:
      self.plot_co2eq( cluster_key=None, co2eq=co2eq, **plot_kwargs )
      self.plot_co2eq( cluster_key='organization', cluster_nbr=15, co2eq=co2eq, **plot_kwargs )
      self.plot_co2eq( cluster_key='presence', co2eq=co2eq, **plot_kwargs )
      self.plot_co2eq( cluster_key='country', cluster_nbr=15, co2eq=co2eq, **plot_kwargs )
      self.plot_co2eq( cluster_key='flight_segments', cluster_nbr=15, co2eq=co2eq, **plot_kwargs )


def get_flight( conf, origin, destination ):
  """ return a flight from origin to destination

  The function tries with default values provided by FlightDB and in case no
  offer is provided performs another lookup with different dates.
  In our cases, the dates are 5 days latter.
  """
  cityDB = CityDB( )
  airportDB = OurAirports()
  goclimateDB = GoClimateDB( conf )
  flightDB = FlightDB( conf, cityDB=cityDB, airportDB=airportDB, goclimateDB=goclimateDB)
  try:
    flight = flightDB.select_flight( origin, destination )
  except ( ValueError ) :
    ## retry with other dates - in this case 5 days later
    departure_date = flightDB.departure_date
    return_date = flightDB.return_date
    alt_departure = datetime.strptime( departure_date + 'T16:41:24+0200', "%Y-%m-%dT%H:%M:%S%z")
    alt_departure = ( alt_departure + timedelta( days=5 ) ).isoformat()
    alt_return = datetime.strptime( return_date + 'T16:41:24+0200', "%Y-%m-%dT%H:%M:%S%z")
    alt_return = ( alt_return + timedelta( days=5 ) ).isoformat()
    flight = flightDB.select_flight( origin, destination, departure_date=alt_departure, return_date=alt_return )
  return flight

