"""Dictionary parsing and buffer bootstrap for XPLT."""

from __future__ import annotations

import struct
from dataclasses import dataclass

import numpy as np

from .binary_reader import BinaryReader
from .enums import FEDataType, Storage_Fmt
from .types import MultLike
from .views import _FieldMeta


def _read_dict_stream(
    reader: BinaryReader,
    dict_type: str,
    kind: str,
    dictionary: dict[str, dict[str, str]],
    order: dict[str, list[str]],
    where: dict[str, str],
) -> None:
    reader.search_block(dict_type)
    while reader.check_block("PLT_DIC_ITEM"):
        reader.search_block("PLT_DIC_ITEM")
        reader.search_block("PLT_DIC_ITEM_TYPE")
        tnum = int(struct.unpack("I", reader.read(4))[0])
        reader.search_block("PLT_DIC_ITEM_FMT")
        fnum = int(struct.unpack("I", reader.read(4))[0])
        reader.search_block("PLT_DIC_ITEM_NAME")
        name = reader.read(64).decode("utf-8", errors="ignore").split("\x00")[0]
        dictionary[name] = {
            "type": FEDataType(tnum).name,
            "format": Storage_Fmt(fnum).name,
        }
        order[kind].append(name)
        where[name] = kind


def read_dictionary(
    reader: BinaryReader,
) -> tuple[dict[str, dict[str, str]], dict[str, list[str]], dict[str, str]]:
    """Return dictionary mapping, order per kind, and location map."""
    dictionary: dict[str, dict[str, str]] = {}
    order: dict[str, list[str]] = {"node": [], "elem": [], "face": []}
    where: dict[str, str] = {}

    reader.search_block("PLT_DICTIONARY")
    _read_dict_stream(reader, "PLT_DIC_NODAL", "node", dictionary, order, where)
    _read_dict_stream(reader, "PLT_DIC_DOMAIN", "elem", dictionary, order, where)
    _read_dict_stream(reader, "PLT_DIC_SURFACE", "face", dictionary, order, where)

    return dictionary, order, where


@dataclass
class BufferStore:
    """Typed buffers allocated for each variable kind."""

    node_global: dict[str, list[np.ndarray | None]]
    node_region: dict[str, dict[str, list[np.ndarray | None]]]
    elem_item: dict[str, dict[str, list[np.ndarray | None]]]
    elem_mult: dict[str, dict[str, list[MultLike | None]]]
    elem_region: dict[str, dict[str, list[np.ndarray | None]]]
    face_item: dict[str, dict[str, list[np.ndarray | None]]]
    face_mult: dict[str, dict[str, list[MultLike | None]]]
    face_region: dict[str, dict[str, list[np.ndarray | None]]]


def prepare_field_meta_and_buffers(
    dictionary: dict[str, dict[str, str]], where: dict[str, str]
) -> tuple[dict[str, _FieldMeta], BufferStore]:
    """Create field metadata and empty buffers keyed by variable.

    Returns:
        tuple[dict[str, _FieldMeta], BufferStore]: Parsed metadata and corresponding buffers.
    """
    field_meta: dict[str, _FieldMeta] = {}
    node_global: dict[str, list[np.ndarray | None]] = {}
    node_region: dict[str, dict[str, list[np.ndarray | None]]] = {}
    elem_item: dict[str, dict[str, list[np.ndarray | None]]] = {}
    elem_mult: dict[str, dict[str, list[MultLike | None]]] = {}
    elem_region: dict[str, dict[str, list[np.ndarray | None]]] = {}
    face_item: dict[str, dict[str, list[np.ndarray | None]]] = {}
    face_mult: dict[str, dict[str, list[MultLike | None]]] = {}
    face_region: dict[str, dict[str, list[np.ndarray | None]]] = {}

    for name, d in dictionary.items():
        dtype = FEDataType[d["type"]]
        fmt = Storage_Fmt[d["format"]]
        fm = _FieldMeta(name, fmt, dtype)
        field_meta[name] = fm
        kind = where[name]
        if kind == "node" and fmt == Storage_Fmt.FMT_NODE:
            node_global[name] = []
            node_region[name] = {}
        elif kind == "elem":
            if fmt == Storage_Fmt.FMT_ITEM:
                elem_item[name] = {}
            elif fmt == Storage_Fmt.FMT_MULT:
                elem_mult[name] = {}
            elif fmt == Storage_Fmt.FMT_REGION:
                elem_region[name] = {}
        elif kind == "face":
            if fmt == Storage_Fmt.FMT_ITEM:
                face_item[name] = {}
            elif fmt == Storage_Fmt.FMT_MULT:
                face_mult[name] = {}
            elif fmt == Storage_Fmt.FMT_REGION:
                face_region[name] = {}

    buffers = BufferStore(
        node_global=node_global,
        node_region=node_region,
        elem_item=elem_item,
        elem_mult=elem_mult,
        elem_region=elem_region,
        face_item=face_item,
        face_mult=face_mult,
        face_region=face_region,
    )
    return field_meta, buffers


__all__ = [
    "BufferStore",
    "prepare_field_meta_and_buffers",
    "read_dictionary",
]
