"""Lightweight helpers for launching local FEBio runs."""

from __future__ import annotations

import asyncio
import atexit
import os
import subprocess
import threading
import time
from collections.abc import Generator, Sequence
from concurrent.futures import Future, ThreadPoolExecutor
from contextlib import suppress
from dataclasses import dataclass, field
from pathlib import Path
from typing import cast


@dataclass
class RunResult:
    """Summary information about a finished simulation command."""

    exit_code: int
    started_at: float
    ended_at: float
    log_path: Path
    metadata: dict[str, object] = field(default_factory=dict)

    @property
    def duration(self) -> float:
        """Execution time in seconds."""
        return self.ended_at - self.started_at


@dataclass
class RunHandle:
    """Future-like wrapper returned by runner implementations."""

    _future: Future[RunResult]

    def wait(self, timeout: float | None = None) -> RunResult:
        """Block until the underlying future completes.

        Returns:
            RunResult containing exit code and metadata.
        """
        return self._future.result(timeout)

    def done(self) -> bool:  # pragma: no cover - passthrough convenience
        """Return True when the run has finished."""
        return self._future.done()

    def cancel(self) -> bool:  # pragma: no cover - passthrough convenience
        """Attempt to cancel the run.

        Returns:
            ``True`` if cancellation succeeded.
        """
        return self._future.cancel()

    def result(self) -> RunResult:  # pragma: no cover - passthrough convenience
        """Return the RunResult, blocking if necessary."""
        return self._future.result()

    def __await__(self) -> Generator[RunResult, None, RunResult]:  # pragma: no cover
        async def _wrap() -> RunResult:
            return await asyncio.wrap_future(self._future)

        return cast(Generator[RunResult, None, RunResult], _wrap().__await__())


class Runner:
    """Minimal interface expected by the optimisation engine."""

    def run(
        self,
        job_dir: str | Path,
        feb_name: str | Path,
        *,
        env: dict[str, str] | None = None,
    ) -> RunHandle:
        """Schedule a FEBio run and return a handle for tracking completion."""
        raise NotImplementedError

    def shutdown(self) -> None:  # pragma: no cover - simple default
        """Clean up any resources held by the runner."""
        return None


class _BaseLocalRunner(Runner):
    """Execute FEBio commands locally, optionally in parallel."""

    def __init__(
        self,
        command: Sequence[str] | None = None,
        *,
        max_workers: int = 1,
        env: dict[str, str] | None = None,
    ):
        """Configure the base runner."""
        self.command = tuple(command or ("febio4", "-i"))
        if not self.command:
            raise ValueError("command may not be empty.")
        self.env = dict(env) if env else None
        self._executor = ThreadPoolExecutor(max_workers=max(1, int(max_workers)))
        self._active: set[subprocess.Popen[bytes]] = set()
        self._active_lock = threading.Lock()
        atexit.register(self.shutdown)

    def run(
        self,
        job_dir: str | Path,
        feb_name: str | Path,
        *,
        env: dict[str, str] | None = None,
    ) -> RunHandle:
        """Schedule a FEBio run and return a handle for tracking completion.

        Returns:
            Handle that can block or poll for completion.
        """
        job_path = Path(job_dir)
        feb_path = Path(feb_name)
        future = self._executor.submit(self._run_once, job_path, feb_path, env)
        return RunHandle(future)

    def shutdown(self) -> None:
        """Terminate any active jobs and stop the worker pool."""
        with self._active_lock:
            procs = list(self._active)
        for proc in procs:
            with suppress(Exception):
                proc.terminate()
        for proc in procs:
            with suppress(Exception):
                proc.wait(timeout=5)
        self._executor.shutdown(wait=False)

    # ---- internal helpers -------------------------------------------------
    def _run_once(
        self,
        job_path: Path,
        feb_name: Path,
        env: dict[str, str] | None = None,
    ) -> RunResult:
        """Execute a single FEBio command within the worker pool.

        Returns:
            RunResult containing exit code, timing, and log path.
        """
        job_path.mkdir(parents=True, exist_ok=True)
        feb_path = feb_name if feb_name.is_absolute() else job_path / feb_name
        if not feb_path.exists():
            raise FileNotFoundError(f"FEB file not found: {feb_path}")

        log_path = feb_path.with_suffix(".log")
        cmd = [*self.command, str(feb_path)]
        started = time.time()
        with open(log_path, "w", encoding="utf-8") as log_file:
            proc = subprocess.Popen(
                cmd,
                cwd=str(job_path),
                stdout=log_file,
                stderr=subprocess.STDOUT,
                env=self._merged_env(env),
                start_new_session=True,
            )
            with self._active_lock:
                self._active.add(proc)
            try:
                proc.wait()
                returncode = proc.returncode
            finally:
                with self._active_lock:
                    self._active.discard(proc)
        ended = time.time()
        result = RunResult(
            exit_code=returncode,
            started_at=started,
            ended_at=ended,
            log_path=log_path,
            metadata={"cmd": cmd},
        )
        return result

    def _merged_env(
        self,
        override: dict[str, str] | None = None,
    ) -> dict[str, str] | None:
        """Merge base and override environment dictionaries.

        Returns:
            Combined environment dictionary or ``None`` when no overrides apply.
        """
        if self.env is None and not override:
            return None
        merged = os.environ.copy()
        if self.env:
            merged.update(self.env)
        if override:
            merged.update(override)
        return merged


class LocalSerialRunner(_BaseLocalRunner):
    """Execute simulations sequentially on the local machine."""

    def __init__(
        self,
        command: Sequence[str] | None = None,
        *,
        env: dict[str, str] | None = None,
    ):
        """Create a serial runner for single-threaded execution."""
        super().__init__(command, max_workers=1, env=env)


class LocalParallelRunner(_BaseLocalRunner):
    """Execute simulations concurrently using a thread pool."""

    def __init__(
        self,
        n_jobs: int,
        command: Sequence[str] | None = None,
        *,
        env: dict[str, str] | None = None,
    ):
        """Create a parallel runner backed by a thread pool."""
        if n_jobs < 1:
            raise ValueError("n_jobs must be >= 1.")
        super().__init__(command, max_workers=n_jobs, env=env)
        self.n_jobs = int(n_jobs)


__all__ = [
    "LocalParallelRunner",
    "LocalSerialRunner",
    "RunHandle",
    "RunResult",
    "Runner",
]
